package com.af.v4.system.common.liuli.utils;

import cn.hutool.core.util.StrUtil;
import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.enums.CompatibleVersion;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.common.liuli.utils.enums.OSTypeEnum;
import com.af.v4.system.common.security.utils.SecurityUtils;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

/**
 * 应用工具
 */
@Component
public final class ApplicationUtils {

    private static final String SYSTEM_V4_VERSION;
    private static final Integer SYSTEM_V4_VERSION_CODE;
    private static final String APP_VERSION;
    private static final Integer APP_VERSION_CODE;
    private static final OSTypeEnum OS_TYPE;

    static {
        // SystemV4版本
        SYSTEM_V4_VERSION = ApplicationUtils.class.getPackage().getImplementationVersion();
        SYSTEM_V4_VERSION_CODE = buildVersionCode(SYSTEM_V4_VERSION);
        // 应用版本
        String appVersion;
        ClassLoader cl = ApplicationUtils.class.getClassLoader();
        URL url = cl.getResource("META-INF/MANIFEST.MF");
        Manifest manifest;
        if (url != null) {
            try {
                manifest = new Manifest(url.openStream());
                Attributes mainAttributes = manifest.getMainAttributes();
                appVersion = mainAttributes.getValue("Implementation-Version");
            } catch (IOException e) {
                appVersion = "0";
            }
        } else {
            appVersion = "0";
        }
        APP_VERSION = appVersion;
        APP_VERSION_CODE = buildVersionCode(APP_VERSION);
        // 操作系统
        String osName = System.getProperty("os.name").toLowerCase();
        if (osName.contains("linux")) {
            OS_TYPE = OSTypeEnum.LINUX;
        } else {
            OS_TYPE = OSTypeEnum.WINDOWS;
        }
    }

    private String DEFAULT_TENANT_NAME = null;
    private EnvType ENV_TYPE = null;
    private String APPLICATION_NAME = null;
    private CompatibleVersion COMPATIBLE_VERSION = null;

    /**
     * 获取琉璃架构版本
     *
     * @return 琉璃架构版本
     */
    public static String getSystemV4Version() {
        return SYSTEM_V4_VERSION;
    }

    /**
     * 获取琉璃架构版本号
     *
     * @return 琉璃架构版本号
     */
    public static Integer getSystemV4VersionCode() {
        return SYSTEM_V4_VERSION_CODE;
    }

    /**
     * 获取当前数据源类型
     *
     * @return 数据源类型
     */
    public static DbType getDbType() {
        return DynamicDataSource.getDbType();
    }

    public static OSTypeEnum getOSType() {
        return OS_TYPE;
    }

    /**
     * 获取应用版本
     *
     * @return 应用版本
     */
    public static String getAppVersion() {
        return APP_VERSION;
    }

    /**
     * 获取应用版本号
     *
     * @return 应用版本号
     */
    public static Integer getAppVersionCode() {
        return APP_VERSION_CODE;
    }

    /**
     * 构建版本编号
     *
     * @param version 版本号字符串
     * @return 版本编号
     */
    public static Integer buildVersionCode(String version) {
        List<String> versionArray = StrUtil.split(version, ".");
        if (versionArray.size() > 4) {
            throw new ServiceException("不支持的版本号[" + version + "])]", HttpStatus.CONFIG_ERROR);
        }
        int versionCode = 0;
        int multipleNum = 1;
        for (int i = versionArray.size() - 1; i >= 0; i--) {
            if (i < 3) {
                versionCode += Integer.parseInt(versionArray.get(i)) * multipleNum;
                multipleNum *= 1000;
            }
        }
        return versionCode;
    }

    /**
     * 判断应用是否运行在idea中
     */
    private static boolean isRunningInIDEA() {
        try {
            Class.forName("com.intellij.rt.execution.application.AppMainV2");
            return true;
        } catch (ClassNotFoundException ignored) {
            return false;
        }
    }

    /**
     * 获取当前运行环境
     *
     * @return 运行环境枚举
     */
    public EnvType getEnvType() {
        return ENV_TYPE;
    }

    @Value("${spring.profiles.active}")
    public void setEnvType(String envType) {
        ENV_TYPE = EnvType.toType(envType);
        if (ENV_TYPE == null) {
            throw new ServiceException("不支持的运行环境类型[" + envType + "]", HttpStatus.CONFIG_ERROR);
        }
        if (ApplicationUtils.isRunningInIDEA()) {
            if (ENV_TYPE == EnvType.DEV) {
                ENV_TYPE = EnvType.LOCAL;
            } else if (ENV_TYPE == EnvType.PROD) {
                throw new ServiceException("本地不允许直接以运行环境[" + envType + "]运行", HttpStatus.CONFIG_ERROR);
            }
        }
    }

    public String getEnvTypeValue() {
        return ENV_TYPE.getValue();
    }

    /**
     * 获取应用名称
     *
     * @return 应用名称
     */
    public String getApplicationName() {
        return APPLICATION_NAME;
    }

    @Value("${spring.application.name}")
    public void setApplicationName(String applicationName) {
        APPLICATION_NAME = applicationName;
    }

    /**
     * 获取租户名称
     *
     * @return 租户名称
     */
    public String getTenantName() {
        return DEFAULT_TENANT_NAME;
    }

    @Value("${tenant.name}")
    public void setTenantName(String tenantName) {
        this.DEFAULT_TENANT_NAME = tenantName;
    }

    /**
     * 从登录人中获取分公司名称,如果获取不到则返回租户名称
     *
     * @return 分公司名称
     */
    public String getOrgName() {
        LoginUser loginUser = SecurityUtils.getLoginUser();
        if (loginUser != null) {
            String userTenantName = loginUser.getTenantName();
            if (!StringUtils.isEmpty(userTenantName)) {
                return userTenantName;
            }
        }
        return getTenantName();
    }

    /**
     * 获取应用兼容版本
     *
     * @return 应用兼容版本
     */
    public CompatibleVersion getCompatibleVersion() {
        return COMPATIBLE_VERSION;
    }

    @Value("${tenant.version}")
    public void setCompatibleVersion(String version) {
        this.COMPATIBLE_VERSION = CompatibleVersion.toType(version);
    }

    /**
     * 获取应用信息的JSON格式
     *
     * @return 应用信息JSON格式
     */
    public JSONObject getValues() {
        return new JSONObject()
                .put("tenantName", getTenantName())
                .put("orgName", getOrgName())
                .put("envType", getEnvTypeValue())
                .put("applicationName", getApplicationName())
                .put("compatibleVersion", COMPATIBLE_VERSION.getValue())
                .put("appVersion", getAppVersion())
                .put("appVersionCode", getAppVersionCode())
                .put("dbType", getDbType().name())
                .put("osType", getOSType().getValue())
                .put("systemV4Version", getSystemV4Version())
                .put("systemV4VersionCode", getSystemV4VersionCode());
    }
}
