package com.af.v4.system.common.liuli.config.service;

import cn.hutool.cache.CacheUtil;
import cn.hutool.cache.impl.LRUCache;
import cn.hutool.core.util.StrUtil;
import com.af.v4.system.api.RemoteLiuLiService;
import com.af.v4.system.api.RemoteLogicService;
import com.af.v4.system.api.factory.DynamicFeignClientFactory;
import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.liuli.api.LiuLiApi;
import com.af.v4.system.common.liuli.config.LiuLiClientConfig;
import com.af.v4.system.common.liuli.config.subscriber.LiuLiCacheSubscriber;
import com.af.v4.system.common.liuli.logic.ILogicServiceProxy;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.liuli.utils.enums.LiuLiClientOperationalModeEnum;
import com.af.v4.system.common.plugins.core.SecureTools;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.redis.RedisService;
import org.json.JSONArray;
import org.json.JSONObject;
import org.redisson.api.RTopic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * 琉璃-配置服务
 *
 * @author Mr.river
 * @since 1.1.10
 */
@Service
public class LiuLiConfigService {

    private static final String AES_KEY = "kxYiWjt/cmRRYaCOT9asaw==";
    /**
     * 本地缓存
     */
    private static final LRUCache<String, JSONObject> LIULI_CONFIG_LOCAL_CACHE = CacheUtil.newLRUCache(200);
    private static final Logger LOGGER = LoggerFactory.getLogger(LiuLiConfigService.class);
    private static final Integer CONFIG_NOT_FOUND_CODE = 406;
    private static final Integer CONFIG_OTHER_ERROR_CODE = 416;
    private static EnvType envType = null;
    private final LiuLiClientConfig liuLiClientConfig;
    private final ApplicationUtils applicationUtils;
    private final RedisService redisService;
    private final DynamicFeignClientFactory dynamicFeignClientFactory;

    public LiuLiConfigService(LiuLiClientConfig liuLiClientConfig, ApplicationUtils applicationUtils, RedisService redisService, DynamicFeignClientFactory dynamicFeignClientFactory) {
        this.liuLiClientConfig = liuLiClientConfig;
        this.applicationUtils = applicationUtils;
        this.redisService = redisService;
        this.dynamicFeignClientFactory = dynamicFeignClientFactory;
        envType = applicationUtils.getEnvType();
    }

    /**
     * 获取配置项的缓存Key
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @return key
     */
    public static String getConfigCacheKey(String tenantName, String namespaceName, String environment, String configName) {
        return CacheConstants.CONFIG_DATA_CACHE_KEY + tenantName + "_" + namespaceName + "_" + environment + "_" + configName;
    }

    /**
     * 获取配置-模块映射KEY
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param moduleName    模块名称
     * @return key
     */
    public static String getConfigModuleMappingKey(String tenantName, String namespaceName, String environment, String moduleName) {
        return CacheConstants.CONFIG_MODULE_MAPPING_KEY + tenantName + "_" + namespaceName + "_" + environment + "_" + moduleName;
    }

    public static LRUCache<String, JSONObject> getLiuliConfigLocalCache() {
        return LIULI_CONFIG_LOCAL_CACHE;
    }

    public static String getAesKey() {
        return AES_KEY;
    }

    /**
     * 获取配置文件
     *
     * @param configName 配置名称
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject get(String configName) {
        return get(configName, false);
    }

    /**
     * 获取配置文件（仅查找本级租户）
     *
     * @param configName 配置名称
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject getBySelf(String configName) {
        return get(configName, SearchMode.BY_CONFIG_NAME, false, false);
    }

    /**
     * 获取当前租户和命名空间下指定模块的所有配置
     *
     * @param moduleName 模块名称
     * @return 配置文件集合，按照[{配置名称:配置内容}]的格式返回
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject getByModule(String moduleName) {
        return getByModule(moduleName, false);
    }

    /**
     * 获取配置文件
     *
     * @param name                             名称
     * @param searchMode                       查询模式
     * @param isThrowExceptionOnConfigNotFound 配置找不到时是否抛出异常
     * @param shouldGetConfigByParent          找不到配置是否寻找上级配置
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject get(String name, SearchMode searchMode, Boolean isThrowExceptionOnConfigNotFound, Boolean shouldGetConfigByParent) {
        if (name == null) {
            if (searchMode == SearchMode.BY_CONFIG_NAME) {
                LOGGER.error("配置名不能为空");
            } else if (searchMode == SearchMode.BY_MODULE_NAME) {
                LOGGER.error("模块名不能为空");
            }
            return null;
        }
        Supplier<JSONObject> fun = () -> getConfigByRequest(name, searchMode, shouldGetConfigByParent, result -> {
            if (!isThrowExceptionOnConfigNotFound) {
                LOGGER.debug("获取配置失败，相关信息：{}", result);
                String tenantName = applicationUtils.getOrgName();
                String namespaceName = applicationUtils.getApplicationName();
                String environment = applicationUtils.getEnvType().getValue();
                if (result.getCode() == CONFIG_NOT_FOUND_CODE) {
                    addCache(tenantName, namespaceName, environment, name, EmptyJSONObject.SELF);
                }
                return null;
            } else {
                throw new ServiceException(result.getMsg(), result.getCode());
            }
        }, true);
        JSONObject configInfo;
        if (searchMode == SearchMode.BY_MODULE_NAME) {
            configInfo = getConfigsByConfigModuleMapping(name, (key) ->
                    redisService.syncLock(key, () -> getConfigsByConfigModuleMapping(name,
                            (key2) -> fun.get())
                    )
            );
        } else {
            configInfo = getConfigByCache(name, (key) ->
                    redisService.syncLock(key, () -> getConfigByCache(name,
                            (key2) -> getConfigByDatabase(name, fun))
                    )
            );
        }
        // 传递新的副本避免外部引用修改只读配置
        return configInfo == null || configInfo instanceof EmptyJSONObject ? null : new JSONObject(configInfo.toString());
    }

    public JSONObject get(String configName, Boolean isThrowExceptionOnConfigNotFound) {
        return get(configName, SearchMode.BY_CONFIG_NAME, isThrowExceptionOnConfigNotFound, true);
    }

    /**
     * 获取当前租户指定模块下所有配置
     *
     * @param moduleName                       模块名
     * @param isThrowExceptionOnConfigNotFound 配置找不到时是否抛出异常
     * @return 配置文件集合，按照[{配置名称:配置内容}]的格式返回
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject getByModule(String moduleName, Boolean isThrowExceptionOnConfigNotFound) {
        JSONObject configResult = get(moduleName, SearchMode.BY_MODULE_NAME, isThrowExceptionOnConfigNotFound, true);
        if (configResult == null) {
            return null;
        }
        JSONArray configs = configResult.optJSONArray("items");
        if (configs == null) {
            return null;
        }
        JSONObject result = new JSONObject();
        configs.forEach(value -> {
            JSONObject valueObj = (JSONObject) value;
            result.put(valueObj.getJSONObject("$globalProp").getString("name"), valueObj);
        });
        return result;
    }

    /**
     * 通过缓存获取配置
     *
     * @param configName  配置名称
     * @param notFoundFun 未找到配置后回调函数
     * @return 结果
     */
    private JSONObject getConfigByCache(String configName, Function<String, JSONObject> notFoundFun) {
        String tenantName = applicationUtils.getOrgName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = envType.getValue();
        String configCacheKey = getConfigCacheKey(tenantName, namespaceName, environment, configName);
        // 从本地缓存获取
        JSONObject configContent = LIULI_CONFIG_LOCAL_CACHE.get(configCacheKey);
        if (envType != EnvType.LOCAL && configContent == null) {
            // 从redis获取
            configContent = redisService.get(configCacheKey);
            LOGGER.debug("通过redis获取琉璃配置[{}]：{}", configName, configContent);
            if (configContent != null) {
                // 缓存到本地
                LIULI_CONFIG_LOCAL_CACHE.put(configCacheKey, configContent);
            }
        }
        return configContent != null ? configContent : notFoundFun.apply(configCacheKey);
    }

    /**
     * 通过配置-模块映射缓存获取配置集合
     *
     * @param moduleName  模块名称
     * @param notFoundFun 未找到配置后回调函数
     * @return 结果
     */
    private JSONObject getConfigsByConfigModuleMapping(String moduleName, Function<String, JSONObject> notFoundFun) {
        String tenantName = applicationUtils.getOrgName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = envType.getValue();
        String key = getConfigModuleMappingKey(tenantName, namespaceName, environment, moduleName);
        if (envType == EnvType.LOCAL) {
            return notFoundFun.apply(key);
        }
        // 从redis获取配置-模块映射信息
        String configs = redisService.get(key);
        if (configs == null) {
            return notFoundFun.apply(key);
        }
        List<String> configNameArray = StrUtil.split(configs, ",");
        // 按配置集合解析
        JSONObject configResult = new JSONObject();
        JSONArray resolveResultArray = new JSONArray();
        configResult.put("items", resolveResultArray);
        // 循环获取配置信息
        for (String configName : configNameArray) {
            JSONObject configObj = get(configName);
            if (configObj != null) {
                resolveResultArray.put(configObj);
            }
        }
        return configResult;
    }

    /**
     * 通过数据库获取配置
     *
     * @param configName  配置名称
     * @param notFoundFun 未找到配置后回调函数
     * @return 结果
     */
    private JSONObject getConfigByDatabase(String configName, Supplier<JSONObject> notFoundFun) {
        if (envType == EnvType.LOCAL) {
            return notFoundFun.get();
        }
        String tenantName = applicationUtils.getOrgName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = applicationUtils.getEnvType().getValue();
        JSONObject params = new JSONObject();
        params.put("tenantName", tenantName);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("configName", configName);
        try {
            RemoteLogicService remoteLogicService = (RemoteLogicService) dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, ServiceNameConstants.SYSTEM_SERVICE);
            R<Object> result = remoteLogicService.run("getLiuLiConfig", params.toString(), SecurityConstants.INNER);
            if (result.getCode() == HttpStatus.SUCCESS) {
                JSONObject response = result.parseResponseJson();
                LOGGER.debug("通过数据库获取琉璃配置[{}]：{}", configName, response);
                String contentStr = response.getJSONObject("data").optString("content", null);
                if (contentStr != null) {
                    JSONObject content = new JSONObject(contentStr);
                    addCache(tenantName, namespaceName, environment, configName, content);
                    return content;
                }
            } else {
                LOGGER.error("服务请求出错，状态码：{}，错误原因：{}", result.getCode(), result.getMsg());
            }
        } catch (Throwable e) {
            LOGGER.error("af-system服务请求失败:{}", e.getMessage());
        }
        return notFoundFun.get();
    }

    /**
     * 通过琉璃中心获取配置
     *
     * @param name                    资源名称
     * @param searchMode              查询模式
     * @param shouldGetConfigByParent 找不到配置是否寻找上级配置
     * @param notFoundFun             未找到配置后回调函数
     * @param isPrecompile            是否预编译
     * @return 结果
     */
    private JSONObject getConfigByRequest(String name, SearchMode searchMode, Boolean shouldGetConfigByParent, Function<R<JSONObject>, JSONObject> notFoundFun, Boolean isPrecompile) {
        if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.OFFLINE) {
            return notFoundFun.apply(R.fail("当前处于离线模式"));
        }
        LOGGER.debug("通过琉璃中心获取配置[{}]", name);
        String tenantName = applicationUtils.getOrgName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = applicationUtils.getEnvType().getValue();
        //调用琉璃中心需传递的参数
        String url;
        JSONObject params = new JSONObject();
        params.put("tenantUUID", tenantName);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("shouldGetConfigByParent", shouldGetConfigByParent);
        params.put("dbType", DynamicDataSource.getDbType().name());
        if (searchMode == SearchMode.BY_MODULE_NAME) {
            params.put("moduleName", name);
            url = LiuLiApi.BATCH_GET_CONFIG_PATH;
        } else {
            params.put("configName", name);
            url = LiuLiApi.GET_CONFIG_PATH;
        }
        R<JSONObject> result;
        if (namespaceName.equals(ServiceNameConstants.LIULI_SERVICE)) {
            // 琉璃自身调用
            try {
                Object localResult = SpringUtils.getBean(ILogicServiceProxy.class).run("getConfigByClientV2", params);
                result = R.ok((JSONObject) localResult);
            } catch (ServiceException e) {
                result = R.build(e);
            }
        } else if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.INTRANET) {
            // 内网环境调用
            RemoteLiuLiService remoteLiuLiService = (RemoteLiuLiService) dynamicFeignClientFactory.getFeignClient(RemoteLiuLiService.class, ServiceNameConstants.LIULI_SERVICE);
            R<Map<String, Object>> remoteResult;
            if (searchMode == SearchMode.BY_MODULE_NAME) {
                remoteResult = remoteLiuLiService.batchGetConfigByClient(params.toString(), SecurityConstants.INNER);
            } else {
                remoteResult = remoteLiuLiService.getConfigByClient(params.toString(), SecurityConstants.INNER);
            }
            result = R.build(remoteResult.parseResponseJson());
        } else {
            // 互联网环境调用
            String cloudConfig = RestTools.post(url, params);
            result = R.build(new JSONObject(cloudConfig));
        }
        if (result.getCode() == HttpStatus.SUCCESS) {
            String plainText = result.getData().getString("plainStr");
            JSONObject data = new JSONObject(SecureTools.AESDecrypt(plainText, AES_KEY));
            if (searchMode == SearchMode.BY_CONFIG_NAME) {
                // 按单一配置解析
                return resolveConfig(data, isPrecompile);
            } else {
                // 按配置集合解析
                JSONObject configResult = new JSONObject();
                JSONArray resolveResultArray = new JSONArray();
                JSONArray configArray = data.getJSONArray("items");
                if (!configArray.isEmpty()) {
                    StringBuilder configsBuilder = new StringBuilder();
                    configArray.forEach(config -> {
                        JSONObject configObj = (JSONObject) config;
                        resolveResultArray.put(resolveConfig(configObj, isPrecompile));
                        String configName = configObj.getString("name");
                        configsBuilder.append(configName).append(",");
                    });
                    configsBuilder.deleteCharAt(configsBuilder.length() - 1);
                    // 缓存配置-模块映射
                    addConfigModuleMapping(tenantName, namespaceName, environment, name, configsBuilder.toString());
                }
                configResult.put("items", resolveResultArray);
                return configResult;
            }
        } else {
            return notFoundFun.apply(result);
        }
    }

    /**
     * 配置解析
     *
     * @param response     响应结果
     * @param isPrecompile 是否预编译
     * @return 解析结果
     */
    private JSONObject resolveConfig(JSONObject response, boolean isPrecompile) {
        JSONObject content;
        // 对配置进行预编译
        if (isPrecompile) {
            content = response.getJSONObject("realContent");
        } else {
            content = response.getJSONObject("content");
        }
        // 附加全局属性
        String name = response.getString("name");
        String module = response.optString("module", "null");
        String version = response.optString("version", "null");
        JSONObject tenant = response.optJSONObject("tenant");
        JSONObject namespace = response.optJSONObject("namespace");
        content.put("$globalProp", new JSONObject()
                .put("name", name)
                .put("module", module)
                .put("environment", envType.getValue())
                .put("version", version)
                .put("namespaceName", namespace != null ? namespace.getString("f_namespace_name") : "null")
                .put("tenantAlias", tenant != null ? tenant.getString("f_tenant_uuid") : "null")
        );
        // 附加配置属性
        if (!content.has("$configProp")) {
            content.put("$configProp", new JSONObject());
        }
        // 存储缓存时，应传入发起方的租户名和环境信息，以便更好的匹配缓存
        String configName = response.getString("name");
        String tenantName = applicationUtils.getOrgName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = envType.getValue();
        // 增加缓存
        addCache(tenantName, namespaceName, environment, configName, content);
        addDatabase(tenantName, namespaceName, environment, configName, content);
        return content;
    }

    /**
     * 通过琉璃中心解析配置
     *
     * @param configType    配置类型
     * @param configContent 配置内容
     * @return 配置内容
     */
    public JSONObject parseConfigByRequest(String configType, Object configContent) {
        if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.OFFLINE) {
            return null;
        }
        LOGGER.debug("通过琉璃中心解析配置");
        JSONObject params = new JSONObject()
                .put("configType", configType)
                .put("configContent", configContent);
        R<JSONObject> result;
        String namespaceName = applicationUtils.getApplicationName();
        if (namespaceName.equals(ServiceNameConstants.LIULI_SERVICE)) {
            // 琉璃自身调用
            try {
                Object localResult = SpringUtils.getBean(ILogicServiceProxy.class).run("getConfigByClientV2", params);
                result = R.ok((JSONObject) localResult);
            } catch (ServiceException e) {
                result = R.build(e);
            }
        } else if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.INTRANET) {
            RemoteLiuLiService remoteLiuLiService = (RemoteLiuLiService) dynamicFeignClientFactory.getFeignClient(RemoteLiuLiService.class, ServiceNameConstants.LIULI_SERVICE);
            R<Map<String, Object>> remoteResult = remoteLiuLiService.parseConfigByClient(params.toString(), SecurityConstants.INNER);
            result = R.build(remoteResult.parseResponseJson());
        } else {
            String cloudConfig = RestTools.post(LiuLiApi.PARSE_CONFIG_PATH, params);
            result = R.build(new JSONObject(cloudConfig));
        }
        if (result.getCode() == HttpStatus.SUCCESS) {
            String plainText = result.getData().getString("plainStr");
            return new JSONObject(SecureTools.AESDecrypt(plainText, AES_KEY));
        } else {
            return result.parseResponseJson();
        }
    }

    /**
     * 通过请求获取未进行预编译的配置
     *
     * @param configName 配置名称
     * @return 原生配置内容
     */
    public JSONObject getNativeConfigByRequest(String configName) {
        return getConfigByRequest(configName, SearchMode.BY_CONFIG_NAME, true, (result) -> {
            throw new ServiceException(result.getMsg(), result.getCode());
        }, false);
    }

    /**
     * 配置存入缓存
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @param content       配置内容
     */
    private void addCache(String tenantName, String namespaceName, String environment, String configName, JSONObject content) {
        if (envType == EnvType.LOCAL) {
            return;
        }
        // 存放配置的key
        String configCacheKey = getConfigCacheKey(tenantName, namespaceName, environment, configName);
        // 缓存到本地
        LIULI_CONFIG_LOCAL_CACHE.put(configCacheKey, content);
        // 缓存到redis
        redisService.set(configCacheKey, content);
    }

    /**
     * 配置存入数据库
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @param content       配置内容
     */
    private void addDatabase(String tenantName, String namespaceName, String environment, String configName, JSONObject content) {
        if (envType == EnvType.LOCAL) {
            return;
        }
        // 增加配置缓存
        JSONObject params = new JSONObject();
        params.put("tenantName", tenantName);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("configName", configName);
        params.put("configContent", content);
        try {
            RemoteLogicService remoteLogicService = (RemoteLogicService) dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, ServiceNameConstants.SYSTEM_SERVICE);
            remoteLogicService.run("addLiuLiConfig", params.toString(), SecurityConstants.INNER);
        } catch (Throwable e) {
            LOGGER.error("af-system服务请求失败:{}", e.getMessage());
        }
    }

    /**
     * 配置-模块映射存入缓存
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param moduleName    模块名称
     * @param configs       配置字符串集合
     */
    private void addConfigModuleMapping(String tenantName, String namespaceName, String environment, String moduleName, String configs) {
        if (envType == EnvType.LOCAL) {
            return;
        }
        // 存放配置的key
        String configModuleMappingKey = getConfigModuleMappingKey(tenantName, namespaceName, environment, moduleName);
        // 缓存到redis
        redisService.set(configModuleMappingKey, configs);
    }

    /**
     * 删除配置缓存
     *
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     */
    public void removeCache(String namespaceName, String environment, String configName, String moduleName) {
        // 消息发布
        RTopic clientTopic = redisService.getTopic(LiuLiCacheSubscriber.CLEAR_CACHE_TOPIC_NAME);
        clientTopic.publishAsync(new JSONObject()
                .put("namespaceName", namespaceName)
                .put("environment", environment)
                .put("configName", configName)
                .put("moduleName", moduleName)
        );
    }

    public void removeCache(String namespaceName, String environment, String configName) {
        removeCache(namespaceName, environment, configName, "*");
    }

    public void removeCacheKeys(String namespaceName, String environment, List<String> cacheKeys) {
        for (String key : cacheKeys) {
            removeCache(namespaceName, "prod", key);
        }
    }

    /**
     * 查询模式
     */
    public enum SearchMode {
        BY_CONFIG_NAME,
        BY_MODULE_NAME
    }

    private static class EmptyJSONObject extends JSONObject {
        public static final EmptyJSONObject SELF = new EmptyJSONObject();

        private EmptyJSONObject() {

        }

        @Override
        public String toString() {
            return null;
        }
    }
}
