package com.af.v4.system.common.liuli.utils;

import cn.hutool.core.date.DateException;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Tuple;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.common.liuli.utils.enums.JoinTypeEnum;
import com.af.v4.system.common.liuli.utils.enums.QueryTypeEnum;
import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * SQL解析工具类
 */
public class SQLParserUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(SQLParserUtil.class);

    private static final String DEFAULT_ALL_VALUE = "全部";

    private static final String DEFAULT_NULL_VALUE = "NULL";

    /**
     * 移动端固定参数，用于多条件模糊查询，查询范围为(= like)
     */
    private static final String MOBILE_FIXED_QUERY_KEY = "$queryValue";

    public SQLParserUtil() {
    }

    /**
     * 生成完整可执行的querySQL和countSQL语句，以及查询列，表名，条件表达式的SQL片段
     *
     * @param queryParamsMap 配置文件
     * @param params         前端传入的表单参数
     * @param sortField      前端传入的排序字段
     * @param sortOrder      前端传入的排序方式
     * @return 完整可执行的querySQL和countSQL语句，以及查询列，表名，条件表达式的SQL片段
     */
    public static JSONObject getSingleStyleQuerySQL(JSONObject queryParamsMap, JSONObject params, String sortField, String sortOrder) {
        JSONObject querySQlMap = getQuerySQL(queryParamsMap, params, sortField, sortOrder);
        String querySql = querySQlMap.getString("querySql");
        String countSql = querySQlMap.getString("countSql");
        JSONObject result = new JSONObject();
        result.put("items", extractSqlPart(querySql, "SELECT", "FROM"));
        result.put("tableName", extractSqlPart(querySql, "FROM", "WHERE"));
        result.put("condition", querySql.substring(querySql.indexOf("WHERE") + "WHERE".length()).trim());
        result.put("countItems", extractSqlPart(countSql, "SELECT", "FROM"));
        result.put("countTableName", extractSqlPart(countSql, "FROM", "WHERE"));
        result.put("countCondition", countSql.substring(countSql.indexOf("WHERE") + "WHERE".length()).trim());
        result.put("querySql", querySql);
        result.put("countSql", countSql);
        return result;
    }

    /**
     * 生成完整可执行的querySQL和countSQL语句
     *
     * @param queryParamsMap 配置文件资源
     * @param params         前端传入的表单参数
     * @param sortField      前端传入的排序字段
     * @param sortOrder      前端传入的排序方式
     * @return 完整可执行的querySQL、countSQL、tableSummarySql语句
     */
    public static JSONObject getQuerySQL(JSONObject queryParamsMap, JSONObject params, String sortField, String sortOrder) {
        // 获取querySql和countSql片段
        StringBuilder querySql = new StringBuilder(queryParamsMap.getString("querySql"));
        StringBuilder countSql = new StringBuilder(queryParamsMap.getString("countSql"));
        StringBuilder tableSummarySql = new StringBuilder(queryParamsMap.optString("tableSummarySql"));
        // 向SQL语句中追加JOIN语句和WHERE条件
        appendWhereAndJoinCondition(querySql, countSql, tableSummarySql, queryParamsMap, params);
        // 向SQL语句中追加OrderBy
        appendOrderBy(querySql, queryParamsMap, sortField, sortOrder);
        // 如果预设了表名后缀 开始拼接后缀
        queryParamsMap.optJSONArray("tabNameSuffix", new JSONArray()).forEach(item -> {
            JSONObject tabNameSuffix = (JSONObject) item;
            String tabName = tabNameSuffix.getString("tableName").split(" ")[0];
            String suffixModel = tabNameSuffix.getString("column");
            String suffixValue = params.optString(suffixModel, null);
            if (querySql.indexOf(tabName) != -1 && suffixValue != null) {
                String suffixTableName = STR."\{tabName}_\{suffixValue.replace("-", "")}";
                replaceInStringBuilder(querySql, tabName, suffixTableName);
                replaceInStringBuilder(countSql, tabName, suffixTableName);
                replaceInStringBuilder(tableSummarySql, tabName, suffixTableName);
            }
        });
        // 返回结果集
        JSONObject result = new JSONObject();
        result.put("querySql", querySql.toString());
        result.put("countSql", countSql.toString());
        result.put("tableSummarySql", tableSummarySql.toString());
        return result;
    }

    /**
     * 追加JOIN表达式
     *
     * @param joinTableAlias        关联的表别名
     * @param tableAliasName        主表的别名
     * @param joinTableNameObject   配置文件预设的关联表集合
     * @param canJoinTableNameArray 查询需要被关联的表集合
     * @param useJoinByCount        生成count语句时是否需要join
     */
    public static void putJoinCondition(String joinTableAlias, String tableAliasName, JSONObject joinTableNameObject,
                                        JSONArray canJoinTableNameArray, Boolean useJoinByCount) {
        // 如果数据字段是主表的，或者是'$'，直接跳过
        if (joinTableAlias.equals(tableAliasName) || joinTableAlias.equals("$")) {
            return;
        }
        // 检查数据字段对应的表别名是否在配置文件预设的关联表集合中
        if (!joinTableNameObject.has(joinTableAlias)) {
            throw new ServiceException(STR."在组织JOIN表达式时，没有找到关于表别名[\{joinTableAlias}]的JOIN表达式配置信息", HttpStatus.BAD_REQUEST);
        }
        // 检查该表别名是否在查询需要被关联的表集合中
        boolean isContains = getCanJoinTableItem(canJoinTableNameArray, joinTableAlias) != null;
        if (!isContains) {
            addNewJoinCondition(canJoinTableNameArray, joinTableNameObject, joinTableAlias, tableAliasName, useJoinByCount);
        } else {
            updateJoinCondition(canJoinTableNameArray, joinTableNameObject, joinTableAlias, tableAliasName, useJoinByCount);
        }
    }

    private static void addNewJoinCondition(JSONArray canJoinTableNameArray,
                                            JSONObject joinTableNameObject,
                                            String joinTableAlias,
                                            String tableAliasName,
                                            boolean useJoinByCount) {
        JSONObject joinObj = joinTableNameObject.getJSONObject(joinTableAlias);
        String joinType = joinObj.getString("type");
        // 取到JOIN语句内容
        String joinCondition = joinObj.getString("value");
        JSONObject params = new JSONObject();
        params.put("type", joinType);
        params.put("alias", joinTableAlias);
        params.put("useJoinByCount", useJoinByCount);
        if (joinType.equals("join")) {
            joinCondition = joinCondition.toLowerCase();
            // 判断该JOIN表达式是否需要处理多表关联的情况
            String shouldJoinTableAlias = getShouldJoinTableAlias(joinTableAlias, tableAliasName, joinCondition);
            // 如果返回了需要额外关联的表别名
            if (shouldJoinTableAlias != null) {
                // 递归处理JOIN表达式
                putJoinCondition(shouldJoinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameArray, useJoinByCount);
            }
            // 默认使用LEFT JOIN，加入到需要被关联的表集合中
            params.put("value", STR."\{JoinTypeEnum.LEFT_OUT_JOIN.getValue()} \{joinCondition}");
        } else {
            params.put("value", joinCondition);
        }
        canJoinTableNameArray.put(params);
    }

    private static void updateJoinCondition(JSONArray canJoinTableNameArray,
                                            JSONObject joinTableNameObject,
                                            String joinTableAlias,
                                            String tableAliasName,
                                            boolean useJoinByCount) {
        // 更新这个JOIN表达式在生成count语句时的JOIN策略
        JSONObject params = getCanJoinTableItem(canJoinTableNameArray, joinTableAlias);
        assert params != null;
        params.put("useJoinByCount", useJoinByCount);
        // 如果这个JOIN表达式在生成count语句时需要JOIN，则需要判断是否有多表关联的情况
        if (!useJoinByCount) {
            return;
        }
        JSONObject joinObj = joinTableNameObject.getJSONObject(joinTableAlias);
        String joinType = joinObj.getString("type");
        if (!joinType.equals("join")) {
            return;
        }
        // 取到JOIN语句内容
        String joinCondition = joinObj.getString("value").toLowerCase();
        // 判断该JOIN表达式是否需要处理多表关联的情况
        String shouldJoinTableAlias = getShouldJoinTableAlias(joinTableAlias, tableAliasName, joinCondition);
        // 检查是否有需要额外关联的表别名
        if (shouldJoinTableAlias == null) {
            return;
        }
        // 更新表别名对应的JOIN表达式在生成count语句时的JOIN策略
        params = getCanJoinTableItem(canJoinTableNameArray, shouldJoinTableAlias);
        assert params != null;
        params.put("useJoinByCount", true);
        // 递归处理JOIN表达式
        putJoinCondition(shouldJoinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameArray, true);
    }

    /**
     * 获取指定别名的表结构的join信息
     *
     * @param canJoinTableNameArray 可以join的表集合
     * @param alias                 表别名
     * @return join信息
     */
    private static JSONObject getCanJoinTableItem(JSONArray canJoinTableNameArray, String alias) {
        for (Object itemObject : canJoinTableNameArray) {
            JSONObject item = (JSONObject) itemObject;
            if (item.getString("alias").equals(alias)) {
                return item;
            }
        }
        return null;
    }

    /**
     * 判断JOIN表达式是否涉及多表关联，返回额外需要关联的表别名。
     *
     * <p>
     * 处理策略：因为不是所有业务都是主表关联子表，还有子表A关联子表B的情况，此时子表B可能还没有加入到需要被关联的表集合中
     * 所以此处需要进行判断，看是否需要处理多表关联的情况
     * 依次拿到JOIN表达式的第一个表别名和第二个表别名，看是不是自身表的别名或者是主表别名，如果不是，则代表有多表关联的情况
     * 如果有多表关联的情况，该方法会返回需要额外关联的表别名，可以再通过putJoinCondition方法追加JOIN表达式
     *
     * @param joinTableAlias 关联的表别名
     * @param tableAliasName 主表的别名
     * @param joinCondition  JOIN表达式
     * @return 需要关联的额外表别名，如果不涉及多表关联则返回null
     */
    private static String getShouldJoinTableAlias(String joinTableAlias, String tableAliasName, String joinCondition) {
        String onCondition = joinCondition.substring(joinCondition.indexOf(" on ") + 4);
        // 第一个表别名
        int firstAliasIndex = onCondition.indexOf(".");
        String firstAlias = onCondition.substring(0, firstAliasIndex);
        // 第二个表别名
        String conditionPart = onCondition.substring(firstAliasIndex + 1);
        String secondAlias = conditionPart.substring(conditionPart.indexOf("=") + 1, conditionPart.indexOf(".")).trim();
        // 检查别名是否是主表别名，如果不是说明要处理多表关联情况
        if (firstAlias.equals(tableAliasName) || secondAlias.equals(tableAliasName)) {
            return null;
        }
        // 如果第一个表别名是传入的表别名，则取第二个表别名
        return firstAlias.equals(joinTableAlias) ? secondAlias : firstAlias;
    }

    /**
     * 根据查询类型拼接条件
     *
     * @param condition   构造的查询条件
     * @param queryParams 查询配置
     * @param realKey     真实字段名
     * @param value       字段值
     * @param dbType      数据库类型
     * @param mobileQueryIndex 移动端固定参数集索引
     * @param mobileQueryLength 移动端固定参数集长度
     */
    private static void appendConditionByType(StringBuilder condition, JSONObject queryParams, String key, String realKey, String value, DbType dbType, int mobileQueryIndex, int mobileQueryLength) {
        // 获取查询类型
        String queryTypeStr = queryParams.getString("queryType");
        if (!QueryTypeEnum.is(queryTypeStr) && !key.equals(MOBILE_FIXED_QUERY_KEY)) {
            throw new ServiceException(STR."表单字段[\{key}]无法生成查询条件，因为配置的查询类型不存在：\{queryTypeStr}", HttpStatus.BAD_REQUEST);
        }
        QueryTypeEnum queryType;
        if (key.equals(MOBILE_FIXED_QUERY_KEY)) {
            queryType = QueryTypeEnum.INNER_LIKE;
        } else {
            queryType = QueryTypeEnum.toType(queryTypeStr);
        }
        // 拼接条件表达式
        switch (queryType) {
            case EQUALS, NO_EQUALS, LESS_THAN, LESS_THAN_EQUALS, GREATER_THAN, GREATER_THAN_EQUALS -> {
                String mark = queryType.getValue();
                if (DEFAULT_NULL_VALUE.equalsIgnoreCase(value)) {
                    condition.append("\n\tAND ").append(realKey).append(STR." IS \{DEFAULT_NULL_VALUE}");
                } else {
                    condition.append("\n\tAND ").append(realKey).append(" ").append(mark).append(" '").append(value).append("'");
                }
            }
            case INNER_LIKE -> {
                if (key.equals(MOBILE_FIXED_QUERY_KEY)) {
                    if (mobileQueryIndex == 0) {
                        condition.append("\n\tAND (").append(realKey).append(" LIKE '%").append(value).append("%'");
                    } else {
                        condition.append("\n\tOR ").append(realKey).append(" LIKE '%").append(value).append("%'");
                        if (mobileQueryIndex == mobileQueryLength - 1) {
                            condition.append(")");
                        }
                    }
                } else {
                    condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("%'");
                }
            }
            case LEFT_LIKE -> condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("'");
            case RIGHT_LIKE ->
                    condition.append("\n\tAND ").append(realKey).append(" LIKE '").append(value).append("%'");
            case IN -> {
                // 检查值是否是 JSON 数组并合并唯一值
                if (value.startsWith("[") && value.endsWith("]")) {
                    value = CommonTools.union(JsonTools.parseArray(value));
                }

                String[] values = value.split(",");
                int length = values.length;

                // 如果值的数量超过 1000，则分批处理
                if (length > 1000) {
                    condition.append("\n\tAND (");
                    // 按每 1000 个值一组处理
                    for (int i = 0; i < length; i += 1000) {
                        if (i > 0) {
                            condition.append("\n\tOR ");
                        }
                        // 拼接当前批次的值
                        String subCondition = String.join(", ", Arrays.copyOfRange(values, i, Math.min(i + 1000, length)));
                        condition.append(realKey).append(" IN (").append(subCondition).append(")");
                    }
                    condition.append("\n\t)");
                } else {
                    // 如果值的数量不超过 1000，则作为一个单独的 IN 子句处理
                    condition.append("\n\tAND ").append(realKey).append(" IN (").append(value).append(")");
                }
            }

            case NOT_IN -> {
                if (value.startsWith("[") && value.endsWith("]")) {
                    value = CommonTools.union(JsonTools.parseArray(value));
                }
                condition.append("\n\tAND ").append(realKey).append(" NOT IN (").append(value).append(")");
            }
            case BETWEEN -> {
                JSONArray array = new JSONArray(value);
                String beginValue = array.get(0).toString();
                String endValue = array.get(1).toString();

                if (!beginValue.isEmpty() && !endValue.isEmpty()) {
                    try {
                        beginValue = DateUtil.parse(beginValue).toString();
                        endValue = DateUtil.parse(endValue).toString();
                        if (DbType.oracle == dbType) {
                            if (beginValue.length() <= 10) {
                                beginValue = STR."TO_DATE('\{beginValue}', 'yyyy-MM-dd')";
                                endValue = STR."TO_DATE('\{endValue}', 'yyyy-MM-dd')";
                            } else {
                                beginValue = STR."TO_DATE('\{beginValue}', 'yyyy-MM-dd HH24:mi:ss')";
                                endValue = STR."TO_DATE('\{endValue}', 'yyyy-MM-dd HH24:mi:ss')";
                            }
                        } else {
                            beginValue = STR."'\{beginValue}'";
                            endValue = STR."'\{endValue}'";
                        }
                    } catch (DateException e) {
                        beginValue = STR."'\{beginValue}'";
                        endValue = STR."'\{endValue}'";
                    }

                    condition.append("\n\tAND ")
                            .append(realKey)
                            .append(" BETWEEN ")
                            .append(beginValue)
                            .append(" AND ")
                            .append(endValue);
                } else if (!beginValue.isEmpty()) {
                    condition.append("\n\tAND ")
                            .append(realKey)
                            .append(" >= ")
                            .append(beginValue);
                } else if (!endValue.isEmpty()) {
                    condition.append("\n\tAND ")
                            .append(realKey)
                            .append(" <= ")
                            .append(endValue);
                }
            }
            default -> throw new ServiceException(STR."不支持的查询类型：\{queryType.getValue()}", HttpStatus.BAD_REQUEST);
        }
    }
    private static void appendConditionByType(StringBuilder condition, JSONObject queryParams, String key, String realKey, String value, DbType dbType) {
        appendConditionByType(condition, queryParams, key, realKey, value, dbType, 0, 0);
    }

    private static void appendWhereAndJoinCondition(StringBuilder querySql, StringBuilder tableSummarySql, StringBuilder countSql, JSONObject queryParamsMap, JSONObject params) {
        // 获取关联表配置信息
        JSONObject joinTableNameObject = queryParamsMap.getJSONObject("joinTableNameObject");
        // 获取查询时需要被关联的表集合
        JSONArray canJoinTableNameArray = queryParamsMap.getJSONArray("canJoinTableNameArray");
        // 获取主表的别名
        String tableAliasName = queryParamsMap.getString("tableAliasName");
        // 获取固定查询条件
        String conditionStr = queryParamsMap.getString("condition");
        // 当配置了主表数据源sql的时候 不强制校验条件是否配置了表单项
        boolean isCheck = queryParamsMap.has("queryBeforeAggSql");
        String whereConditionStr = getConditionValue(queryParamsMap.getJSONObject("selectColumn"), params, tableAliasName, joinTableNameObject, canJoinTableNameArray, conditionStr, isCheck);
        // 遍历需要JOIN的表，向SQL语句中追加JOIN条件
        canJoinTableNameArray.forEach(item -> {
            JSONObject joinObject = (JSONObject) item;
            String condition = joinObject.getString("value");
            querySql.append("\n\t").append(condition);
            // 判断是否需要在计数SQL中使用JOIN
            if (joinObject.getBoolean("useJoinByCount")) {
                countSql.append("\n\t").append(condition);
                tableSummarySql.append("\n\t").append(condition);
            }
        });
        // 追加where表达式到SQL中
        querySql.append("\nWHERE ").append(whereConditionStr);
        countSql.append("\nWHERE ").append(whereConditionStr);
        tableSummarySql.append("\nWHERE ").append(whereConditionStr);
    }

    private static void appendOrderBy(StringBuilder querySql, JSONObject queryParamsMap, String sortField, String sortOrder) {
        // 获取查询的排序字段
        String orderBy;
        if (sortField == null) {
            // 默认排序
            orderBy = queryParamsMap.getString("orderBy");
        } else {
            JSONObject queryParams = queryParamsMap.getJSONObject("selectColumn").getJSONObject(sortField);
            Tuple tuple = getRealKeyAndTableAlias(queryParams, sortField);
            String realKey = tuple.get(0);
            // 根据排序方式选择升序或降序
            orderBy = Optional.ofNullable(sortOrder).filter("descend"::equals).map(o -> STR."\{realKey} DESC").orElse(realKey);
        }
        // 追加orderBy表达式到SQL中
        querySql.append("\nORDER BY ").append(orderBy);
    }

    /**
     * 生成查询条件
     *
     * @param queryMap              动态生成表单查询条件所用的数据列集合
     * @param params                前端传入的表单参数
     * @param tableAliasName        主表的别名
     * @param joinTableNameObject   获取配置文件预设的关联表集合
     * @param canJoinTableNameArray 获取查询时需要被关联的表集合
     * @param conditionStr          固定查询表达式
     * @return 查询条件的SQL表达式
     */
    public static String getConditionValue(JSONObject queryMap, JSONObject params, String tableAliasName,
                                           JSONObject joinTableNameObject, JSONArray canJoinTableNameArray, String conditionStr) {
        return getConditionValue(queryMap, params, tableAliasName, joinTableNameObject, canJoinTableNameArray, conditionStr, false);
    }

    private static String getConditionValue(JSONObject selectColumn, JSONObject params, String tableAliasName,
                                            JSONObject joinTableNameObject, JSONArray canJoinTableNameArray, String conditionStr, boolean skipCheck) {
        DbType dbType = DynamicDataSource.getDbType();
        // 查询条件
        StringBuilder condition = new StringBuilder(conditionStr);
        // 遍历前端传入的表单参数
        Iterator<String> iterator = params.keys();
        while (iterator.hasNext()) {
            String key = iterator.next();
            String value = String.valueOf(params.get(key));
            // 判断是否作为查询表达式参数
            if (conditionStr.contains(STR."{\{key}}")) {
                replaceInStringBuilder(condition, STR."{\{key}}", value);
                continue;
            }
            // 检查前端传入的表单参数
            if (value.isEmpty() || DEFAULT_ALL_VALUE.equals(value)) {
                continue;
            }
            if (!selectColumn.has(key) && !key.equals(MOBILE_FIXED_QUERY_KEY)) {
                if (!skipCheck) {
                    LOGGER.warn("表单字段[{}]无法生成查询条件，因为没有配置这个字段的相关信息", key);
                }
                continue;
            }
            if (key.equals(MOBILE_FIXED_QUERY_KEY)) {
                List<Object> queryList = selectColumn.toMap().values().stream().filter(item -> {
                    Map<String, Object> queryParams = (Map<String, Object>) item;
                    QueryTypeEnum queryTypeEnum = QueryTypeEnum.toType(queryParams.get("queryType").toString());
                    // 判断queryType，是否是移动端可以支持的查询类型
                    return queryTypeEnum == QueryTypeEnum.EQUALS ||
                            queryTypeEnum == QueryTypeEnum.LEFT_LIKE ||
                            queryTypeEnum == QueryTypeEnum.RIGHT_LIKE ||
                            queryTypeEnum == QueryTypeEnum.INNER_LIKE;
                    }).toList();
                // 循环queryList，拿到所有可以生成查询条件的列
                int index = 0;
                int length = queryList.size();
                for (Object item : queryList){
                    JSONObject queryParams = new JSONObject((Map<String, Object>) item);
                    Tuple tuple = getRealKeyAndTableAlias(queryParams, key);
                    // 根据查询类型拼接条件
                    appendConditionByType(condition, queryParams, key, tuple.get(0), value, dbType, index, length);
                    // 追加JOIN表达式，在执行countSQL语句时,用于查询条件的子表数据列需要JOIN子表
                    putJoinCondition(tuple.get(1), tableAliasName, joinTableNameObject, canJoinTableNameArray, true);
                    index++;
                }
            } else {
                JSONObject queryParams = selectColumn.getJSONObject(key);
                Tuple tuple = getRealKeyAndTableAlias(queryParams, key);
                // 根据查询类型拼接条件
                appendConditionByType(condition, queryParams, key, tuple.get(0), value, dbType);
                // 追加JOIN表达式，在执行countSQL语句时,用于查询条件的子表数据列需要JOIN子表
                putJoinCondition(tuple.get(1), tableAliasName, joinTableNameObject, canJoinTableNameArray, true);
            }
        }
        return condition.toString();
    }

    private static Tuple getRealKeyAndTableAlias(JSONObject queryParams, String key) {
        // 处理key，获取真实的字段名
        if (key.indexOf('.') == -1 || key.startsWith("$_")) {
            key = key.replaceFirst("_", ".");
        }
        // 获取该列名称（表别名.字段名）
        String realKey = queryParams.getString("key");
        String tableAlias = realKey.substring(0, realKey.indexOf('.'));
        // 表别名如果是$，代表是函数，格式：$.datediff(u.date,i.date) day
        if (tableAlias.equals("$")) {
            String[] spiltKeyArray = key.split(" ");
            realKey = spiltKeyArray[0].substring(2);
        }
        return new Tuple(realKey, tableAlias);
    }

    /**
     * 截取SQL片段
     */
    private static String extractSqlPart(String sql, String startKeyword, String endKeyword) {
        int startIndex = sql.indexOf(startKeyword) + startKeyword.length();
        int endIndex = sql.indexOf(endKeyword, startIndex);
        return sql.substring(startIndex, endIndex).trim();
    }

    /**
     * @param fields 需要查询的字段别名
     *               例如：['s_id','s_f_collection']
     * @param sql    查询SQL
     *               替换查询字段 仅支持处理查询配置生成的 sql
     */
    public static String replaceQueryItem(JSONArray fields, String sql) {
        // 获取原始字段
        String originalField = extractSqlPart(sql, "SELECT", "FROM").trim();
        if (originalField.isEmpty()) {
            LOGGER.warn("发现指定查询字段，但是获取原始sql查询字段错误，返回原SQL");
            return sql;
        }

        // 将目标字段转为集合
        List<String> targetFields = fields.toList().stream()
                .map(Object::toString)
                .toList();

        // 构造一个映射表：别名 -> 完整字段
        Map<String, String> aliasToFieldMap = Arrays.stream(originalField.split(",\n"))
                .map(String::trim)
                .collect(Collectors.toMap(
                        item -> {
                            String[] parts = item.split(" ");
                            return parts[parts.length - 1].replace("\"", ""); // 获取最后一个部分作为别名
                        },
                        item -> item,
                        (a, b) -> a // 处理重复字段，取第一个
                ));

        // 按目标字段列表顺序过滤并生成新的字段列表
        List<String> filteredFields = targetFields.stream()
                .filter(aliasToFieldMap::containsKey) // 检查别名是否存在于映射表中
                .map(aliasToFieldMap::get)            // 根据别名获取完整字段
                .toList();

        // 检查过滤后的字段是否为空
        if (filteredFields.isEmpty()) {
            LOGGER.warn("发现指定查询字段，但是未在原始sql筛选出字段，返回原SQL");
            return sql;
        }

        // 拼接新的字段列表
        String newField = String.join(",\n ", filteredFields);

        // 替换原始字段
        return sql.replace(originalField, newField);
    }

    // 在 StringBuilder 中替换字符串
    private static void replaceInStringBuilder(StringBuilder sb, String oldStr, String newStr) {
        int index = sb.indexOf(oldStr);
        while (index != -1) {
            sb.replace(index, index + oldStr.length(), newStr);
            index = sb.indexOf(oldStr, index + newStr.length());
        }
    }

    //获取最内层第一个表名
    public static String findDeepestTableName(String sql) {
        Pattern fromPattern = Pattern.compile("FROM\\s+(\\w+)", Pattern.CASE_INSENSITIVE);
        Matcher fromMatcher = fromPattern.matcher(sql); // 创建Matcher对象用于在sql字符串上查找FROM后的第一个表名
        String result = null; // 初始化结果字符串
        if (fromMatcher.find()) {
            // 如果成功匹配到FROM后的第一个表名，则将表名赋值给result
            result = fromMatcher.group(1);
        }
        return result;
    }

    /**
     * 获取子表项集合
     *
     * @param formJson 表单配置
     * @param formData 表单数据
     * @return 子表项集合
     */
    public static List<JSONObject> getChildTableItems(JSONArray formJson, JSONObject formData) {
        List<JSONObject> childTableFormItems = new ArrayList<>(1);
        // 筛选出表单中子表
        formJson.forEach(item -> {
            JSONObject formItem = (JSONObject) item;
            if (formItem.getString("type").equals("childTable")) {
                childTableFormItems.add(formItem);
            }
        });
        if (childTableFormItems.isEmpty()) {
            return childTableFormItems;
        }
        // 传入表单项中包含了子表项才处理
        return childTableFormItems.stream().filter(item -> {
            String modal = item.getString("model");
            return formData.has(modal);
        }).toList();
    }
}
