package com.af.v4.system.common.liuli.service;

import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.utils.ApplicationUtils;
import com.af.v4.system.common.liuli.utils.LiuLiUtil;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.redis.RedisService;
import org.json.JSONObject;
import org.springframework.stereotype.Service;

import java.util.concurrent.atomic.AtomicReference;

/**
 * 琉璃-配置服务
 *
 * @author Mr.river
 */
@Service
public class LiuLiConfigService {
    /**
     * 获取客户端配置
     */
    private static final String GET_CONFIG_PATH = "/liuli/logic/getConfigByClient";

    private final LiuLiUtil liuLiUtil;
    private final RedisService redisService;
    private final ApplicationUtils applicationUtils;

    public LiuLiConfigService(LiuLiUtil liuLiUtil, RedisService redisService, ApplicationUtils applicationUtils) {
        this.liuLiUtil = liuLiUtil;
        this.redisService = redisService;
        this.applicationUtils = applicationUtils;
    }

    /**
     * 获取配置文件
     *
     * @param configName 配置名称
     * @return 配置文件信息
     */
    public JSONObject get(String configName) {
        String configRedisKey = CacheConstants.CONFIG_DATA_CACHE_KEY + configName;
        //根据配置名称从本地redis中获取配置文件
        AtomicReference<JSONObject> config = new AtomicReference<>(redisService.get(configRedisKey));
        if (config.get() == null) {
            redisService.lock(configRedisKey, () -> {
                config.set(redisService.get(configRedisKey));
                if (config.get() == null) {
                    String tenantUUID = redisService.get(CacheConstants.TENANT_UUID_KEY);
                    //调用琉璃中心需传递的参数
                    JSONObject jsonObject = new JSONObject();
                    jsonObject.put("tenantUUID", tenantUUID);
                    jsonObject.put("namespaceName", applicationUtils.getApplicationName());
                    jsonObject.put("environment", applicationUtils.getEnvType().getValue());
                    jsonObject.put("configName", configName);

                    String requestPath = liuLiUtil.getCloudUrl() + GET_CONFIG_PATH;
                    String cloudConfig = RestTools.post(requestPath, jsonObject);
                    JSONObject result = new JSONObject(cloudConfig);
                    if (result.getInt("code") == 200) {
                        JSONObject content = result.getJSONObject("data").getJSONObject("content");
                        redisService.set(configRedisKey, content);
                        config.set(content);
                    }
                    config.set(result);
                }
            });
        }
        return config.get();
    }

    /**
     * 删除指定配置的缓存
     *
     * @param configName 配置名称
     */
    public void removeCache(String configName) {
        //存放配置的key
        String configRedisKey = CacheConstants.CONFIG_DATA_CACHE_KEY + configName;
        redisService.lock(configRedisKey, () -> {
            redisService.delete(configRedisKey);
        });
    }
}
