package com.af.v4.system.common.liuli.utils;

import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.enums.CompatibleVersion;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.common.liuli.utils.enums.OSTypeEnum;
import com.af.v4.system.common.security.utils.SecurityUtils;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.net.URL;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

/**
 * 应用工具
 */
@Component
public final class ApplicationUtils {

    private static final String SYSTEM_V4_VERSION;
    private static final String APP_VERSION;
    private static final OSTypeEnum OS_TYPE;

    static {
        // SystemV4版本
        SYSTEM_V4_VERSION = ApplicationUtils.class.getPackage().getImplementationVersion();
        // 应用版本
        String appVersion;
        ClassLoader cl = ApplicationUtils.class.getClassLoader();
        URL url = cl.getResource("META-INF/MANIFEST.MF");
        Manifest manifest;
        if (url != null) {
            try {
                manifest = new Manifest(url.openStream());
                Attributes mainAttributes = manifest.getMainAttributes();
                appVersion = mainAttributes.getValue("Implementation-Version");
            } catch (IOException e) {
                appVersion = "0";
            }
        } else {
            appVersion = "0";
        }
        APP_VERSION = appVersion;
        // 操作系统
        String osName = System.getProperty("os.name").toLowerCase();
        if (osName.contains("linux")) {
            OS_TYPE = OSTypeEnum.LINUX;
        } else {
            OS_TYPE = OSTypeEnum.WINDOWS;
        }
    }

    private String DEFAULT_TENANT_NAME = null;
    private EnvType ENV_TYPE = null;
    private String APPLICATION_NAME = null;
    private CompatibleVersion COMPATIBLE_VERSION = null;

    /**
     * 获取琉璃架构版本
     *
     * @return 琉璃架构版本
     */
    public static String getSystemV4Version() {
        return SYSTEM_V4_VERSION;
    }

    /**
     * 获取当前数据源类型
     *
     * @return 数据源类型
     */
    public static DbType getDbType() {
        return DynamicDataSource.getDbType();
    }

    public static OSTypeEnum getOSType() {
        return OS_TYPE;
    }

    /**
     * 获取应用版本
     *
     * @return 应用版本
     */
    public static String getAppVersion() {
        return APP_VERSION;
    }

    /**
     * 获取当前运行环境
     *
     * @return 运行环境枚举
     */
    public EnvType getEnvType() {
        return ENV_TYPE;
    }

    @Value("${spring.profiles.active}")
    public void setEnvType(String envType) {
        ENV_TYPE = EnvType.toType(envType);
        if (ENV_TYPE == null) {
            throw new ServiceException("不支持的运行环境类型[" + envType + "]", HttpStatus.CONFIG_ERROR);
        }
    }

    public String getEnvTypeValue() {
        return ENV_TYPE.getValue();
    }

    /**
     * 获取应用名称
     *
     * @return 应用名称
     */
    public String getApplicationName() {
        return APPLICATION_NAME;
    }

    @Value("${spring.application.name}")
    public void setApplicationName(String applicationName) {
        APPLICATION_NAME = applicationName;
    }

    /**
     * 获取租户名称
     *
     * @return 租户名称
     */
    public String getTenantName() {
        LoginUser loginUser = SecurityUtils.getLoginUser();
        if (loginUser != null) {
            String userTenantName = loginUser.getTenantName();
            if (userTenantName != null) {
                return userTenantName;
            }
        }
        return DEFAULT_TENANT_NAME;
    }

    @Value("${tenant.name}")
    public void setTenantName(String tenantName) {
        this.DEFAULT_TENANT_NAME = tenantName;
    }

    /**
     * 获取应用兼容版本
     *
     * @return 应用兼容版本
     */
    public CompatibleVersion getCompatibleVersion() {
        return COMPATIBLE_VERSION;
    }

    @Value("${tenant.version}")
    public void setCompatibleVersion(String version) {
        this.COMPATIBLE_VERSION = CompatibleVersion.toType(version);
    }

    /**
     * 获取JSON内容
     *
     * @return JSON内容
     */
    public JSONObject getValues() {
        return new JSONObject()
                .put("tenantName", getTenantName())
                .put("envType", getEnvTypeValue())
                .put("applicationName", getApplicationName())
                .put("compatibleVersion", COMPATIBLE_VERSION.getValue())
                .put("appVersion", getAppVersion())
                .put("dbType", getDbType().name())
                .put("osType", getOSType().getValue())
                .put("systemV4Version", getSystemV4Version());
    }
}
