package com.af.v4.system.common.liuli.config.service;

import cn.hutool.cache.CacheUtil;
import cn.hutool.cache.impl.LRUCache;
import com.af.v4.system.api.RemoteLogicService;
import com.af.v4.system.api.factory.DynamicFeignClientFactory;
import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.liuli.config.ConfigParser;
import com.af.v4.system.common.liuli.config.enums.LiuLiConfigTypeEnum;
import com.af.v4.system.common.liuli.config.subscriber.LiuLiCacheSubscriber;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.liuli.utils.LiuLiUtil;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.redis.RedisService;
import org.json.JSONArray;
import org.json.JSONObject;
import org.redisson.api.RTopic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Objects;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * 琉璃-配置服务
 *
 * @author Mr.river
 */
@Service
public class LiuLiConfigService {
    /**
     * 用于获取命名空间下所有配置的配置名称
     */
    public static final String NAMESPACE_CONFIGS_KEY = "_nameSpaceGroup_";
    /**
     * 本地缓存
     */
    private static final LRUCache<String, JSONObject> LIULI_CONFIG_LOCAL_CACHE = CacheUtil.newLRUCache(200);
    /**
     * 获取客户端配置
     */
    private static final String GET_CONFIG_PATH = "/af-liuli/logic/openapi/getConfigByClient";
    private static final Logger LOGGER = LoggerFactory.getLogger(LiuLiConfigService.class);
    private static EnvType envType = null;
    private final LiuLiUtil liuLiUtil;
    private final ApplicationUtils applicationUtils;
    private final RedisService redisService;
    private final DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory;

    public LiuLiConfigService(LiuLiUtil liuLiUtil, ApplicationUtils applicationUtils, RedisService redisService, DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory) {
        this.liuLiUtil = liuLiUtil;
        this.applicationUtils = applicationUtils;
        this.redisService = redisService;
        this.dynamicFeignClientFactory = dynamicFeignClientFactory;
        envType = applicationUtils.getEnvType();
    }

    /**
     * 获取配置项的缓存Key
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @return key
     */
    public static String getConfigCacheKey(String tenantName, String namespaceName, String environment, String configName) {
        return CacheConstants.CONFIG_DATA_CACHE_KEY + tenantName + "_" + namespaceName + "_" + environment + "_" + configName;
    }

    public static LRUCache<String, JSONObject> getLiuliConfigLocalCache() {
        return LIULI_CONFIG_LOCAL_CACHE;
    }

    /**
     * 获取配置文件
     *
     * @param configName 配置名称
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject get(String configName) {
        return get(configName, false);
    }

    /**
     * 获取当前租户的配置文件（不从上级寻找）
     *
     * @param configName 配置名称
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject getBySelf(String configName) {
        return get(configName, false, false);
    }

    /**
     * 获取当前租户和命名空间下的所有配置
     *
     * @return 配置文件集合，按照[{配置名称:配置内容}]的格式返回
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject getAll() {
        return getAll(false);
    }

    /**
     * 获取配置文件
     *
     * @param configName                       配置名称
     * @param isThrowExceptionOnConfigNotFound 配置找不到时是否抛出异常
     * @param shouldGetConfigByParent          找不到配置是否寻找上级配置
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject get(String configName, Boolean isThrowExceptionOnConfigNotFound, Boolean shouldGetConfigByParent) {
        if (configName == null) {
            LOGGER.error("配置名不能为null");
            return null;
        }
        JSONObject configInfo = getConfigByCache(configName, (key) ->
                redisService.syncLock(key, () -> getConfigByCache(configName,
                        (key2) -> getConfigByDatabase(configName,
                                () -> getConfigByRequest(configName, shouldGetConfigByParent, result -> {
                                    if (!isThrowExceptionOnConfigNotFound) {
                                        LOGGER.debug("获取配置失败，相关信息：{}", result);
                                        String tenantName = applicationUtils.getTenantName();
                                        String namespaceName = applicationUtils.getApplicationName();
                                        String environment = applicationUtils.getEnvType().getValue();
                                        addCache(tenantName, namespaceName, environment, configName, EmptyJSONObject.SELF);
                                        return null;
                                    } else {
                                        throw new ServiceException(result.getMsg(), result.getCode());
                                    }
                                }, true)
                        ))
                )
        );
        // 传递新的副本避免外部引用修改只读配置
        return configInfo == null || configInfo instanceof EmptyJSONObject ? null : new JSONObject(configInfo.toString());
    }

    public JSONObject get(String configName, Boolean isThrowExceptionOnConfigNotFound) {
        return get(configName, isThrowExceptionOnConfigNotFound, true);
    }

    /**
     * 获取当前租户命名空间下所有配置
     *
     * @param isThrowExceptionOnConfigNotFound 配置找不到时是否抛出异常
     * @return 配置文件集合，按照[{配置名称:配置内容}]的格式返回
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject getAll(Boolean isThrowExceptionOnConfigNotFound) {
        JSONArray configs = get(NAMESPACE_CONFIGS_KEY, isThrowExceptionOnConfigNotFound, true).getJSONArray("value");
        JSONObject result = new JSONObject();
        configs.forEach(value -> {
            JSONObject valueObj = (JSONObject) value;
            result.put(valueObj.getString("name"), valueObj.getJSONObject("content"));
        });
        return result;
    }

    /**
     * 通过缓存获取配置
     *
     * @param configName  配置名称
     * @param notFoundFun 未找到配置后回调函数
     * @return 结果
     */
    private JSONObject getConfigByCache(String configName, Function<String, JSONObject> notFoundFun) {
        String tenantName = applicationUtils.getTenantName();
        String namespaceName = applicationUtils.getApplicationName();
        String configCacheKey = getConfigCacheKey(tenantName, namespaceName, envType.getValue(), configName);
        // 从本地缓存获取
        JSONObject configContent = LIULI_CONFIG_LOCAL_CACHE.get(configCacheKey);
        if (envType != EnvType.DEV) {
            if (configContent == null) {
                // 从redis获取
                configContent = redisService.get(configCacheKey);
                LOGGER.debug("通过redis获取琉璃配置[{}]：{}", configName, configContent);
                if (configContent != null) {
                    // 缓存到本地
                    LIULI_CONFIG_LOCAL_CACHE.put(configCacheKey, configContent);
                }
            }
        }
        return configContent != null ? configContent : notFoundFun.apply(configCacheKey);
    }

    /**
     * 通过数据库获取配置
     *
     * @param configName  配置名称
     * @param notFoundFun 未找到配置后回调函数
     * @return 结果
     */
    private JSONObject getConfigByDatabase(String configName, Supplier<JSONObject> notFoundFun) {
        if (envType == EnvType.DEV) {
            return notFoundFun.get();
        }
        String tenantName = applicationUtils.getTenantName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = applicationUtils.getEnvType().getValue();
        JSONObject params = new JSONObject();
        params.put("tenantName", tenantName);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("configName", configName);
        try {
            RemoteLogicService remoteLogicService = dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, ServiceNameConstants.SYSTEM_SERVICE);
            R<Object> result = remoteLogicService.run("getLiuLiConfig", params.toString(), SecurityConstants.INNER);
            if (result.getCode() == HttpStatus.SUCCESS) {
                JSONObject response = result.parseResponseJson();
                LOGGER.debug("通过数据库获取琉璃配置[{}]：{}", configName, response);
                String contentStr = response.getJSONObject("data").optString("content", null);
                if (contentStr != null) {
                    JSONObject content = new JSONObject(contentStr);
                    addCache(tenantName, namespaceName, environment, configName, content);
                    return content;
                }
            } else {
                LOGGER.error("服务请求出错，状态码：{}，错误原因：{}", result.getCode(), result.getMsg());
            }
        } catch (Throwable e) {
            LOGGER.error("af-system服务请求失败:{}", e.getMessage());
        }
        return notFoundFun.get();
    }

    /**
     * 通过琉璃中心获取配置
     *
     * @param configName   配置名称
     * @param notFoundFun  未找到配置后回调函数
     * @param isPrecompile 是否预编译
     * @return 结果
     */
    private JSONObject getConfigByRequest(String configName, Boolean shouldGetConfigByParent, Function<R<JSONObject>, JSONObject> notFoundFun, Boolean isPrecompile) {
        String tenantName = applicationUtils.getTenantName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = applicationUtils.getEnvType().getValue();
        boolean hasConfigName = !Objects.equals(configName, NAMESPACE_CONFIGS_KEY);
        //调用琉璃中心需传递的参数
        JSONObject params = new JSONObject();
        params.put("tenantUUID", tenantName);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("shouldGetConfigByParent", shouldGetConfigByParent);
        if (hasConfigName) {
            params.put("configName", configName);
        }

        String requestPath = liuLiUtil.getCloudUrl() + GET_CONFIG_PATH;
        String cloudConfig = RestTools.post(requestPath, params);
        LOGGER.debug("通过琉璃中心获取配置[{}]：{}", configName, cloudConfig);
        R<JSONObject> result = R.build(new JSONObject(cloudConfig));
        if (result.getCode() == HttpStatus.SUCCESS) {
            JSONObject data = result.getData();
            JSONObject content = hasConfigName ? data.getJSONObject("content") : data;
            // 附加全局属性
            String module = data.optString("module", "null");
            String version = data.optString("version", "null");
            JSONObject tenant = data.optJSONObject("tenant");
            JSONObject namespace = data.optJSONObject("namespace");
            content.put("$globalProp", new JSONObject()
                    .put("module", module)
                    .put("environment", environment)
                    .put("version", version)
                    .put("namespaceName", namespace != null ? namespace.getString("f_namespace_name") : "null")
                    .put("tenantAlias", tenant != null ? tenant.getString("f_tenant_uuid") : "null")
            );
            // 附加配置属性
            if(!content.has("$configProp")){
                content.put("$configProp",new JSONObject());
            }
            // 对配置进行预编译
            if (isPrecompile && hasConfigName) {
                LiuLiConfigTypeEnum type = LiuLiConfigTypeEnum.toType(data.getString("type"));
                content = ConfigParser.build(type).parse(content);
            }
            // 增加缓存
            addCache(tenantName, namespaceName, environment, configName, content);
            addDatabase(tenantName, namespaceName, environment, configName, content);
            return content;
        } else {
            return notFoundFun.apply(result);
        }
    }

    /**
     * 通过请求获取未进行预编译的配置
     *
     * @param configName 配置名称
     * @return 原生配置内容
     */
    public JSONObject getNativeConfigByRequest(String configName) {
        return getConfigByRequest(configName, true, (result) -> {
            throw new ServiceException(result.getMsg(), result.getCode());
        }, false);
    }

    /**
     * 配置存入缓存
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @param content       配置内容
     */
    private void addCache(String tenantName, String namespaceName, String environment, String configName, JSONObject content) {
        if (envType == EnvType.DEV) {
            return;
        }
        // 存放配置的key
        String configCacheKey = getConfigCacheKey(tenantName, namespaceName, environment, configName);
        // 缓存到本地
        LIULI_CONFIG_LOCAL_CACHE.put(configCacheKey, content);
        // 缓存到redis
        redisService.set(configCacheKey, content);
    }

    /**
     * 配置存入数据库
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @param content       配置内容
     */
    private void addDatabase(String tenantName, String namespaceName, String environment, String configName, JSONObject content) {
        if (envType == EnvType.DEV) {
            return;
        }
        // 增加配置缓存
        JSONObject params = new JSONObject();
        params.put("tenantName", tenantName);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("configName", configName);
        params.put("configContent", content);
        try {
            RemoteLogicService remoteLogicService = dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, ServiceNameConstants.SYSTEM_SERVICE);
            remoteLogicService.run("addLiuLiConfig", params.toString(), SecurityConstants.INNER);
        } catch (Throwable e) {
            LOGGER.error("af-system服务请求失败:{}", e.getMessage());
        }
    }

    /**
     * 删除配置缓存
     *
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     */
    public void removeCache(String namespaceName, String environment, String configName) {
        // 消息发布
        RTopic clientTopic = redisService.getTopic(LiuLiCacheSubscriber.CLEAR_CACHE_TOPIC_NAME);
        clientTopic.publishAsync(new JSONObject()
                .put("namespaceName", namespaceName)
                .put("environment", environment)
                .put("configName", configName)
        );
    }

    private static class EmptyJSONObject extends JSONObject {
        public static final EmptyJSONObject SELF = new EmptyJSONObject();

        private EmptyJSONObject() {

        }

        @Override
        public String toString() {
            return null;
        }
    }
}
