package com.af.v4.system.common.liuli.config.parser.curd;

import cn.hutool.core.lang.Tuple;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.liuli.config.ConfigParser;
import com.af.v4.system.common.liuli.config.enums.LiuLiConfigTypeEnum;
import com.af.v4.system.common.liuli.config.parser.curd.entity.DataMode;
import com.af.v4.system.common.liuli.config.parser.curd.enums.*;
import com.af.v4.system.common.liuli.config.parser.curd.utils.ConfigUtil;
import com.af.v4.system.common.liuli.config.parser.curd.utils.FormBuildUtil;
import com.af.v4.system.common.liuli.config.parser.curd.utils.SQLParserUtil;
import com.af.v4.system.common.plugins.json.JsonTools;
import com.alibaba.druid.DbType;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;

/**
 * CRUD动态表单配置解析器
 * <p>
 * 详见 /docs/config/queryConfig/查询配置介绍.md
 *
 * @author Mr.river
 */
@Component
public class CRUDFormConfigParser extends ConfigParser {

    private static final Logger LOGGER = LoggerFactory.getLogger(CRUDFormConfigParser.class);

    /**
     * 根据输入字符串生成短哈希表名
     *
     * @param input 输入字符串
     * @return 短哈希表名
     */
    private static String generateShortHashTableName(String input) {
        try {
            // 使用MD5哈希
            MessageDigest md = MessageDigest.getInstance("MD5");
            byte[] messageDigest = md.digest(input.getBytes());
            // 转换为16进制数
            BigInteger no = new BigInteger(1, messageDigest);
            String hashText = no.toString(16);
            // 截取前8位作为表名
            return "tbl" + hashText.substring(0, 8);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public LiuLiConfigTypeEnum getType() {
        return LiuLiConfigTypeEnum.CRUD_FORM;
    }

    /**
     * 根据配置文件生成查询用的SQL片段以及前端动态生成表单和表格的数据集合
     *
     * @param config 配置文件内容
     * @return tableName      主表名
     * tableAliasName 主表别名
     * countSql countSQL片段，不能直接用于查询，需要结合getQuerySQL方法组织出完整SQL
     * querySql querySQL片段，不能直接用于查询，需要结合getQuerySQL方法组织出完整SQL
     * joinTableNameObject 配置文件预设的关联表集合
     * canJoinTableNameArray 查询需要被关联的表集合
     * selectColumn 动态生成表单查询条件的数据列集合
     * buttonState  查询页面按钮状态
     * apiSlot      通用插槽
     * orderBy      查询的排序字段
     * condition    固定的查询表达式
     * columnJson   生成的表格展示列集合
     * formJson     生成的表单项集合
     * chineseQueryColumnArray SQL查询列中文名集合
     * excelImportTemplate excel导入模板
     */
    @Override
    public JSONObject parse(JSONObject config) {
        // 转义被xss过滤的特殊字符
        config = new JSONObject(config.toString()
                .replace("&lt;", "<")
                .replace("&gt;", ">")
        );
        // 组织querySQL
        StringBuilder sql = new StringBuilder("SELECT ");
        // 组织countSQL
        StringBuilder countSql = new StringBuilder("SELECT COUNT(0) n");
        // 查询的主表名称
        String mainTableName = ConfigUtil.parseKey(config.getString("tableName"));
        // 查询的主表别名
        int mainTableNameSpiltIndex = mainTableName.indexOf(' ');
        String mainTableMetaName = mainTableName.substring(0, mainTableNameSpiltIndex);
        String mainTableAliasName = mainTableName.substring(mainTableNameSpiltIndex + 1);
        // 配置文件预设的关联表集合
        JSONObject joinTableNameObject = config.getJSONObject("joinArray");
        for (String key : joinTableNameObject.keySet()) {
            joinTableNameObject.put(key, new JSONObject()
                    .put("value", ConfigUtil.parseKey(joinTableNameObject.getString(key)))
                    .put("type", "join")
            );
        }
        // 查询页面按钮状态
        JSONObject buttonState = config.has("buttonState") ? config.getJSONObject("buttonState") : null;
        // 获取请求通用logic插槽
        JSONObject apiSlot = config.has("apiSlot") ? config.getJSONObject("apiSlot") : null;
        // 配置文件的数据列集合
        JSONArray queryColumnArray = config.getJSONArray("column");
        // SQL查询列中文名集合
        JSONArray chineseQueryColumnArray = new JSONArray(queryColumnArray.length());
        // excel导入模板
        JSONArray excelImportTemplate = new JSONArray();
        // 组织查询需要被关联的表集合
        JSONArray canJoinTableNameArray = new JSONArray();
        // 组织动态生成表单查询条件的数据列集合
        JSONObject selectColumnObject = new JSONObject();
        // 组织生成的表格展示列集合
        JSONArray showColumnArray = new JSONArray(queryColumnArray.length());
        // 组织生成的表单项集合
        JSONArray formItemArray = new JSONArray(queryColumnArray.length());
        // 遍历数据列集合
        for (Object columnItem : queryColumnArray) {
            JSONObject column = (JSONObject) columnItem;
            // 获取该列的数据字段（表别名.字段名）
            String key = column.getString("key");
            String realKey = getRealKey(key);
            // 列别名
            String columnAliasName = null;
            // 获取该列的数据模式
            DataMode dataModeObject;
            if (column.has("dataModeArray")) {
                dataModeObject = DataMode.getDataMode(column.getJSONArray("dataModeArray"));
            } else {
                dataModeObject = DataMode.getDefaultValue();
            }
            // 生成SQL查询项
            if (dataModeObject.sqlQueryItem()) {
                Tuple tuple = buildSQLQueryItem(key, realKey, column, sql, mainTableAliasName, joinTableNameObject,
                        canJoinTableNameArray, chineseQueryColumnArray);
                key = tuple.get(0);
                columnAliasName = tuple.get(1);
            }
            // 生成表单查询条件
            if (dataModeObject.sqlQueryCondition()) {
                buildSqlQueryCondition(column, selectColumnObject, key);
            }
            // 生成表格列
            if (dataModeObject.table()) {
                buildTableColumn(showColumnArray, column, columnAliasName, key, realKey, dataModeObject);
            }
            // 生成表单项
            if (dataModeObject.queryForm() || dataModeObject.addOrEditForm()) {
                buildFormItem(column, key, dataModeObject, selectColumnObject, formItemArray, sql);
            }
            // 生成表格导入项
            if (dataModeObject.excelImportItem()) {
                // 只处理主表的数据项
                String tableAlias = getTableAliasByColumn(key);
                if (tableAlias.equals(mainTableAliasName)) {
                    buildExcelImportItem(mainTableMetaName, column, realKey, excelImportTemplate);
                }
            }
            // 生成组
            if (dataModeObject.group()) {
                buildGroup(dataModeObject, column, key, joinTableNameObject, showColumnArray, chineseQueryColumnArray, formItemArray,
                        selectColumnObject, sql, mainTableAliasName, canJoinTableNameArray);
            }
        }
        String orderBy = buildOrderBy(config, mainTableAliasName, joinTableNameObject, canJoinTableNameArray);
        String conditionStr = buildFixedQueryCondition(config, mainTableAliasName, joinTableNameObject, canJoinTableNameArray);
        // 移除查询列中最后多余的逗号
        sql.deleteCharAt(sql.length() - 1);
        // 追加SQL片段到querySQL中
        sql.append("\n").append("FROM ").append(mainTableName);
        // 追加SQL片段到countSQL中
        countSql.append("\n").append("FROM ").append(mainTableName);
        JSONObject result = new JSONObject();
        result.put("tableName", mainTableName);
        result.put("tableAliasName", mainTableAliasName);
        result.put("countSql", countSql.toString());
        result.put("querySql", sql.toString());
        result.put("joinTableNameObject", joinTableNameObject);
        result.put("canJoinTableNameArray", canJoinTableNameArray);
        result.put("selectColumn", selectColumnObject);
        result.put("orderBy", orderBy);
        result.put("condition", conditionStr);
        result.put("columnJson", showColumnArray);
        result.put("formJson", formItemArray);
        result.put("buttonState", buttonState);
        result.put("apiSlot", apiSlot);
        result.put("chineseQueryColumnArray", chineseQueryColumnArray);
        result.put("excelImportTemplate", excelImportTemplate);
        return result;
    }

    /**
     * 获取该列的数据字段，即实际字段名
     *
     * @param key 数据字段
     * @return 实际字段名
     */
    private String getRealKey(String key) {

        int index = key.indexOf('.');
        if (index == -1) {
            index = 0;
        } else {
            index = index + 1;
        }
        return key.substring(index);
    }

    /**
     * 获取数据列的表别名
     *
     * @param column 数据列
     * @return 表别名
     */
    private String getTableAliasByColumn(String column) {
        int spiltIndex = column.indexOf('.');
        if (spiltIndex == -1) {
            throw new ServiceException("列[" + column + "]没有配置表别名，请检查配置", HttpStatus.BAD_REQUEST);
        }
        return column.substring(0, spiltIndex);
    }

    /**
     * 生成SQL查询项集合
     *
     * @return { 列名, 列别名 }
     */
    private Tuple buildSQLQueryItem(String key, String realKey, JSONObject column, StringBuilder sql, String tableAliasName,
                                    JSONObject joinTableNameObject, JSONArray canJoinTableNameArray, JSONArray chineseQueryColumnArray) {
        String columnAliasName;
        String columnName;
        // 获取这个数据字段对应的表别名
        String tableAlias = getTableAliasByColumn(key);
        String queryKey;
        // 表别名如果是$，代表是函数，格式：$.datediff(u.date,i.date) day
        if (tableAlias.equals("$")) {
            int split = key.lastIndexOf(' ');
            columnAliasName = "$_" + key.substring(split + 1);
            key = key.substring(0, split);
            queryKey = key.substring(2);
        } else {
            columnAliasName = tableAlias + "_" + realKey;
            queryKey = key;
        }
        queryKey = ConfigUtil.parseKey(queryKey);
        if (column.has("default")) {
            String defaultValue = String.valueOf(column.get("default"));
            if (!defaultValue.startsWith("$")) {
                defaultValue = "'" + defaultValue + "'";
            } else {
                defaultValue = defaultValue.substring(1);
            }
            DbType dbType = DynamicDataSource.getDbType();
            columnName = switch (dbType) {
                case sqlserver -> "ISNULL(" + queryKey + "," + defaultValue + ") AS \"" + columnAliasName + "\"";
                case mysql -> "IFNULL(" + queryKey + "," + defaultValue + ") AS \"" + columnAliasName + "\"";
                case oracle -> "NVL(" + queryKey + "," + defaultValue + ") AS \"" + columnAliasName + "\"";
                default -> throw new ServiceException("数据库类型不支持：" + dbType);
            };
        } else {
            columnName = queryKey + " AS \"" + columnAliasName + "\"";
        }
        // 追加数据列到生成的查询SQL片段中
        sql.append("\n\t").append(columnName).append(",");
        // 根据数据列情况追加JOIN表达式
        // useJoinByCount:false 用于查询的子表数据列在生成count语句时不需要join
        SQLParserUtil.putJoinCondition(tableAlias, tableAliasName, joinTableNameObject, canJoinTableNameArray, false);
        // 追加数据列和标题到映射集合中
        if (chineseQueryColumnArray != null) {
            chineseQueryColumnArray.put(column.getString("title"));
        }
        return new Tuple(key, columnAliasName);
    }

    /**
     * 生成SQL查询表达式
     */
    private void buildSqlQueryCondition(JSONObject column, JSONObject selectColumnObject, String key) {
        // 获取该列的查询类型
        QueryTypeEnum queryTypeEnum;
        if (column.has("queryType")) {
            String queryType = column.getString("queryType");
            queryTypeEnum = QueryTypeEnum.toType(queryType);
            if (queryTypeEnum == null) {
                throw new ServiceException("列[" + key + "]无法生成表单项，因为这个表单查询类型不存在：" + queryType, HttpStatus.BAD_REQUEST);
            }
        } else {
            queryTypeEnum = QueryTypeEnum.EQUALS;
        }
        // 组织动态生成表单查询条件的数据列集合
        JSONObject selectColumnItem = new JSONObject();
        selectColumnItem.put("queryType", queryTypeEnum.getValue());
        selectColumnItem.put("key", key);
        selectColumnObject.put(key, selectColumnItem);
    }

    /**
     * 生成表格列
     */
    private void buildTableColumn(JSONArray showColumnArray, JSONObject column, String columnAliasName, String key, String realKey, DataMode dataModeObject) {
        // 组织表格展示列
        JSONObject item = new JSONObject();
        item.put("title", column.getString("title"));
        item.put("dataIndex", columnAliasName == null ? realKey : columnAliasName);
        // 只有生成查询项和查询条件的才可以进行排序
        item.put("sorter", dataModeObject.sqlQueryItem() && dataModeObject.sqlQueryCondition());
        // 指定宽度
        int width = 0;
        JSONObject slot;
        if (!column.has("slot")) {
            // 如果没有插槽，设置一个默认的文本溢出省略(length:16)
            slot = new JSONObject();
            slot.put("type", "ellipsis");
            slot.put("value", 16);
        } else {
            slot = column.getJSONObject("slot");
        }
        JSONObject scopedSlots = new JSONObject();
        scopedSlots.put("customRender", columnAliasName == null ? realKey : columnAliasName);
        item.put("scopedSlots", scopedSlots);
        item.put("slots", scopedSlots);
        if (slot.has("type")) {
            // 插槽类型
            String slotType = slot.getString("type");
            SlotTypeEnum slotTypeEnum = SlotTypeEnum.toType(slotType);
            if (slotTypeEnum == null) {
                throw new ServiceException("列[" + key + "]无法适用插槽，因为这个插槽类型不存在：" + slotType, HttpStatus.BAD_REQUEST);
            }
            item.put("slotType", slotType);
            switch (slotTypeEnum) {
                case DATE_TIME -> width = 160;
                case ELLIPSIS -> {
                    int ellipsisValue = slot.getInt("value");
                    BigDecimal b = new BigDecimal(String.valueOf(ellipsisValue * 7 + 20 + 32));
                    width = b.setScale(0, RoundingMode.UP).intValue();
                    item.put("slotValue", ellipsisValue);
                }
                case BADGE -> {
                    item.put("slotKeyMap", String.valueOf(slot.get("keyMap")));
                    width = 130;
                }
                case ACTION -> {
                    String actionText = slot.optString("actionText", "详情");
                    item.put("slotValue", actionText);
                }
            }
        } else {
            throw new ServiceException("列[" + key + "]无法适用插槽，因为没有指定插槽类型参数", HttpStatus.BAD_REQUEST);
        }
        if (width > 0) {
            item.put("width", width);
        }
        // 如果有固定列
        if (column.has("fixed")) {
            String fixedType = column.getString("fixed");
            ColumnFixedTypeEnum columnFixedTypeEnum = ColumnFixedTypeEnum.toType(fixedType);
            if (columnFixedTypeEnum == null) {
                throw new ServiceException("列[" + key + "]无法适用固定列，因为这个固定列类型不存在：" + fixedType, HttpStatus.BAD_REQUEST);
            }
            item.put("fixed", fixedType);
        }
        showColumnArray.put(item);
    }

    /**
     * 生成表单项
     */
    private void buildFormItem(JSONObject column, String key, DataMode dataModeObject, JSONObject selectColumnObject, JSONArray formItemArray, StringBuilder sql) {
        // 获取该列的表单类型
        FormTypeEnum formTypeEnum = FormBuildUtil.getFormType(column, key);
        // 获取该列的新增/修改场景类型
        AddOrEditTypeEnum addOrEditTypeEnum;
        if (dataModeObject.addOrEditForm()) {
            if (column.has("addOrEdit")) {
                String addOrEditType = column.getString("addOrEdit");
                addOrEditTypeEnum = AddOrEditTypeEnum.toType(addOrEditType);
                if (addOrEditTypeEnum == null) {
                    throw new ServiceException("列[" + key + "]无法生成表单项，因为这个新增/修改场景类型不存在：" + addOrEditType, HttpStatus.BAD_REQUEST);
                }
            } else {
                addOrEditTypeEnum = AddOrEditTypeEnum.ALL;
            }
        } else {
            addOrEditTypeEnum = AddOrEditTypeEnum.NO;
        }
        // 组织表单项
        JSONObject item = new JSONObject();
        item.put("model", key.replaceFirst("\\.", "_"));
        item.put("type", formTypeEnum.getValue());
        item.put("queryType", column.optString("queryType", null));
        item.put("name", column.getString("title"));
        if (formTypeEnum == FormTypeEnum.SELECT ||
                formTypeEnum == FormTypeEnum.CHECKBOX ||
                formTypeEnum == FormTypeEnum.RADIO ||
                formTypeEnum == FormTypeEnum.TREE_SELECT ||
                formTypeEnum == FormTypeEnum.CASCADER
        ) {
            if (column.has("selectKeyName")) {
                // 获取这个数据字段对应的表别名
                String selectKeyName = column.getString("selectKeyName");
                String tableAlias = selectKeyName.substring(0, selectKeyName.indexOf('.'));
                // 获取该列的数据字段（字段名）
                int index = key.indexOf('.');
                if (index == -1) {
                    index = 0;
                } else {
                    index = index + 1;
                }
                String realSelectKey = selectKeyName.substring(index);
                if (!selectKeyName.equals(key)) {
                    // 追加数据列到生成的查询SQL片段中
                    sql.append("\n\t").append(selectKeyName).append(" ").append(tableAlias).append("_").append(realSelectKey).append(",");
                    // 组织动态生成表单查询条件的数据列集合
                    JSONObject selectColumnItem = new JSONObject();
                    selectColumnItem.put("queryType", column.optString("queryType", QueryTypeEnum.EQUALS.getValue()));
                    selectColumnItem.put("key", selectKeyName);
                    selectColumnObject.put(selectKeyName, selectColumnItem);
                    // 修改表单项的model
                    item.put("model", selectKeyName.replace('.', '_'));
                }
            }
            if (column.has("selectKey")) {
                Object selectKey = column.get("selectKey");
                if (selectKey instanceof JSONArray) {
                    item.put("keys", selectKey);
                } else {
                    item.put("keyName", selectKey);
                }
                if (column.has("lazyLoad")) {
                    item.put("lazyLoad", column.get("lazyLoad"));
                }
            }
        } else {
            FormBuildUtil.buildFileRes(column, formTypeEnum, item);
        }
        FormBuildUtil.buildRuleAndPlaceholder(column, key, item);
        item.put("addOrEdit", addOrEditTypeEnum.getValue());
        item.put("isOnlyAddOrEdit", !dataModeObject.queryForm());
        if (addOrEditTypeEnum == AddOrEditTypeEnum.SILENCE_ADD) {
            // 获取该列的字段用途
            SilencePurposeEnum silencePurposeEnum;
            if (column.has("silencePurpose")) {
                String silencePurpose = column.getString("silencePurpose");
                silencePurposeEnum = SilencePurposeEnum.toType(silencePurpose);
                if (silencePurposeEnum == null) {
                    throw new ServiceException("列[" + key + "]无法生成表单项，因为这个静默新增场景指定的字段用途不存在：" + silencePurpose, HttpStatus.BAD_REQUEST);
                } else if (silencePurposeEnum == SilencePurposeEnum.CUSTOMIZE) {
                    // 如果是自定义用途，需要获取Logic名称
                    if (column.has("silenceSource")) {
                        item.put("silenceSource", column.getString("silenceSource"));
                    } else {
                        throw new ServiceException("列[" + key + "]无法生成表单项，因为静默新增场景的自定义字段用途必须指定业务逻辑名称(silenceSource)", HttpStatus.BAD_REQUEST);
                    }
                }
                item.put("silencePurpose", column.getString("silencePurpose"));
            } else {
                throw new ServiceException("列[" + key + "]无法生成表单项，因为静默新增场景必须指定字段用途", HttpStatus.BAD_REQUEST);
            }
        }
        formItemArray.put(item);
    }

    /**
     * 生成字段组
     */
    private void buildGroup(DataMode dataModeObject, JSONObject column, String key, JSONObject joinTableNameObject, JSONArray showColumnArray,
                            JSONArray chineseQueryColumnArray, JSONArray formItemArray, JSONObject selectColumnObject,
                            StringBuilder sql, String tableAliasName, JSONArray canJoinTableNameArray) {
        if (!dataModeObject.sqlQueryItem()) {
            return;
        }
        // 内部表别名
        String innerTableAlias = generateShortHashTableName(key);
        // 源配置列集合
        JSONArray metaJsonArray = column.getJSONArray("column");
        JSONObject jsonTable = new JSONObject();
        jsonTable.put("column", metaJsonArray);
        jsonTable.put("tableName", innerTableAlias + " " + innerTableAlias);
        jsonTable.put("joinArray", new JSONObject());
        // JSON虚拟表加入集合
        List<String> innerKeys = new ArrayList<>(metaJsonArray.length());
        for (Object item : metaJsonArray) {
            JSONObject jsonColumnItem = (JSONObject) item;
            // 重写JSON列里的key
            String innerKey = jsonColumnItem.getString("key");
            innerKey = innerTableAlias + "." + innerKey;
            jsonColumnItem.put("key", innerKey);
            innerKeys.add(innerKey);
        }
        StringBuilder jsonTableContent = new StringBuilder();
        DbType dbType = DynamicDataSource.getDbType();
        switch (dbType) {
            case sqlserver -> {
                jsonTableContent.append("CROSS APPLY OPENJSON(ISNULL(").append(key).append(",'{}')) WITH(");
                innerKeys.forEach(item -> jsonTableContent.append("\n\t\t").append(getRealKey(item)).append(" ").append("varchar(100),"));
                jsonTableContent.deleteCharAt(jsonTableContent.length() - 1);
                jsonTableContent.append("\n\t) ").append(innerTableAlias);
            }
            case oracle -> {
                jsonTableContent.append("CROSS JOIN JSON_TABLE(NVL(").append(key).append(",'{}'), '$' COLUMNS (");
                innerKeys.forEach(item -> {
                    String realKeyValue = getRealKey(item);
                    jsonTableContent.append("\n\t\t").append(realKeyValue).append(" ").append("varchar2(100) PATH '$.").append(realKeyValue).append("',");
                });
                jsonTableContent.deleteCharAt(jsonTableContent.length() - 1);
                jsonTableContent.append("\n\t)) ").append(innerTableAlias);
            }
            case mysql -> {
                jsonTableContent.append("CROSS JOIN JSON_TABLE(IFNULL(").append(key).append(",'{}'), '$' COLUMNS (");
                innerKeys.forEach(item -> {
                    String realKeyValue = getRealKey(item);
                    jsonTableContent.append("\n\t\t").append(realKeyValue).append(" ").append("VARCHAR(100) PATH '$.").append(realKeyValue).append("',");
                });
                jsonTableContent.deleteCharAt(jsonTableContent.length() - 1);
                jsonTableContent.append("\n\t)) ").append(innerTableAlias);
            }
            default -> throw new ServiceException("数据库类型不支持：" + dbType);
        }
        joinTableNameObject.put(innerTableAlias, new JSONObject()
                .put("value", jsonTableContent.toString())
                .put("type", "json")
        );
        // 配置解析
        JSONObject parseJsonColumns = parse(jsonTable);
        JSONArray jsonColumns = parseJsonColumns.getJSONArray("formJson");
        showColumnArray.putAll(parseJsonColumns.getJSONArray("columnJson"));
        chineseQueryColumnArray.putAll(parseJsonColumns.getJSONArray("chineseQueryColumnArray"));
        formItemArray.put(new JSONObject().put("type", "group")
                .put("name", column.getString("title"))
                .put("groupItems", jsonColumns)
                .put("model", key.replaceFirst("\\.", "_")));
        JsonTools.addJSON(selectColumnObject, parseJsonColumns.getJSONObject("selectColumn"));
        // 组织SQL查询项
        for (Object item : metaJsonArray) {
            JSONObject jsonColumnItem = (JSONObject) item;
            String innerKey = jsonColumnItem.getString("key");
            String innerRealKey = getRealKey(innerKey);
            buildSQLQueryItem(innerKey, innerRealKey, jsonColumnItem, sql, tableAliasName, joinTableNameObject,
                    canJoinTableNameArray, null);
        }
    }

    /**
     * 生成orderBy
     */
    private String buildOrderBy(JSONObject config, String tableAliasName, JSONObject joinTableNameObject,
                                JSONArray canJoinTableNameArray) {
        // 取得OrderBy配置字段的表别名，判断是否需要Join
        String orderBy = config.optString("orderBy", null);
        if (orderBy != null) {
            String tableAlias = orderBy.substring(0, orderBy.indexOf('.'));
            // 根据OrderBy的列情况追加JOIN表达式，用于OrderBy的数据列在生成count语句时不需要join
            SQLParserUtil.putJoinCondition(tableAlias, tableAliasName, joinTableNameObject, canJoinTableNameArray, false);
        }
        return orderBy;
    }

    /**
     * 生成固定查询表达式
     */
    private String buildFixedQueryCondition(JSONObject config, String tableAliasName, JSONObject joinTableNameObject,
                                            JSONArray canJoinTableNameArray) {
        // 如果使用了固定查询表达式，则追加该表达式
        String conditionStr;
        if (config.has("condition")) {
            JSONObject condition = config.getJSONObject("condition");
            conditionStr = condition.getString("value");
            if (condition.has("join")) {
                JSONArray joinArray = condition.getJSONArray("join");
                for (Object item : joinArray) {
                    String joinItem = item.toString();
                    // 追加JOIN表达式
                    // 在执行countSQL语句时,用于查询条件的子表数据列需要JOIN子表
                    SQLParserUtil.putJoinCondition(joinItem, tableAliasName, joinTableNameObject, canJoinTableNameArray, true);
                }
            }
        } else {
            conditionStr = "1=1";
        }
        return conditionStr;
    }

    /**
     * 生成excel表格导入项
     */
    private void buildExcelImportItem(String tableName, JSONObject column, String realKey, JSONArray excelImportTemplate) {
        JSONObject excelImportTemplateItem = null;
        for (Object item : excelImportTemplate) {
            JSONObject excelItem = (JSONObject) item;
            if (excelItem.getString("tableName").equals(tableName)) {
                excelImportTemplateItem = excelItem;
                break;
            }
        }
        if (excelImportTemplateItem == null) {
            excelImportTemplateItem = new JSONObject();
        }
        JSONArray titleArray;
        JSONArray columnArray;
        // 追加数据列和标题到映射集合中
        if (excelImportTemplateItem.has("title")) {
            titleArray = excelImportTemplateItem.getJSONArray("title");
            columnArray = excelImportTemplateItem.getJSONArray("column");
        } else {
            titleArray = new JSONArray();
            columnArray = new JSONArray();
            excelImportTemplateItem.put("tableName", tableName);
            excelImportTemplateItem.put("title", titleArray);
            excelImportTemplateItem.put("column", columnArray);
            excelImportTemplate.put(excelImportTemplateItem);
        }
        titleArray.put(column.getString("title"));
        columnArray.put(realKey);
    }
}
