package com.af.v4.system.common.liuli.config.parser.curd.utils;

import cn.hutool.core.date.DateException;
import cn.hutool.core.date.DateUtil;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.liuli.config.parser.curd.enums.JoinTypeEnum;
import com.af.v4.system.common.liuli.config.parser.curd.enums.QueryTypeEnum;
import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import com.alibaba.druid.DbType;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Iterator;
import java.util.Optional;

/**
 * SQL解析工具类
 */
public class SQLParserUtil {

    private static final String DEFAULT_ALL_VALUE = "全部";

    private static final String DEFAULT_NULL_VALUE = "NULL";

    public SQLParserUtil() {
    }

    /**
     * 截取SQL片段
     */
    private static String extractSqlPart(String sql, String startKeyword, String endKeyword) {
        int startIndex = sql.indexOf(startKeyword) + startKeyword.length();
        int endIndex = sql.indexOf(endKeyword, startIndex);
        return sql.substring(startIndex, endIndex).trim();
    }

    public static JSONObject getSingleStyleQuerySQL(JSONObject queryParams) {
        String querySql = queryParams.getString("querySql");
        String countSql = queryParams.getString("countSql");
        JSONObject result = new JSONObject();
        result.put("items", extractSqlPart(querySql, "SELECT", "FROM"));
        result.put("tableName", extractSqlPart(querySql, "FROM", "WHERE"));
        result.put("condition", querySql.substring(querySql.indexOf("WHERE") + "WHERE".length()).trim());
        result.put("countItems", extractSqlPart(countSql, "SELECT", "FROM"));
        result.put("countTableName", extractSqlPart(countSql, "FROM", "WHERE"));
        result.put("countCondition", countSql.substring(countSql.indexOf("WHERE") + "WHERE".length()).trim());
        result.put("querySql", querySql);
        result.put("countSql", countSql);
        return result;
    }

    /**
     * 追加JOIN表达式
     *
     * @param joinTableAlias        关联的表别名
     * @param tableAliasName        主表的别名
     * @param joinTableNameObject   配置文件预设的关联表集合
     * @param canJoinTableNameArray 查询需要被关联的表集合
     * @param useJoinByCount        生成count语句时是否需要join
     */
    public static void putJoinCondition(String joinTableAlias, String tableAliasName, JSONObject joinTableNameObject,
                                        JSONArray canJoinTableNameArray, Boolean useJoinByCount) {
        // 如果数据字段是主表的，或者是'$'，直接跳过
        if (joinTableAlias.equals(tableAliasName) || joinTableAlias.equals("$")) {
            return;
        }
        // 检查配置信息中是否存在对应的joinTableAlias配置
        if (!joinTableNameObject.has(joinTableAlias)) {
            throw new ServiceException("在组织JOIN表达式时，没有找到关于表别名[" + joinTableAlias + "]的JOIN表达式配置信息", HttpStatus.BAD_REQUEST);
        }
        JSONObject joinObj = joinTableNameObject.getJSONObject(joinTableAlias);
        boolean isContains = getCanJoinTableItem(canJoinTableNameArray, joinTableAlias) != null;
        if (isContains) {
            // 如果已存在，则更新JOIN条件，特别是在生成count语句时
            updateJoinConditionForCountSQL(canJoinTableNameArray, joinTableAlias, useJoinByCount, joinObj);
        } else {
            // 如果不存在，则添加新的JOIN条件
            addNewJoinCondition(joinTableAlias, tableAliasName, useJoinByCount, joinObj, canJoinTableNameArray);
        }
    }

    // 添加新的JOIN条件
    private static void addNewJoinCondition(String joinTableAlias, String tableAliasName, Boolean useJoinByCount,
                                            JSONObject joinObj, JSONArray canJoinTableNameArray) {
        String joinType = joinObj.getString("type");
        String joinCondition = joinObj.getString("value");
        JSONObject params = new JSONObject();
        params.put("type", joinType);
        params.put("alias", joinTableAlias);
        params.put("useJoinByCount", useJoinByCount);

        if ("join".equals(joinType)) {
            // 处理JOIN类型的条件，可能涉及递归调用
            handleJoinType(joinTableAlias, tableAliasName, joinObj, canJoinTableNameArray, useJoinByCount, joinCondition, params);
        } else {
            // 非join类型，直接添加
            params.put("value", joinCondition);
        }
        canJoinTableNameArray.put(params);
    }

    // 处理JOIN类型的条件
    private static void handleJoinType(String joinTableAlias, String tableAliasName, JSONObject joinObj, JSONArray canJoinTableNameArray,
                                       Boolean useJoinByCount, String joinCondition, JSONObject params) {
        joinCondition = joinCondition.toLowerCase();
        // 获取应该JOIN的表别名
        String shouldJoinTableAlias = getShouldJoinTableAlias(joinTableAlias, tableAliasName, joinCondition);
        if (shouldJoinTableAlias != null) {
            // 如果存在依赖的表，则递归调用putJoinCondition进行处理
            putJoinCondition(shouldJoinTableAlias, tableAliasName, joinObj, canJoinTableNameArray, useJoinByCount);
        }
        // 设置JOIN条件
        params.put("value", ConfigUtil.parseKey(JoinTypeEnum.LEFT_OUT_JOIN.getValue() + " " + joinCondition));
    }

    // 更新生成count语句时的JOIN条件
    private static void updateJoinConditionForCountSQL(JSONArray canJoinTableNameArray, String joinTableAlias,
                                                       Boolean useJoinByCount, JSONObject joinObj) {
        JSONObject params = getCanJoinTableItem(canJoinTableNameArray, joinTableAlias);
        if (params == null) {
            return;
        }
        // 更新useJoinByCount标志
        params.put("useJoinByCount", useJoinByCount);
        if (!useJoinByCount) {
            return;
        }
        String joinType = joinObj.getString("type");
        if (!"join".equals(joinType)) {
            return;
        }
        // 如果是join类型，还需要进一步处理依赖的表
        String joinCondition = joinObj.getString("value").toLowerCase();
        String shouldJoinTableAlias = getShouldJoinTableAlias(joinTableAlias, joinTableAlias, joinCondition);
        if (shouldJoinTableAlias == null) {
            return;
        }
        params = getCanJoinTableItem(canJoinTableNameArray, shouldJoinTableAlias);
        if (params == null) {
            return;
        }
        // 更新依赖表的useJoinByCount标志
        params.put("useJoinByCount", true);
        // 递归调用，处理依赖表的JOIN条件
        putJoinCondition(shouldJoinTableAlias, joinTableAlias, joinObj, canJoinTableNameArray, true);
    }

    public static JSONObject getCanJoinTableItem(JSONArray canJoinTableNameArray, String alias) {
        for (Object itemObject : canJoinTableNameArray) {
            JSONObject item = (JSONObject) itemObject;
            if (item.getString("alias").equals(alias)) {
                return item;
            }
        }
        return null;
    }

    /**
     * 判断JOIN表达式是否涉及多表关联，返回额外需要关联的表别名。
     *
     * <p>
     * 处理策略：因为不是所有业务都是主表关联子表，还有子表A关联子表B的情况，此时子表B可能还没有加入到需要被关联的表集合中
     * 所以此处需要进行判断，看是否需要处理多表关联的情况
     * 依次拿到JOIN表达式的第一个表别名和第二个表别名，看是不是自身表的别名或者是主表别名，如果不是，则代表有多表关联的情况
     * 如果有多表关联的情况，该方法会返回需要额外关联的表别名，可以再通过putJoinCondition方法追加JOIN表达式
     *
     * @param joinTableAlias 关联的表别名
     * @param tableAliasName 主表的别名
     * @param joinCondition  JOIN表达式
     * @return 需要关联的额外表别名，如果不涉及多表关联则返回null
     */
    private static String getShouldJoinTableAlias(String joinTableAlias, String tableAliasName, String joinCondition) {
        String onCondition = joinCondition.substring(joinCondition.indexOf(" on ") + 4);
        // 第一个表别名
        String firstAlias = onCondition.substring(0, onCondition.indexOf("."));
        // 第二个表别名
        String secondAlias = onCondition.substring(onCondition.lastIndexOf("=") + 2, onCondition.lastIndexOf(".")).trim();
        // 如果两个别名没有一个是主表别名，那么说明要处理多表关联情况
        if (!firstAlias.equals(tableAliasName) && !secondAlias.equals(tableAliasName)) {
            // 如果第一个表别名是传入的表别名，则取第二个表别名
            return firstAlias.equals(joinTableAlias) ? secondAlias : firstAlias;
        }
        return null;
    }

    /**
     * 根据查询类型拼接条件
     *
     * @param condition 构造的查询条件
     * @param queryType 查询类型
     * @param realKey   真实字段名
     * @param value     字段值
     * @param dbType    数据库类型
     */
    private static void appendConditionByType(StringBuilder condition, QueryTypeEnum queryType, String realKey, String value, DbType dbType) {
        // 拼接条件表达式
        switch (queryType) {
            case EQUALS, NO_EQUALS, LESS_THAN, LESS_THAN_EQUALS, GREATER_THAN, GREATER_THAN_EQUALS -> {
                String mark = queryType.getValue();
                if (DEFAULT_NULL_VALUE.equalsIgnoreCase(value)) {
                    condition.append("\n\tAND ").append(realKey).append(" IS " + DEFAULT_NULL_VALUE);
                } else {
                    condition.append("\n\tAND ").append(realKey).append(" ").append(mark).append(" '").append(value).append("'");
                }
            }
            case INNER_LIKE ->
                    condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("%'");
            case LEFT_LIKE -> condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("'");
            case RIGHT_LIKE ->
                    condition.append("\n\tAND ").append(realKey).append(" LIKE '").append(value).append("%'");
            case IN -> {
                if (value.startsWith("[") && value.endsWith("]")) {
                    value = CommonTools.union(JsonTools.parseArray(value));
                }
                condition.append("\n\tAND ").append(realKey).append(" IN (").append(value).append(")");
            }
            case NOT_IN -> {
                if (value.startsWith("[") && value.endsWith("]")) {
                    value = CommonTools.union(JsonTools.parseArray(value));
                }
                condition.append("\n\tAND ").append(realKey).append(" NOT IN (").append(value).append(")");
            }
            case BETWEEN -> {
                JSONArray array = new JSONArray(value);
                String beginValue = array.get(0).toString();
                String endValue = array.get(1).toString();
                try {
                    beginValue = DateUtil.parse(beginValue).toString();
                    endValue = DateUtil.parse(endValue).toString();
                    if (DbType.oracle == dbType) {
                        if (beginValue.length() <= 10) {
                            beginValue = "TO_DATE('" + beginValue + "', 'yyyy-MM-dd')";
                            endValue = "TO_DATE('" + endValue + "', 'yyyy-MM-dd')";
                        } else {
                            beginValue = "TO_DATE('" + beginValue + "', 'yyyy-MM-dd HH24:mi:ss')";
                            endValue = "TO_DATE('" + endValue + "', 'yyyy-MM-dd HH24:mi:ss')";
                        }
                    } else {
                        beginValue = "'" + beginValue + "'";
                        endValue = "'" + endValue + "'";
                    }
                } catch (DateException e) {
                    beginValue = "'" + beginValue + "'";
                    endValue = "'" + endValue + "'";
                }
                condition.append("\n\tAND ").append(realKey).append(" BETWEEN ").append(beginValue).append(" AND ").append(endValue);
            }
            default -> throw new ServiceException("不支持的查询类型：" + queryType.getValue(), HttpStatus.BAD_REQUEST);
        }
    }

    /**
     * 生成完整可执行的querySQL和countSQL语句
     *
     * @param queryParamsMap 配置文件资源
     * @param params         前端传入的表单参数
     * @param sortField      前端传入的排序字段
     * @param sortOrder      前端传入的排序方式
     * @return 完整可执行的querySQL和countSQL语句
     */
    public JSONObject getQuerySQL(JSONObject queryParamsMap, JSONObject params, String sortField, String sortOrder) {
        // 获取主表的别名
        String tableAliasName = queryParamsMap.getString("tableAliasName");
        // 获取querySql和countSql片段
        StringBuilder querySql = new StringBuilder(queryParamsMap.getString("querySql"));
        StringBuilder countSql = new StringBuilder(queryParamsMap.getString("countSql"));
        // 获取关联表配置信息
        JSONObject joinTableNameObject = queryParamsMap.getJSONObject("joinTableNameObject");
        JSONArray canJoinTableNameArray = queryParamsMap.getJSONArray("canJoinTableNameArray");
        // 获取排序条件
        String orderBy = getOrderBy(queryParamsMap, sortField, sortOrder);
        // 获取WHERE条件
        String whereCondition = getWhereCondition(queryParamsMap, params, tableAliasName, joinTableNameObject, canJoinTableNameArray);

        // 向SQL语句中追加JOIN语句
        appendJoinStatements(querySql, countSql, canJoinTableNameArray);
        // 追加WHERE条件
        querySql.append("\nWHERE ").append(whereCondition);
        countSql.append("\nWHERE ").append(whereCondition);
        // 追加排序条件
        querySql.append("\nORDER BY ").append(orderBy);

        // 构建并返回结果
        JSONObject result = new JSONObject();
        result.put("querySql", querySql.toString());
        result.put("countSql", countSql.toString());
        return result;
    }

    private String getOrderBy(JSONObject queryParamsMap, String sortField, String sortOrder) {
        // 若排序字段非空，则处理排序逻辑
        if (sortField != null && !sortField.isEmpty()) {
            String realKey = getRealKey(queryParamsMap, sortField);
            // 根据排序方式选择升序或降序
            return Optional.ofNullable(sortOrder).filter("descend"::equals).map(o -> realKey + " DESC").orElse(realKey);
        }
        // 默认排序
        return queryParamsMap.getString("orderBy");
    }

    private String getRealKey(JSONObject queryParamsMap, String sortField) {
        // 处理排序字段，获取真实的字段名
        if (sortField.indexOf('.') == -1 || sortField.startsWith("$_")) {
            sortField = sortField.replaceFirst("_", ".");
        }
        JSONObject queryParams = queryParamsMap.getJSONObject("selectColumn").getJSONObject(sortField);
        String realKey = queryParams.getString("key");
        // 特殊处理函数字段
        if (realKey.startsWith("$")) {
            realKey = realKey.substring(realKey.indexOf('.') + 1);
        }
        return ConfigUtil.parseKey(realKey);
    }

    private String getWhereCondition(JSONObject queryParamsMap, JSONObject params, String tableAliasName, JSONObject joinTableNameObject, JSONArray canJoinTableNameArray) {
        // 获取固定查询条件
        String conditionStr = queryParamsMap.getString("condition");
        // 结合动态表单参数生成WHERE条件
        return getValue(queryParamsMap.getJSONObject("selectColumn"), params, tableAliasName, joinTableNameObject, canJoinTableNameArray, conditionStr);
    }

    private void appendJoinStatements(StringBuilder querySql, StringBuilder countSql, JSONArray canJoinTableNameArray) {
        // 遍历需要JOIN的表，向SQL语句中追加JOIN条件
        canJoinTableNameArray.forEach(item -> {
            JSONObject joinObject = (JSONObject) item;
            String condition = joinObject.getString("value");
            querySql.append("\n\t").append(condition);
            // 判断是否需要在计数SQL中使用JOIN
            if (joinObject.getBoolean("useJoinByCount")) {
                countSql.append("\n\t").append(condition);
            }
        });
    }

    /**
     * 生成完整可执行的querySQL和countSQL语句，以及查询列，表名，条件表达式的SQL片段
     *
     * @param queryParamsMap 配置文件资源
     * @param params         前端传入的表单参数
     * @param sortField      前端传入的排序字段
     * @param sortOrder      前端传入的排序方式
     * @return 完整可执行的querySQL和countSQL语句，以及查询列，表名，条件表达式的SQL片段
     */
    public JSONObject getSingleStyleQuerySQL(JSONObject queryParamsMap, JSONObject params, String sortField, String sortOrder) {
        JSONObject queryParams = getQuerySQL(queryParamsMap, params, sortField, sortOrder);
        return getSingleStyleQuerySQL(queryParams);
    }

    /**
     * 生成查询条件
     *
     * @param queryMap              动态生成表单查询条件所用的数据列集合
     * @param params                前端传入的表单参数
     * @param tableAliasName        主表的别名
     * @param joinTableNameObject   获取配置文件预设的关联表集合
     * @param canJoinTableNameArray 获取查询时需要被关联的表集合
     * @param conditionStr          固定查询表达式
     * @return 查询条件的SQL表达式
     */
    public String getValue(JSONObject queryMap, JSONObject params, String tableAliasName,
                           JSONObject joinTableNameObject, JSONArray canJoinTableNameArray, String conditionStr) {
        DbType dbType = DynamicDataSource.getDbType();
        // 查询条件
        StringBuilder condition = new StringBuilder(conditionStr);
        // 遍历前端传入的表单参数
        Iterator<String> iterator = params.keys();
        while (iterator.hasNext()) {
            String key = iterator.next();
            String value = String.valueOf(params.get(key));
            // 如果前端传入的表单参数有值
            if (!value.isEmpty() && (!DEFAULT_ALL_VALUE.equals(value))) {
                if (key.indexOf('.') == -1 || key.startsWith("$_")) {
                    key = key.replaceFirst("_", ".");
                }
                if (queryMap.has(key)) {
                    JSONObject queryParams = queryMap.getJSONObject(key);
                    // 获取查询类型
                    String queryTypeStr = queryParams.getString("queryType");
                    if (!QueryTypeEnum.is(queryTypeStr)) {
                        throw new ServiceException("表单字段[" + key + "]无法生成查询条件，因为配置的查询类型不存在：" + queryTypeStr, HttpStatus.BAD_REQUEST);
                    } else {
                        QueryTypeEnum queryType = QueryTypeEnum.toType(queryTypeStr);
                        // 获取该列名称（表别名.字段名）
                        String realKey = queryParams.getString("key");
                        String joinTableAlias = realKey.substring(0, realKey.indexOf('.'));
                        // 表别名如果是$，代表是函数，格式：$.datediff(u.date,i.date) day
                        if (joinTableAlias.equals("$")) {
                            String[] spiltKeyArray = key.split(" ");
                            realKey = spiltKeyArray[0].substring(2);
                        }
                        realKey = ConfigUtil.parseKey(realKey);
                        // 根据查询类型拼接条件
                        appendConditionByType(condition, queryType, realKey, value, dbType);
                        // 追加JOIN表达式，在执行countSQL语句时,用于查询条件的子表数据列需要JOIN子表
                        putJoinCondition(joinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameArray, true);
                    }
                } else {
                    throw new ServiceException("表单字段[" + key + "]无法生成查询条件，因为没有配置这个字段的相关信息", HttpStatus.BAD_REQUEST);
                }
            }
        }
        return condition.toString();
    }
}

