package com.af.v4.system.common.liuli.utils;

import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.enums.CompatibleVersion;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.security.utils.SecurityUtils;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;

/**
 * 应用属性工具
 */
@Component
public final class ApplicationUtils {

    private String DEFAULT_TENANT_NAME = null;
    private EnvType ENV_TYPE = null;
    private String APPLICATION_NAME = null;
    private CompatibleVersion COMPATIBLE_VERSION = null;
    private String APP_VERSION;
    private String SYSTEM_V4_VERSION;

    @PostConstruct
    public void init() {
        setSystemV4Version(ApplicationUtils.class.getPackage().getImplementationVersion());
    }


    /**
     * 获取当前运行环境
     *
     * @return 运行环境枚举
     */
    public EnvType getEnvType() {
        return ENV_TYPE;
    }

    @Value("${spring.profiles.active}")
    public void setEnvType(String envType) {
        ENV_TYPE = EnvType.toType(envType);
        if (ENV_TYPE == null) {
            throw new ServiceException("不支持的运行环境类型[" + envType + "]", HttpStatus.CONFIG_ERROR);
        }
    }

    public String getEnvTypeValue() {
        return ENV_TYPE.getValue();
    }

    /**
     * 获取应用名称
     *
     * @return 应用名称
     */
    public String getApplicationName() {
        return APPLICATION_NAME;
    }

    @Value("${spring.application.name}")
    public void setApplicationName(String applicationName) {
        APPLICATION_NAME = applicationName;
    }

    /**
     * 获取租户名称
     *
     * @return 租户名称
     */
    public String getTenantName() {
        LoginUser loginUser = SecurityUtils.getLoginUser();
        if (loginUser != null) {
            String userTenantName = loginUser.getTenantName();
            if (userTenantName != null) {
                return userTenantName;
            }
        }
        return DEFAULT_TENANT_NAME;
    }

    @Value("${tenant.name}")
    public void setTenantName(String tenantName) {
        this.DEFAULT_TENANT_NAME = tenantName;
    }

    /**
     * 获取应用兼容版本
     *
     * @return 应用兼容版本
     */
    public CompatibleVersion getCompatibleVersion() {
        return COMPATIBLE_VERSION;
    }

    @Value("${tenant.version}")
    public void setCompatibleVersion(String version) {
        this.COMPATIBLE_VERSION = CompatibleVersion.toType(version);
    }

    /**
     * 获取应用版本
     *
     * @return 应用版本
     */
    public String getAppVersion() {
        return APP_VERSION;
    }

    public void setAppVersion(String version) {
        this.APP_VERSION = version;
    }

    /**
     * 获取SystemV4版本
     *
     * @return SystemV4版本
     */
    public String getSystemV4Version() {
        return SYSTEM_V4_VERSION;
    }

    public void setSystemV4Version(String version) {
        this.SYSTEM_V4_VERSION = version;
    }

    /**
     * 获取当前数据源类型
     *
     * @return 数据源类型
     */
    public String getDbType() {
        return DynamicDataSource.getDbType().name();
    }

    /**
     * 获取JSON内容
     *
     * @return JSON内容
     */
    public JSONObject getValues() {
        return new JSONObject()
                .put("tenantName", getTenantName())
                .put("envType", getEnvTypeValue())
                .put("applicationName", getApplicationName())
                .put("compatibleVersion", COMPATIBLE_VERSION.getValue())
                .put("appVersion", getAppVersion())
                .put("dbType", getDbType())
                .put("systemV4Version", getSystemV4Version());
    }
}
