package com.af.v4.system.common.liuli.monitor.service;

import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.liuli.utils.LiuLiUtil;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.plugins.http.RestAsyncTools;
import com.af.v4.system.common.plugins.http.core.response.EmptyAsyncResponse;
import com.af.v4.system.common.plugins.http.core.response.StandardAsyncResponse;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;

/**
 * 琉璃-监控服务
 *
 * @author Mr.river
 */
@Service
public class LiuLiMonitorService {

    private static final Logger LOGGER = LoggerFactory.getLogger(LiuLiMonitorService.class);

    /**
     * 上报应用错误日志
     */
    private static final String UPLOAD_ERROR_LOG_PATH = "/af-liuli/logic/openapi/uploadErrorLog";
    /**
     * 保存生产环境应用信息
     */
    private static final String SAVE_APPLICATION_INFO_PATH = "/af-liuli/logic/openapi/saveApplicationInfo";
    /**
     * 检查SystemV4版本
     */
    private static final String CHECK_V4_VERSION_PATH = "/af-liuli/logic/openapi/checkV4Version";

    private final ApplicationUtils applicationUtils;
    private final LiuLiUtil liuLiUtil;

    public LiuLiMonitorService(ApplicationUtils applicationUtils, LiuLiUtil liuLiUtil) {
        this.applicationUtils = applicationUtils;
        this.liuLiUtil = liuLiUtil;
    }

    @PostConstruct
    public void init() {
        EnvType envType = applicationUtils.getEnvType();
        if (envType == EnvType.DEV) {
            checkV4Version();
        } else if (envType == EnvType.PROD) {
            uploadApplicationInfo();
        }
    }

    /**
     * 上传生产环境应用信息
     */
    public void uploadApplicationInfo() {
        // 自己目前还不能请求自己
        String applicationName = applicationUtils.getApplicationName();
        if (applicationName.equals("af-liuli")) {
            return;
        }
        String requestPath = liuLiUtil.getCloudUrl() + SAVE_APPLICATION_INFO_PATH;
        //调用琉璃中心需传递的参数
        JSONObject jsonObject = new JSONObject()
                .put("tenantUUID", applicationUtils.getTenantName())
                .put("name", applicationName)
                .put("coreVersion", applicationUtils.getSystemV4Version())
                .put("lastStartDate", DateTools.getNow2());
        RestAsyncTools.post(requestPath, jsonObject.toString(), null,
                new StandardAsyncResponse(response -> {
                    R<JSONObject> result = R.build(new JSONObject(response));
                    if (result.isSuccess()) {
                        LOGGER.info("应用信息上传成功");
                    } else {
                        LOGGER.warn("应用信息上传失败:{}", result.getMsg());
                    }
                }, e -> LOGGER.error("上传生产环境应用信息失败:{}", e.getMessage())));
    }

    /**
     * 上报错误日志
     */
    public void uploadLogicErrorLog(JSONObject operLog) {
        EnvType envType = applicationUtils.getEnvType();
        if (envType == EnvType.DEV) {
            return;
        }
        LOGGER.info("推送错误日志到琉璃中心");
        String requestPath = liuLiUtil.getCloudUrl() + UPLOAD_ERROR_LOG_PATH;
        //调用琉璃中心需传递的参数
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("tenantUUID", applicationUtils.getTenantName());
        jsonObject.put("applicationName", applicationUtils.getApplicationName());
        jsonObject.put("environment", applicationUtils.getEnvType().getValue());
        jsonObject.put("method", operLog.opt("f_method"));
        jsonObject.put("requestMethod", operLog.opt("f_request_method"));
        jsonObject.put("operIp", operLog.opt("f_oper_ip"));
        jsonObject.put("operUrl", operLog.opt("f_oper_url"));
        jsonObject.put("operName", operLog.opt("f_oper_name"));
        jsonObject.put("operParam", operLog.opt("f_oper_param"));
        jsonObject.put("jsonResult", operLog.opt("f_json_result"));
        jsonObject.put("errorMsg", operLog.opt("f_error_msg"));
        jsonObject.put("businessTitle", operLog.opt("f_title"));
        jsonObject.put("businessType", operLog.opt("f_business_type"));
        jsonObject.put("operatorType", operLog.opt("f_operator_type"));
        jsonObject.put("costTime", operLog.opt("f_cost_time"));

        RestAsyncTools.post(requestPath, jsonObject.toString(), null, new EmptyAsyncResponse());
    }

    /**
     * 检查SystemV4版本
     */
    public void checkV4Version() {
        String nowVersion = applicationUtils.getSystemV4Version();
        LOGGER.info("\uD83C\uDF89\uD83C\uDF89\uD83C\uDF89欢迎使用SystemV4，版本：V" + nowVersion);
        // 自己目前还不能请求自己
        String applicationName = applicationUtils.getApplicationName();
        if (applicationName.equals("af-liuli")) {
            return;
        }
        String requestPath = liuLiUtil.getCloudUrl() + CHECK_V4_VERSION_PATH;
        RestAsyncTools.post(requestPath, new JSONObject().put("version", nowVersion).toString(), null,
                new StandardAsyncResponse(response -> {
                    R<JSONObject> result = R.build(new JSONObject(response));
                    if (result.isSuccess()) {
                        if (result.getData().getBoolean("hasUpdate")) {
                            LOGGER.warn(result.getData().getString("msg"));
                        }
                    } else {
                        LOGGER.warn("检查SystemV4版本失败:{}", result.getMsg());
                    }
                }, e -> LOGGER.error("检查SystemV4版本失败:{}", e.getMessage())));
    }
}
