package com.af.v4.system.common.liuli.config.service;

import cn.hutool.cache.Cache;
import cn.hutool.cache.CacheUtil;
import com.af.v4.system.api.RemoteLogicService;
import com.af.v4.system.api.factory.DynamicFeignClientFactory;
import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.liuli.config.ConfigParser;
import com.af.v4.system.common.liuli.config.enums.LiuLiConfigTypeEnum;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.liuli.utils.LiuLiUtil;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.redis.RedisService;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Objects;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * 琉璃-配置服务
 *
 * @author Mr.river
 */
@Service
public class LiuLiConfigService {
    /**
     * 获取客户端配置
     */
    private static final String GET_CONFIG_PATH = "/af-liuli/logic/openapi/getConfigByClient";
    // 用于获取命名空间下所有配置的配置名称
    private static final String NAMESPACE_CONFIGS_KEY = "_nameSpaceGroup_";
    /**
     * 本地缓存 TODO 暂时去掉 redis缓存过期监听
     */
    private static final Cache<String, JSONObject> LIULI_CONFIG_LOCAL_CACHE = CacheUtil.newLRUCache(100);
    /**
     * 熔断间隔时间
     */
    private static final Long BLOW_INTERVAL_VALUE = 60 * 10 * 1000L;

    private static final Logger LOGGER = LoggerFactory.getLogger(LiuLiConfigService.class);
    private static EnvType envType = null;
    private final LiuLiUtil liuLiUtil;
    private final ApplicationUtils applicationUtils;
    private final RedisService redisService;
    private final DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory;

    public LiuLiConfigService(LiuLiUtil liuLiUtil, ApplicationUtils applicationUtils, RedisService redisService, DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory) {
        this.liuLiUtil = liuLiUtil;
        this.applicationUtils = applicationUtils;
        this.redisService = redisService;
        this.dynamicFeignClientFactory = dynamicFeignClientFactory;
        envType = applicationUtils.getEnvType();
    }

    /**
     * 获取配置文件
     *
     * @param configName 配置名称
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject get(String configName) {
        return get(configName, false);
    }

    /**
     * 获取当前租户的配置文件
     *
     * @param configName 配置名称
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject getBySelf(String configName) {
        return get(configName, false, false);
    }


    /**
     * 获取当前租户命名空间下所有配置
     *
     * @return 配置文件集合，按照[{配置名称:配置内容}]的格式返回
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject getAll() {
        return getAll(false);
    }

    /**
     * 获取配置文件
     *
     * @param configName                       配置名称
     * @param isThrowExceptionOnConfigNotFound 配置找不到时是否抛出异常
     * @param shouldGetConfigByParent          找不到配置是否寻找上级配置
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject get(String configName, Boolean isThrowExceptionOnConfigNotFound, Boolean shouldGetConfigByParent) {
        return getConfigByCache(configName, (key) ->
                redisService.lock(key, () -> getConfigByCache(configName,
                        (key2) -> getConfigByDatabase(configName,
                                () -> getConfigByRequest(configName, shouldGetConfigByParent, result -> {
                                    if (!isThrowExceptionOnConfigNotFound) {
                                        LOGGER.warn("获取配置失败，相关信息：" + result);
                                        // 加入熔断Map
                                        String tenantName = applicationUtils.getTenantName();
                                        String namespaceName = applicationUtils.getApplicationName();
                                        String environment = applicationUtils.getEnvType().getValue();
                                        String configCacheKey = getConfigCacheKey(tenantName, namespaceName, environment, configName);
                                        Long nextGetTime = System.currentTimeMillis() + BLOW_INTERVAL_VALUE;
                                        redisService.set(getConfigCacheBlowKey(configCacheKey), nextGetTime);
                                        return null;
                                    } else {
                                        throw new ServiceException(result.getMsg(), result.getCode());
                                    }
                                }, true)
                        ))
                )
        );
    }

    public JSONObject get(String configName, Boolean isThrowExceptionOnConfigNotFound) {
        return get(configName, isThrowExceptionOnConfigNotFound, true);
    }

    /**
     * 获取当前租户命名空间下所有配置
     *
     * @param isThrowExceptionOnConfigNotFound 配置找不到时是否抛出异常
     * @return 配置文件集合，按照[{配置名称:配置内容}]的格式返回
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject getAll(Boolean isThrowExceptionOnConfigNotFound) {
        JSONArray configs = get(NAMESPACE_CONFIGS_KEY, isThrowExceptionOnConfigNotFound, true).getJSONArray("value");
        JSONObject result = new JSONObject();
        configs.forEach(value -> {
            JSONObject valueObj = (JSONObject) value;
            result.put(valueObj.getString("name"), valueObj.getJSONObject("content"));
        });
        return result;
    }

    /**
     * 通过缓存获取配置
     *
     * @param configName  配置名称
     * @param notFoundFun 未找到配置后回调函数
     * @return 结果
     */
    private JSONObject getConfigByCache(String configName, Function<String, JSONObject> notFoundFun) {
        String tenantName = applicationUtils.getTenantName();
        String namespaceName = applicationUtils.getApplicationName();
        String configCacheKey = getConfigCacheKey(tenantName, namespaceName, envType.getValue(), configName);
        JSONObject configContent = null;
        if (envType != EnvType.DEV) {
            // 从本地缓存获取
            configContent = LIULI_CONFIG_LOCAL_CACHE.get(configCacheKey);
            if (configContent == null) {
                // 从redis获取
                configContent = redisService.get(configCacheKey);
                // 缓存到本地 TODO 暂时去掉 redis缓存过期监听
                // LIULI_CONFIG_LOCAL_CACHE.put(configCacheKey, configContent);
            }
        }
        return configContent != null ? configContent : notFoundFun.apply(configCacheKey);
    }

    /**
     * 通过数据库获取配置
     *
     * @param configName  配置名称
     * @param notFoundFun 未找到配置后回调函数
     * @return 结果
     */
    private JSONObject getConfigByDatabase(String configName, Supplier<JSONObject> notFoundFun) {
        if (envType == EnvType.DEV) {
            return notFoundFun.get();
        }
        String tenantName = applicationUtils.getTenantName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = applicationUtils.getEnvType().getValue();
        JSONObject params = new JSONObject();
        params.put("tenantName", tenantName);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("configName", configName);
        try {
            RemoteLogicService remoteLogicService = dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, ServiceNameConstants.SYSTEM_SERVICE);
            R<Object> result = remoteLogicService.run("getLiuLiConfig", params.toString(), SecurityConstants.INNER);
            if (result.getCode() == HttpStatus.SUCCESS) {
                String contentStr = result.parseResponseJson().getJSONObject("data").optString("content", null);
                if (contentStr != null) {
                    JSONObject content = new JSONObject(contentStr);
                    addCache(tenantName, namespaceName, environment, configName, content);
                    return content;
                }
            } else {
                LOGGER.error("服务请求出错，状态码：{}，错误原因：{}", result.getCode(), result.getMsg());
            }
        } catch (Throwable e) {
            LOGGER.error("服务请求失败:" + e.getMessage());
        }
        return notFoundFun.get();
    }

    /**
     * 通过琉璃中心获取配置
     *
     * @param configName   配置名称
     * @param notFoundFun  未找到配置后回调函数
     * @param isPrecompile 是否预编译
     * @return 结果
     */
    private JSONObject getConfigByRequest(String configName, Boolean shouldGetConfigByParent, Function<R<JSONObject>, JSONObject> notFoundFun, Boolean isPrecompile) {
        String tenantName = applicationUtils.getTenantName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = applicationUtils.getEnvType().getValue();
        // 判断熔断间隔时间
        String configCacheKey = getConfigCacheKey(tenantName, namespaceName, environment, configName);
        Long nextGetTime = redisService.get(getConfigCacheBlowKey(configCacheKey));
        if (nextGetTime != null && nextGetTime > System.currentTimeMillis()) {
            LOGGER.info("触发熔断，暂停从配置中心获取");
            return null;
        }
        boolean hasConfigName = !Objects.equals(configName, NAMESPACE_CONFIGS_KEY);
        //调用琉璃中心需传递的参数
        JSONObject params = new JSONObject();
        params.put("tenantUUID", tenantName);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("shouldGetConfigByParent", shouldGetConfigByParent);
        if (hasConfigName) {
            params.put("configName", configName);
        }

        String requestPath = liuLiUtil.getCloudUrl() + GET_CONFIG_PATH;
        String cloudConfig = RestTools.post(requestPath, params);
        R<JSONObject> result = R.build(new JSONObject(cloudConfig));
        if (result.getCode() == HttpStatus.SUCCESS) {
            JSONObject data = result.getData();
            JSONObject content = hasConfigName ? data.getJSONObject("content") : data;
            String module = data.optString("module", null);
            String env = data.getString("environment");
            String version = data.get("version").toString();
            String tenantAlias = data.getJSONObject("tenant").getString("f_tenant_uuid");
            String configNamespaceName = data.getJSONObject("namespace").getString("f_namespace_name");
            // 附加全局属性
            content.put("$globalProp", new JSONObject()
                    .put("module", module)
                    .put("environment", env)
                    .put("version", version)
                    .put("namespaceName", configNamespaceName)
                    .put("tenantAlias", tenantAlias)
            );
            if (isPrecompile && hasConfigName) {
                LiuLiConfigTypeEnum type = LiuLiConfigTypeEnum.toType(data.getString("type"));
                // 对配置进行预编译
                content = ConfigParser.build(type).parse(content);
                addCache(tenantName, namespaceName, environment, configName, content);
                addDatabase(tenantName, namespaceName, environment, configName, content);
            }
            return content;
        } else {
            return notFoundFun.apply(result);
        }
    }

    /**
     * 通过请求获取未进行预编译的配置
     *
     * @param configName 配置名称
     * @return 原生配置内容
     */
    public JSONObject getNativeConfigByRequest(String configName) {
        return getConfigByRequest(configName, true, (result) -> {
            throw new ServiceException(result.getMsg(), result.getCode());
        }, false);
    }

    /**
     * 配置存入缓存
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @param content       配置内容
     */
    public void addCache(String tenantName, String namespaceName, String environment, String configName, JSONObject content) {
        if (envType == EnvType.DEV) {
            return;
        }
        // 存放配置的key
        String configCacheKey = getConfigCacheKey(tenantName, namespaceName, environment, configName);
        redisService.set(configCacheKey, content);
        // 缓存到本地 TODO 暂时去掉 redis缓存过期监听
        // LIULI_CONFIG_LOCAL_CACHE.put(configCacheKey, content);
        // 清理熔断
        redisService.delete(getConfigCacheBlowKey(configCacheKey));
    }

    /**
     * 配置存入数据库
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @param content       配置内容
     */
    public void addDatabase(String tenantName, String namespaceName, String environment, String configName, JSONObject content) {
        if (envType == EnvType.DEV) {
            return;
        }
        // 增加配置缓存
        JSONObject params = new JSONObject();
        params.put("tenantName", tenantName);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("configName", configName);
        params.put("configContent", content);
        try {
            RemoteLogicService remoteLogicService = dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, ServiceNameConstants.SYSTEM_SERVICE);
            remoteLogicService.run("addLiuLiConfig", params.toString(), SecurityConstants.INNER);
        } catch (Throwable e) {
            LOGGER.error("服务请求失败:" + e.getMessage());
        }
    }

    /**
     * 删除配置缓存
     *
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     */
    public void removeCache(String namespaceName, String environment, String configName) {
        // 存放配置的key
        String configCacheKey = getConfigCacheKey("*", namespaceName, environment, configName);
        redisService.lock(configCacheKey, () -> {
            // 删除命名空间配置集合key的缓存
            redisService.delete(NAMESPACE_CONFIGS_KEY);
            // 模糊匹配 key 删除
            redisService.deleteList(redisService.getKeys(configCacheKey));
            // 清理熔断
            redisService.deleteList(redisService.getKeys(getConfigCacheBlowKey(configCacheKey)));
        });
        LIULI_CONFIG_LOCAL_CACHE.remove(configCacheKey);
    }

    /**
     * 获取配置项的缓存Key
     *
     * @param tenantName    租户名称
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @return key
     */
    private String getConfigCacheKey(String tenantName, String namespaceName, String environment, String configName) {
        return CacheConstants.CONFIG_DATA_CACHE_KEY + tenantName + "_" + namespaceName + "_" + environment + "_" + configName;
    }

    /**
     * 获取配置项不存在时的熔断标识Key
     *
     * @param configCacheKey 配置项的缓存Key
     * @return key
     */
    private String getConfigCacheBlowKey(String configCacheKey) {
        return configCacheKey + "_BLOW";
    }
}
