package com.af.v4.system.common.liuli.config.service;

import com.af.v4.system.api.RemoteLogicService;
import com.af.v4.system.api.factory.DynamicFeignClientFactory;
import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.ApplicationUtils;
import com.af.v4.system.common.liuli.config.ConfigParser;
import com.af.v4.system.common.liuli.config.enums.LiuLiConfigTypeEnum;
import com.af.v4.system.common.liuli.utils.LiuLiUtil;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.redis.RedisService;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.function.Function;
import java.util.function.Supplier;

/**
 * 琉璃-配置服务
 *
 * @author Mr.river
 */
@Service
public class LiuLiConfigService {
    /**
     * 获取客户端配置
     */
    private static final String GET_CONFIG_PATH = "/liuli/logic/getConfigByClient";

    private static final Logger LOGGER = LoggerFactory.getLogger(LiuLiConfigService.class);
    private static EnvType envType = null;
    private final LiuLiUtil liuLiUtil;
    private final ApplicationUtils applicationUtils;
    private final RedisService redisService;
    private final DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory;

    public LiuLiConfigService(LiuLiUtil liuLiUtil, ApplicationUtils applicationUtils, RedisService redisService, DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory) {
        this.liuLiUtil = liuLiUtil;
        this.applicationUtils = applicationUtils;
        this.redisService = redisService;
        this.dynamicFeignClientFactory = dynamicFeignClientFactory;
        envType = applicationUtils.getEnvType();
    }

    /**
     * 获取配置文件
     *
     * @param configName 配置名称
     * @return 配置文件信息
     */
    public JSONObject get(String configName) {
        return get(configName, false);
    }

    /**
     * 获取配置文件
     *
     * @param configName                       配置名称
     * @param isThrowExceptionOnConfigNotFound 配置找不到时是否抛出异常
     * @return 配置文件信息
     * @apiNote 配置将依次从本地缓存，redis，数据库，琉璃中心获取，并缓存
     */
    public JSONObject get(String configName, Boolean isThrowExceptionOnConfigNotFound) {
        return getConfigByCache(configName, (key) ->
                redisService.lock(key, () -> getConfigByCache(configName,
                        (key2) -> getConfigByDatabase(configName,
                                () -> getConfigByRequest(configName, result -> {
                                    if (!isThrowExceptionOnConfigNotFound) {
                                        LOGGER.warn("获取配置失败，相关信息：" + result);
                                        return null;
                                    } else {
                                        throw new ServiceException(result.getMsg(), result.getCode());
                                    }
                                }, true)
                        ))
                )
        );
    }

    /**
     * 通过缓存获取配置
     *
     * @param configName  配置名称
     * @param notFoundFun 未找到配置后回调函数
     * @return 结果
     */
    private JSONObject getConfigByCache(String configName, Function<String, JSONObject> notFoundFun) {
        String namespaceName = applicationUtils.getApplicationName();
        String configCacheKey = getConfigCacheKey(namespaceName, envType.getValue(), configName);
        JSONObject configContent = null;
        if (envType != EnvType.DEV) {
            // 从redis获取
            configContent = redisService.get(configCacheKey);
        }
        return configContent != null ? configContent : notFoundFun.apply(configCacheKey);
    }

    /**
     * 通过数据库获取配置
     *
     * @param configName  配置名称
     * @param notFoundFun 未找到配置后回调函数
     * @return 结果
     */
    private JSONObject getConfigByDatabase(String configName, Supplier<JSONObject> notFoundFun) {
        if (envType == EnvType.DEV) {
            return notFoundFun.get();
        }
        String namespaceName = applicationUtils.getApplicationName();
        String environment = applicationUtils.getEnvType().getValue();
        JSONObject params = new JSONObject();
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("configName", configName);
        try {
            RemoteLogicService remoteLogicService = dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, ServiceNameConstants.SYSTEM_SERVICE);
            R<Object> result = remoteLogicService.run("getLiuLiConfig", params.toString(), SecurityConstants.INNER);
            if (result.getCode() == HttpStatus.SUCCESS) {
                String contentStr = result.parseResponseJson().getJSONObject("data").optString("content", null);
                if (contentStr != null) {
                    JSONObject content = new JSONObject(contentStr);
                    addCache(namespaceName, environment, configName, content);
                    return content;
                }
            } else {
                LOGGER.error("服务请求出错，状态码：{}，错误原因：{}", result.getCode(), result.getMsg());
            }
        } catch (Throwable e) {
            LOGGER.error("服务请求失败:" + e.getMessage());
        }
        return notFoundFun.get();
    }

    /**
     * 通过琉璃中心获取配置
     *
     * @param configName   配置名称
     * @param notFoundFun  未找到配置后回调函数
     * @param isPrecompile 是否预编译
     * @return 结果
     */
    private JSONObject getConfigByRequest(String configName, Function<R<JSONObject>, JSONObject> notFoundFun, Boolean isPrecompile) {
        String tenantUUID = applicationUtils.getTenantName();
        String namespaceName = applicationUtils.getApplicationName();
        String environment = applicationUtils.getEnvType().getValue();
        //调用琉璃中心需传递的参数
        JSONObject params = new JSONObject();
        params.put("tenantUUID", tenantUUID);
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("configName", configName);

        String requestPath = liuLiUtil.getCloudUrl() + GET_CONFIG_PATH;
        String cloudConfig = RestTools.post(requestPath, params);
        R<JSONObject> result = R.build(new JSONObject(cloudConfig));
        if (result.getCode() == HttpStatus.SUCCESS) {
            JSONObject data = result.getData();
            JSONObject content = data.getJSONObject("content");
            if (isPrecompile) {
                LiuLiConfigTypeEnum type = LiuLiConfigTypeEnum.toType(data.getString("type"));
                // 对配置进行预编译
                content = ConfigParser.build(type).parse(content);
                addCache(namespaceName, environment, configName, content);
                addDatabase(namespaceName, environment, configName, content);
            }
            return content;
        } else {
            return notFoundFun.apply(result);
        }
    }

    /**
     * 通过请求获取未进行预编译的配置
     *
     * @param configName 配置名称
     * @return 原生配置内容
     */
    public JSONObject getNativeConfigByRequest(String configName) {
        return getConfigByRequest(configName, (result) -> {
            throw new ServiceException(result.getMsg(), result.getCode());
        }, false);
    }

    /**
     * 配置存入缓存
     *
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @param content       配置内容
     */
    public void addCache(String namespaceName, String environment, String configName, JSONObject content) {
        // 存放配置的key
        String configCacheKey = getConfigCacheKey(namespaceName, environment, configName);
        if (envType != EnvType.DEV) {
            redisService.set(configCacheKey, content);
        }
    }

    /**
     * 配置存入数据库
     *
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @param content       配置内容
     */
    public void addDatabase(String namespaceName, String environment, String configName, JSONObject content) {
        if (envType == EnvType.DEV) {
            return;
        }
        // 增加配置缓存
        JSONObject params = new JSONObject();
        params.put("namespaceName", namespaceName);
        params.put("environment", environment);
        params.put("configName", configName);
        params.put("configContent", content);
        try {
            RemoteLogicService remoteLogicService = dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, ServiceNameConstants.SYSTEM_SERVICE);
            remoteLogicService.run("addLiuLiConfig", params.toString(), SecurityConstants.INNER);
        } catch (Throwable e) {
            LOGGER.error("服务请求失败:" + e.getMessage());
        }
    }

    /**
     * 删除配置缓存
     *
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     */
    public void removeCache(String namespaceName, String environment, String configName) {
        //存放配置的key
        String configCacheKey = getConfigCacheKey(namespaceName, environment, configName);
        redisService.lock(configCacheKey, () -> {
            redisService.delete(configCacheKey);
        });
    }

    /**
     * 获取配置项的缓存Key
     *
     * @param namespaceName 命名空间名称
     * @param environment   环境名称
     * @param configName    配置名称
     * @return key
     */
    private String getConfigCacheKey(String namespaceName, String environment, String configName) {
        return CacheConstants.CONFIG_DATA_CACHE_KEY + namespaceName + "_" + environment + "_" + configName;
    }
}
