package com.af.v4.system.common.liuli.config.parser.curd.utils;

import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.liuli.config.parser.curd.enums.JoinTypeEnum;
import com.af.v4.system.common.liuli.config.parser.curd.enums.QueryTypeEnum;
import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

/**
 * SQL解析工具类
 */
public class SQLParserUtil {

    private static final String DEFAULT_ALL_VALUE = "全部";

    private static final String DEFAULT_NULL_VALUE = "NULL";

    public SQLParserUtil() {
    }

    public static JSONObject getSingleStyleQuerySQL(JSONObject queryParams) {
        String querySql = queryParams.getString("querySql");
        String countSql = queryParams.getString("countSql");
        JSONObject result = new JSONObject();
        result.put("items", querySql.substring(querySql.indexOf("SELECT") + 6, querySql.indexOf("FROM")));
        result.put("tableName", querySql.substring(querySql.indexOf("FROM") + 4, querySql.indexOf("WHERE")));
        result.put("condition", querySql.substring(querySql.indexOf("WHERE") + 5));
        result.put("countItems", countSql.substring(countSql.indexOf("SELECT") + 6, countSql.indexOf("FROM")));
        result.put("countTableName", countSql.substring(countSql.indexOf("FROM") + 4, countSql.indexOf("WHERE")));
        result.put("countCondition", countSql.substring(countSql.indexOf("WHERE") + 5));
        result.put("querySql", querySql);
        result.put("countSql", countSql);
        return result;
    }

    /**
     * 追加JOIN表达式
     *
     * @param joinTableAlias        关联的表别名
     * @param tableAliasName        主表的别名
     * @param joinTableNameObject   配置文件预设的关联表集合
     * @param canJoinTableNameArray 查询需要被关联的表集合
     * @param useJoinByCount        生成count语句时是否需要join
     */
    public static void putJoinCondition(String joinTableAlias, String tableAliasName, JSONObject joinTableNameObject,
                                        JSONArray canJoinTableNameArray, Boolean useJoinByCount) {
        // 如果数据字段是主表的，或者是'$'，直接跳过
        if (joinTableAlias.equals(tableAliasName) || joinTableAlias.equals("$")) {
            return;
        }
        // 如果数据字段对应的表别名在配置文件预设的关联表集合中
        if (joinTableNameObject.has(joinTableAlias)) {
            JSONObject params;
            // 如果该表别名不在查询需要被关联的表集合中
            boolean isContains = getCanJoinTableItem(canJoinTableNameArray, joinTableAlias) != null;
            if (!isContains) {
                JSONObject joinObj = joinTableNameObject.getJSONObject(joinTableAlias);
                String joinType = joinObj.getString("type");
                // 取到JOIN语句内容
                String joinCondition = joinObj.getString("value");
                params = new JSONObject();
                params.put("type", joinType);
                params.put("alias", joinTableAlias);
                params.put("useJoinByCount", useJoinByCount);
                if (joinType.equals("join")) {
                    joinCondition = joinCondition.toLowerCase();
                    // 判断该JOIN表达式是否需要处理多表关联的情况
                    String shouldJoinTableAlias = getShouldJoinTableAlias(joinTableAlias, tableAliasName, joinCondition);
                    // 如果返回了需要额外关联的表别名
                    if (shouldJoinTableAlias != null) {
                        // 递归处理JOIN表达式
                        putJoinCondition(shouldJoinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameArray, useJoinByCount);
                    }
                    // 默认使用LEFT JOIN，加入到需要被关联的表集合中
                    params.put("value", ConfigUtil.parseKey(JoinTypeEnum.LEFT_OUT_JOIN.getValue() + " " + joinCondition));
                } else {
                    params.put("value", joinCondition);
                }
                canJoinTableNameArray.put(params);
            } else {
                // 更新这个JOIN表达式在生成count语句时的JOIN策略
                params = getCanJoinTableItem(canJoinTableNameArray, joinTableAlias);
                assert params != null;
                params.put("useJoinByCount", useJoinByCount);
                // 如果这个JOIN表达式在生成count语句时需要JOIN，则需要判断是否有多表关联的情况
                if (useJoinByCount) {
                    JSONObject joinObj = joinTableNameObject.getJSONObject(joinTableAlias);
                    String joinType = joinObj.getString("type");
                    // 取到JOIN语句内容
                    String joinCondition = joinObj.getString("value");
                    if (joinType.equals("join")) {
                        joinCondition = joinCondition.toLowerCase();
                        // 判断该JOIN表达式是否需要处理多表关联的情况
                        String shouldJoinTableAlias = getShouldJoinTableAlias(joinTableAlias, tableAliasName, joinCondition);
                        // 如果返回了需要额外关联的表别名
                        if (shouldJoinTableAlias != null) {
                            // 则更新 这个表别名对应的JOIN表达式在生成count语句时的JOIN策略
                            params = getCanJoinTableItem(canJoinTableNameArray, shouldJoinTableAlias);
                            assert params != null;
                            params.put("useJoinByCount", true);
                            // 递归处理JOIN表达式
                            putJoinCondition(shouldJoinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameArray, true);
                        }
                    }
                }
            }
        } else {
            throw new ServiceException("在组织JOIN表达式时，没有找到关于表别名[" + joinTableAlias + "]的JOIN表达式配置信息", HttpStatus.BAD_REQUEST);
        }
    }

    public static JSONObject getCanJoinTableItem(JSONArray canJoinTableNameArray, String alias) {
        for (Object itemObject : canJoinTableNameArray) {
            JSONObject item = (JSONObject) itemObject;
            if (item.getString("alias").equals(alias)) {
                return item;
            }
        }
        return null;
    }

    /**
     * 判断该JOIN表达式是否需要处理多表关联的情况
     * <p>
     * 处理策略：因为不是所有业务都是主表关联子表，还有子表A关联子表B的情况，此时子表B可能还没有加入到需要被关联的表集合中
     * 所以此处需要进行判断，看是否需要处理多表关联的情况
     * 依次拿到JOIN表达式的第一个表别名和第二个表别名，看是不是自身表的别名或者是主表别名，如果不是，则代表有多表关联的情况
     * 如果有多表关联的情况，该方法会返回需要额外关联的表别名，可以再通过putJoinCondition方法追加JOIN表达式
     *
     * @param joinTableAlias 关联的表别名
     * @param tableAliasName 主表的别名
     * @param joinCondition  JOIN表达式
     * @return 需要关联的表别名
     */
    private static String getShouldJoinTableAlias(String joinTableAlias, String tableAliasName, String joinCondition) {
        String onCondition = joinCondition.substring(joinCondition.indexOf(" on ") + 4);
        // 第一个表别名
        String canJoinTableAliasA = onCondition.substring(0, onCondition.indexOf("."));
        // 第二个表别名
        String canJoinTableAliasB = onCondition.substring(onCondition.lastIndexOf("=") + 1, onCondition.lastIndexOf(".")).trim();
        // 如果两个别名没有一个是主表别名，那么说明要处理多表关联情况
        if (!canJoinTableAliasA.equals(tableAliasName) && !canJoinTableAliasB.equals(tableAliasName)) {
            String shouldJoinTableAlias;
            // 如果别名A是传入的表别名，则取别名B
            if (canJoinTableAliasA.equals(joinTableAlias)) {
                shouldJoinTableAlias = canJoinTableAliasB;
            } else {
                shouldJoinTableAlias = canJoinTableAliasA;
            }
            return shouldJoinTableAlias;
        } else {
            return null;
        }
    }


    /**
     * 生成完整可执行的querySQL和countSQL语句
     *
     * @param queryParamsMap 配置文件资源
     * @param params         前端传入的表单参数
     * @param sortField      前端传入的排序字段
     * @param sortOrder      前端传入的排序方式
     * @return 完整可执行的querySQL和countSQL语句
     */
    public JSONObject getQuerySQL(JSONObject queryParamsMap, JSONObject params, String sortField, String sortOrder) {
        // 获取主表的别名
        String tableAliasName = queryParamsMap.getString("tableAliasName");
        // 获取querySql和countSql片段
        StringBuilder querySql = new StringBuilder(queryParamsMap.getString("querySql"));
        StringBuilder countSql = new StringBuilder(queryParamsMap.getString("countSql"));
        // 获取配置文件预设的关联表集合
        JSONObject joinTableNameObject = queryParamsMap.getJSONObject("joinTableNameObject");
        // 获取查询时需要被关联的表集合
        JSONArray canJoinTableNameArray = queryParamsMap.getJSONArray("canJoinTableNameArray");
        // 获取查询的排序字段
        String orderBy;
        if (sortField != null) {
            if (sortField.indexOf('.') == -1 || sortField.startsWith("$_")) {
                sortField = sortField.replaceFirst("_", ".");
            }
            // 获取该列名称（表别名.字段名）
            JSONObject queryParams = queryParamsMap.getJSONObject("selectColumn").getJSONObject(sortField);
            String realKey = queryParams.getString("key");
            String joinTableAlias = realKey.substring(0, realKey.indexOf('.'));
            // 表别名如果是$，代表是函数，格式：$.datediff(u.date,i.date) day
            if (joinTableAlias.equals("$")) {
                String[] spiltKeyArray = sortField.split(" ");
                realKey = spiltKeyArray[0].substring(2);
            }
            realKey = ConfigUtil.parseKey(realKey);
            if (Optional.ofNullable(sortOrder).isPresent() && "descend".equals(sortOrder)) {
                orderBy = realKey + " DESC";
            } else {
                orderBy = realKey;
            }
        } else {
            orderBy = queryParamsMap.getString("orderBy");
        }
        // 获取固定查询表达式
        String conditionStr = queryParamsMap.getString("condition");
        // 根据前端传入的表单参数，结合动态生成表单查询条件所用的数据列集合，生成where表达式
        String whereCondition = getValue(queryParamsMap.getJSONObject("selectColumn"),
                params, tableAliasName, joinTableNameObject, canJoinTableNameArray, conditionStr);
        // 取出需要JOIN的子表集合
        Consumer<Object> consumer = item -> {
            Map<String, Object> joinObject = (Map<String, Object>) item;
            String condition = joinObject.get("value").toString();
            // 追加需要JOIN的表达式到SQL中
            querySql.append("\n\t").append(condition);
            // 根据需要在countSQL中追加join表达式
            if (Boolean.parseBoolean(joinObject.get("useJoinByCount").toString())) {
                countSql.append("\n\t").append(condition);
            }
        };
        canJoinTableNameArray.toList().stream().filter(item -> ((Map<String, Object>) item).get("type").equals("json"))
                .forEach(consumer);
        canJoinTableNameArray.toList().stream().filter(item -> ((Map<String, Object>) item).get("type").equals("join"))
                .forEach(consumer);
        // 追加where表达式到SQL中
        querySql.append("\nWHERE ").append(whereCondition);
        countSql.append("\nWHERE ").append(whereCondition);
        // 追加orderBy表达式到SQL中
        querySql.append("\nORDER BY ").append(orderBy);
        JSONObject result = new JSONObject();
        result.put("querySql", querySql.toString());
        result.put("countSql", countSql.toString());
        return result;
    }

    /**
     * 生成完整可执行的querySQL和countSQL语句，以及查询列，表名，条件表达式的SQL片段
     *
     * @param queryParamsMap 配置文件资源
     * @param params         前端传入的表单参数
     * @param sortField      前端传入的排序字段
     * @param sortOrder      前端传入的排序方式
     * @return 完整可执行的querySQL和countSQL语句，以及查询列，表名，条件表达式的SQL片段
     */
    public JSONObject getSingleStyleQuerySQL(JSONObject queryParamsMap, JSONObject params, String sortField, String sortOrder) {
        JSONObject queryParams = getQuerySQL(queryParamsMap, params, sortField, sortOrder);
        return getSingleStyleQuerySQL(queryParams);
    }

    /**
     * 生成查询条件
     *
     * @param queryMap              动态生成表单查询条件所用的数据列集合
     * @param params                前端传入的表单参数
     * @param tableAliasName        主表的别名
     * @param joinTableNameObject   获取配置文件预设的关联表集合
     * @param canJoinTableNameArray 获取查询时需要被关联的表集合
     * @param conditionStr          固定查询表达式
     * @return 查询条件的SQL表达式
     */
    public String getValue(JSONObject queryMap, JSONObject params, String tableAliasName,
                           JSONObject joinTableNameObject, JSONArray canJoinTableNameArray, String conditionStr) {
        // 查询条件
        StringBuilder condition = new StringBuilder(conditionStr);
        // 遍历前端传入的表单参数
        Iterator<String> iterator = params.keys();
        while (iterator.hasNext()) {
            String key = iterator.next();
            String value = String.valueOf(params.get(key));
            // 如果前端传入的表单参数有值
            if (!value.isEmpty() && (!DEFAULT_ALL_VALUE.equals(value))) {
                if (key.indexOf('.') == -1 || key.startsWith("$_")) {
                    key = key.replaceFirst("_", ".");
                }
                if (queryMap.has(key)) {
                    JSONObject queryParams = queryMap.getJSONObject(key);
                    // 获取查询类型
                    String queryTypeStr = queryParams.getString("queryType");
                    if (!QueryTypeEnum.is(queryTypeStr)) {
                        throw new ServiceException("表单字段[" + key + "]无法生成查询条件，因为配置的查询类型不存在：" + queryTypeStr, HttpStatus.BAD_REQUEST);
                    } else {
                        QueryTypeEnum queryType = QueryTypeEnum.toType(queryTypeStr);
                        // 获取该列名称（表别名.字段名）
                        String realKey = queryParams.getString("key");
                        String joinTableAlias = realKey.substring(0, realKey.indexOf('.'));
                        // 表别名如果是$，代表是函数，格式：$.datediff(u.date,i.date) day
                        if (joinTableAlias.equals("$")) {
                            String[] spiltKeyArray = key.split(" ");
                            realKey = spiltKeyArray[0].substring(2);
                        }
                        realKey = ConfigUtil.parseKey(realKey);
                        // 拼接条件表达式
                        switch (queryType) {
                            case EQUALS, NO_EQUALS, LESS_THAN, LESS_THAN_EQUALS, GREATER_THAN, GREATER_THAN_EQUALS -> {
                                String mark = queryType.getValue();
                                if (DEFAULT_NULL_VALUE.equalsIgnoreCase(value)) {
                                    condition.append("\n\tAND ").append(realKey).append(" IS " + DEFAULT_NULL_VALUE);
                                } else {
                                    condition.append("\n\tAND ").append(realKey).append(" ").append(mark).append(" '").append(value).append("'");
                                }
                            }
                            case INNER_LIKE ->
                                    condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("%'");
                            case LEFT_LIKE ->
                                    condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("'");
                            case RIGHT_LIKE ->
                                    condition.append("\n\tAND ").append(realKey).append(" LIKE '").append(value).append("%'");
                            case IN -> {
                                if (value.startsWith("[") && value.endsWith("]")) {
                                    value = CommonTools.union(JsonTools.parseArray(value));
                                }
                                condition.append("\n\tAND ").append(realKey).append(" IN (").append(value).append(")");
                            }
                            case NOT_IN -> {
                                if (value.startsWith("[") && value.endsWith("]")) {
                                    value = CommonTools.union(JsonTools.parseArray(value));
                                }
                                condition.append("\n\tAND ").append(realKey).append(" NOT IN (").append(value).append(")");
                            }
                            case BETWEEN -> {
                                JSONArray array = new JSONArray(value);
                                condition.append("\n\tAND ").append(realKey).append(" BETWEEN '").append(array.get(0)).append("' AND '").append(array.get(1)).append("'");
                            }
                            default ->
                                    throw new ServiceException("不支持的查询类型：" + queryType.getValue(), HttpStatus.BAD_REQUEST);
                        }
                        // 追加JOIN表达式
                        // 在执行countSQL语句时,用于查询条件的子表数据列需要JOIN子表
                        // 获取这个数据字段对应的表别名
                        putJoinCondition(joinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameArray, true);
                    }
                } else {
                    throw new ServiceException("表单字段[" + key + "]无法生成查询条件，因为没有配置这个字段的相关信息", HttpStatus.BAD_REQUEST);
                }
            }
        }
        return condition.toString();
    }
}

