package com.af.v4.system.common.liuli.utils;

import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.plugins.http.RestAsyncTools;
import com.af.v4.system.common.plugins.http.core.response.StandardAsyncResponse;
import com.af.v4.system.common.redis.RedisService;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

/**
 * 琉璃工具类
 *
 * @author Mr.river
 */
@Component
public class LiuLiUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(LiuLiUtil.class);

    /**
     * 琉璃服务端地址-RedisKey
     */
    private static final String CLOUD_URL_KEY = CacheConstants.CONFIG_DATA_CACHE_KEY + "CloudUrl";
    /**
     * 默认琉璃端服务地址
     */
    private static final String DEFAULT_CLOUD_URL = "http://aote-office.8866.org:31039";
    /**
     * 检查SystemV4版本
     */
    private static final String CHECK_V4_VERSION_PATH = "/liuli/logic/checkV4Version";

    private final RedisService redisService;

    public LiuLiUtil(RedisService redisService) {
        this.redisService = redisService;
    }

    /**
     * 检查SystemV4版本
     *
     * @param version 当前版本
     */
    public void checkV4Version(String version) {
        String requestPath = getCloudUrl() + CHECK_V4_VERSION_PATH;
        RestAsyncTools.post(requestPath, new JSONObject().put("version", version).toString(), null,
                new StandardAsyncResponse(response -> {
                    R<JSONObject> result = R.build(new JSONObject(response));
                    if (result.isSuccess()) {
                        if (result.getData().getBoolean("hasUpdate")) {
                            LOGGER.warn(result.getData().getString("msg"));
                        }
                    } else {
                        LOGGER.warn("检查SystemV4版本失败:{}", result.getMsg());
                    }
                }, e -> LOGGER.error("检查SystemV4版本失败:{}", e.getMessage())));
    }

    /**
     * 获取琉璃服务端地址
     * 返回值：琉璃配置中心路径
     */
    public String getCloudUrl() {
        String cloudUrl;
        JSONObject url = redisService.get(CLOUD_URL_KEY);
        if (url == null) {
            cloudUrl = DEFAULT_CLOUD_URL;
            JSONObject jsonObject = new JSONObject();
            //存入redis
            jsonObject.put("cloudUrl", cloudUrl);
            redisService.set(CLOUD_URL_KEY, jsonObject);
        } else {
            cloudUrl = url.getString("cloudUrl");
        }
        return cloudUrl;
    }

    /**
     * 更新琉璃服务端地址
     *
     * @param cloudUrl 接口地址
     */
    public void updateCloudUrl(String cloudUrl) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("cloudUrl", cloudUrl);
        redisService.lock(CLOUD_URL_KEY, () -> redisService.set(CLOUD_URL_KEY, jsonObject));
    }
}
