package com.af.v4.system.common.liuli.config.parser;

import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.liuli.config.ConfigParser;
import com.af.v4.system.common.liuli.config.enums.LiuLiConfigTypeEnum;
import com.af.v4.system.common.liuli.config.parser.query.DataMode;
import com.af.v4.system.common.liuli.config.parser.query.enums.*;
import com.af.v4.system.common.liuli.config.parser.query.utils.ConfigUtil;
import com.af.v4.system.common.liuli.config.parser.query.utils.FormBuildUtil;
import com.af.v4.system.common.liuli.config.parser.query.utils.SQLParserUtil;
import com.alibaba.druid.DbType;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * CRUD动态表单配置解析器
 * <p>
 * 详见 /docs/config/queryConfig/查询配置介绍.md
 *
 * @author Mr.river
 */
@Component
public class CRUDFormConfigParser extends ConfigParser {

    private static final Logger LOGGER = LoggerFactory.getLogger(CRUDFormConfigParser.class);

    @Override
    public LiuLiConfigTypeEnum getType() {
        return LiuLiConfigTypeEnum.CRUD_FORM;
    }

    /**
     * 根据配置文件生成查询用的SQL片段以及前端动态生成表单和表格的数据集合
     *
     * @param config 配置文件内容
     * @return tableName      主表名
     * tableAliasName 主表别名
     * countSql countSQL片段，不能直接用于查询，需要结合getQuerySQL方法组织出完整SQL
     * querySql querySQL片段，不能直接用于查询，需要结合getQuerySQL方法组织出完整SQL
     * joinTableNameObject 配置文件预设的关联表集合
     * canJoinTableNameObject 查询需要被关联的表集合
     * selectColumn 动态生成表单查询条件的数据列集合
     * buttonState  查询页面按钮状态
     * apiSlot      通用插槽
     * orderBy      查询的排序字段
     * condition    固定的查询表达式
     * columnJson   生成的表格展示列集合
     * formJson     生成的表单项集合
     */
    @Override
    public JSONObject parse(JSONObject config) {
        // 组织querySQL
        StringBuilder sql = new StringBuilder("SELECT ");
        // 组织countSQL
        StringBuilder countSql = new StringBuilder("SELECT COUNT(0) n");
        // 调用服务名称
        String serviceName = config.optString("serviceName", ServiceNameConstants.SYSTEM_SERVICE);
        // 查询的主表名称
        String tableName = ConfigUtil.parseKey(config.getString("tableName"));
        // 查询的主表别名
        String tableAliasName = tableName.substring(tableName.indexOf(' ') + 1);
        // 配置文件预设的关联表集合
        JSONObject joinTableNameObject = config.getJSONObject("joinArray");
        for (String key : joinTableNameObject.keySet()) {
            joinTableNameObject.put(key, ConfigUtil.parseKey(joinTableNameObject.getString(key)));
        }
        // 查询页面按钮状态
        JSONObject buttonState = config.has("buttonState") ? config.getJSONObject("buttonState") : null;
        // 获取请求通用logic插槽
        JSONObject apiSlot = config.has("apiSlot") ? config.getJSONObject("apiSlot") : null;
        // 配置文件的数据列集合
        JSONArray queryColumnArray = config.getJSONArray("column");
        // 数据列中文列集合
        JSONArray chineseQueryColumnArray = new JSONArray(queryColumnArray.length());
        // 组织查询需要被关联的表集合
        JSONArray canJoinTableNameObject = new JSONArray();
        // 组织动态生成表单查询条件的数据列集合
        JSONObject selectColumnArray = new JSONObject();
        // 组织生成的表格展示列集合
        JSONArray showColumnArray = new JSONArray(queryColumnArray.length());
        // 组织生成的表单项集合
        JSONArray formItemArray = new JSONArray(queryColumnArray.length());
        // 遍历数据列集合
        for (Object columnItem : queryColumnArray) {
            JSONObject column = (JSONObject) columnItem;
            // 获取该列的数据字段（表别名.字段名）
            String key = column.getString("key");
            // 获取该列的数据字段（字段名）
            int index = key.indexOf('.');
            if (index == -1) {
                index = 0;
            } else {
                index = index + 1;
            }
            String realKey = key.substring(index);
            // 列别名
            String columnAliasName = null;
            // 获取该列的数据模式
            JSONArray dataModeArray = column.has("dataModeArray") ? column.getJSONArray("dataModeArray") : null;
            DataMode dataModeObject;
            if (dataModeArray == null) {
                // 旧模式兼容
                DataModeEnum dataModeEnum;
                if (!column.has("dataMode")) {
                    dataModeEnum = DataModeEnum.ALL;
                } else {
                    String dataMode = column.getString("dataMode");
                    dataModeEnum = DataModeEnum.toType(dataMode);
                    if (dataModeEnum == null) {
                        throw new ServiceException("列[" + key + "]无法被处理，因为这个数据模式不存在：" + dataMode, HttpStatus.BAD_REQUEST);
                    }
                }
                dataModeObject = dataModeEnum.getDataMode();
            } else {
                boolean queryForm = false;
                boolean table = false;
                boolean addOrEditForm = false;
                boolean sqlQueryItem = false;
                boolean sqlQueryCondition = false;
                for (Object item : dataModeArray) {
                    switch ((String) item) {
                        case "queryForm" -> queryForm = true;
                        case "table" -> table = true;
                        case "addOrEditForm" -> addOrEditForm = true;
                        case "sqlQueryItem" -> sqlQueryItem = true;
                        case "sqlQueryCondition" -> sqlQueryCondition = true;
                        default -> {
                        }
                    }
                }
                dataModeObject = new DataMode(queryForm, table, addOrEditForm, sqlQueryItem, sqlQueryCondition);
            }

            // 生成SQL查询项集合
            if (dataModeObject.sqlQueryItem()) {
                String columnName;
                // 获取这个数据字段对应的表别名
                String joinTableAlias = key.substring(0, key.indexOf('.'));
                String queryKey;
                // 表别名如果是$，代表是函数，格式：$.datediff(u.date,i.date) day
                if (joinTableAlias.equals("$")) {
                    String[] spiltKeyArray = key.split(" ");
                    columnAliasName = "$_" + spiltKeyArray[1];
                    key = spiltKeyArray[0];
                    queryKey = spiltKeyArray[0].substring(2);
                } else {
                    columnAliasName = joinTableAlias + "_" + realKey;
                    queryKey = key;
                }
                queryKey = ConfigUtil.parseKey(queryKey);
                if (column.has("default")) {
                    String defaultValue = String.valueOf(column.get("default"));
                    if (!defaultValue.startsWith("$")) {
                        defaultValue = "'" + defaultValue + "'";
                    } else {
                        defaultValue = defaultValue.substring(1);
                    }
                    DbType dbType = DynamicDataSource.getDbType();
                    columnName = switch (dbType) {
                        case sqlserver -> "ISNULL(" + queryKey + "," + defaultValue + ") AS\"" + columnAliasName + "\"";
                        case mysql -> "IFNULL(" + queryKey + "," + defaultValue + ") AS\"" + columnAliasName + "\"";
                        case oracle -> "NVL(" + queryKey + "," + defaultValue + ") AS\"" + columnAliasName + "\"";
                        default -> throw new ServiceException("数据库类型不支持：" + dbType);
                    };
                } else {
                    columnName = queryKey + " AS \"" + columnAliasName + "\"";
                }
                // 追加数据列到生成的查询SQL片段中
                sql.append("\n\t").append(columnName).append(",");
                // 根据数据列情况追加JOIN表达式
                // useJoinByCount:false 用于查询的子表数据列在生成count语句时不需要join
                SQLParserUtil.putJoinCondition(joinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameObject, false);
                // 追加数据列和标题到映射集合中
                chineseQueryColumnArray.put(column.getString("title"));
            }
            // 生成表单查询条件集合
            if (dataModeObject.sqlQueryCondition()) {
                // 获取该列的查询类型
                QueryTypeEnum queryTypeEnum;
                if (column.has("queryType")) {
                    String queryType = column.getString("queryType");
                    queryTypeEnum = QueryTypeEnum.toType(queryType);
                    if (queryTypeEnum == null) {
                        throw new ServiceException("列[" + key + "]无法生成表单项，因为这个表单查询类型不存在：" + queryType, HttpStatus.BAD_REQUEST);
                    }
                } else {
                    queryTypeEnum = QueryTypeEnum.EQUALS;
                }
                // 组织动态生成表单查询条件的数据列集合
                JSONObject selectColumnItem = new JSONObject();
                selectColumnItem.put("queryType", queryTypeEnum.getValue());
                selectColumnItem.put("key", key);
                selectColumnArray.put(key, selectColumnItem);
            }
            // 生成表格列集合
            if (dataModeObject.table()) {
                // 组织表格展示列
                JSONObject item = new JSONObject();
                item.put("title", column.getString("title"));
                item.put("dataIndex", columnAliasName == null ? realKey : columnAliasName);
                // 只有生成查询项的才可以进行排序
                item.put("sorter", dataModeObject.sqlQueryItem());
                // 指定宽度
                int width = 0;
                JSONObject slot;
                if (!column.has("slot")) {
                    // 如果没有插槽，设置一个默认的文本溢出省略(length:16)
                    slot = new JSONObject();
                    slot.put("type", "ellipsis");
                    slot.put("value", 16);
                } else {
                    slot = column.getJSONObject("slot");
                }
                JSONObject scopedSlots = new JSONObject();
                scopedSlots.put("customRender", columnAliasName == null ? realKey : columnAliasName);
                item.put("scopedSlots", scopedSlots);
                item.put("slots", scopedSlots);
                if (slot.has("type")) {
                    // 插槽类型
                    String slotType = slot.getString("type");
                    SlotTypeEnum slotTypeEnum = SlotTypeEnum.toType(slotType);
                    if (slotTypeEnum == null) {
                        throw new ServiceException("列[" + key + "]无法适用插槽，因为这个插槽类型不存在：" + slotType, HttpStatus.BAD_REQUEST);
                    }
                    item.put("slotType", slotType);
                    switch (slotTypeEnum) {
                        case DATE_TIME -> width = 160;
                        case ELLIPSIS -> {
                            int ellipsisValue = slot.getInt("value");
                            BigDecimal b = new BigDecimal(String.valueOf(ellipsisValue * 7 + 20 + 32));
                            width = b.setScale(0, RoundingMode.UP).intValue();
                            item.put("slotValue", ellipsisValue);
                        }
                        case BADGE -> {
                            item.put("slotKeyMap", String.valueOf(slot.get("keyMap")));
                            width = 130;
                        }
                        case ACTION -> {
                            String actionText = slot.optString("actionText", "详情");
                            item.put("slotValue", actionText);
                        }
                    }
                } else {
                    throw new ServiceException("列[" + key + "]无法适用插槽，因为没有指定插槽类型参数", HttpStatus.BAD_REQUEST);
                }
                if (width > 0) {
                    item.put("width", width);
                }
                // 如果有固定列
                if (column.has("fixed")) {
                    String fixedType = column.getString("fixed");
                    ColumnFixedTypeEnum columnFixedTypeEnum = ColumnFixedTypeEnum.toType(fixedType);
                    if (columnFixedTypeEnum == null) {
                        throw new ServiceException("列[" + key + "]无法适用固定列，因为这个固定列类型不存在：" + fixedType, HttpStatus.BAD_REQUEST);
                    }
                    item.put("fixed", fixedType);
                }
                showColumnArray.put(item);
            }
            // 生成表单项集合
            if (dataModeObject.queryForm() || dataModeObject.addOrEditForm()) {
                // 获取该列的表单类型
                FormTypeEnum formTypeEnum = FormBuildUtil.getFormType(column, key);
                // 获取该列的新增/修改场景类型
                AddOrEditTypeEnum addOrEditTypeEnum;
                if (dataModeObject.addOrEditForm()) {
                    if (column.has("addOrEdit")) {
                        String addOrEditType = column.getString("addOrEdit");
                        addOrEditTypeEnum = AddOrEditTypeEnum.toType(addOrEditType);
                        if (addOrEditTypeEnum == null) {
                            throw new ServiceException("列[" + key + "]无法生成表单项，因为这个新增/修改场景类型不存在：" + addOrEditType, HttpStatus.BAD_REQUEST);
                        }
                    } else {
                        addOrEditTypeEnum = AddOrEditTypeEnum.ALL;
                    }
                } else {
                    addOrEditTypeEnum = AddOrEditTypeEnum.NO;
                }
                // 组织表单项
                JSONObject item = new JSONObject();
                FormBuildUtil.buildFileRes(column, formTypeEnum, item);
                item.put("model", key.replaceFirst("\\.", "_"));
                item.put("type", formTypeEnum.getValue());
                item.put("queryType", column.optString("queryType", null));
                item.put("name", column.getString("title"));
                if (column.has("selectKeyName")) {
                    // 获取这个数据字段对应的表别名
                    String selectKeyName = column.getString("selectKeyName");
                    String joinTableAlias = selectKeyName.substring(0, selectKeyName.indexOf('.'));
                    // 获取该列的数据字段（字段名）
                    index = key.indexOf('.');
                    if (index == -1) {
                        index = 0;
                    } else {
                        index = index + 1;
                    }
                    String realSelectKey = selectKeyName.substring(index);
                    if (!selectKeyName.equals(key)) {
                        // 追加数据列到生成的查询SQL片段中
                        sql.append("\n\t").append(selectKeyName).append(" ").append(joinTableAlias).append("_").append(realSelectKey).append(",");
                        // 组织动态生成表单查询条件的数据列集合
                        JSONObject selectColumnItem = new JSONObject();
                        selectColumnItem.put("queryType", column.optString("queryType", QueryTypeEnum.EQUALS.getValue()));
                        selectColumnItem.put("key", selectKeyName);
                        selectColumnArray.put(selectKeyName, selectColumnItem);
                        // 修改表单项的model
                        item.put("model", selectKeyName.replace('.', '_'));
                    }
                }
                if (column.has("selectKey")) {
                    Object selectKey = column.get("selectKey");
                    if (selectKey instanceof JSONArray) {
                        item.put("keys", selectKey);
                    } else {
                        item.put("keyName", selectKey);
                    }
                    if (column.has("lazyLoad")) {
                        item.put("lazyLoad", column.get("lazyLoad"));
                    }
                }
                FormBuildUtil.buildRuleAndPlaceholder(column, key, item);
                item.put("addOrEdit", addOrEditTypeEnum.getValue());
                item.put("isOnlyAddOrEdit", !dataModeObject.queryForm());
                if (addOrEditTypeEnum == AddOrEditTypeEnum.SILENCE_ADD) {
                    // 获取该列的字段用途
                    SilencePurposeEnum silencePurposeEnum;
                    if (column.has("silencePurpose")) {
                        String silencePurpose = column.getString("silencePurpose");
                        silencePurposeEnum = SilencePurposeEnum.toType(silencePurpose);
                        if (silencePurposeEnum == null) {
                            throw new ServiceException("列[" + key + "]无法生成表单项，因为这个静默新增场景指定的字段用途不存在：" + silencePurpose, HttpStatus.BAD_REQUEST);
                        } else if (silencePurposeEnum == SilencePurposeEnum.CUSTOMIZE) {
                            // 如果是自定义用途，需要获取Logic名称
                            if (column.has("silenceSource")) {
                                item.put("silenceSource", column.getString("silenceSource"));
                            } else {
                                throw new ServiceException("列[" + key + "]无法生成表单项，因为静默新增场景的自定义字段用途必须指定业务逻辑名称(silenceSource)", HttpStatus.BAD_REQUEST);
                            }
                        }
                        item.put("silencePurpose", column.getString("silencePurpose"));
                    } else {
                        throw new ServiceException("列[" + key + "]无法生成表单项，因为静默新增场景必须指定字段用途", HttpStatus.BAD_REQUEST);
                    }
                }
                formItemArray.put(item);
            }
        }
        // 取得OrderBy配置字段的表别名，判断是否需要Join
        String orderBy = config.getString("orderBy");
        String joinTableAlias = orderBy.substring(0, orderBy.indexOf('.'));
        // 根据OrderBy的列情况追加JOIN表达式，用于OrderBy的数据列在生成count语句时不需要join
        SQLParserUtil.putJoinCondition(joinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameObject, false);
        // 如果使用了固定查询表达式，则追加该表达式
        String conditionStr;
        if (config.has("condition")) {
            JSONObject condition = config.getJSONObject("condition");
            conditionStr = condition.getString("value");
            if (condition.has("join")) {
                JSONArray joinArray = condition.getJSONArray("join");
                for (Object item : joinArray) {
                    String joinItem = item.toString();
                    // 追加JOIN表达式
                    // 在执行countSQL语句时,用于查询条件的子表数据列需要JOIN子表
                    SQLParserUtil.putJoinCondition(joinItem, tableAliasName, joinTableNameObject, canJoinTableNameObject, true);
                }
            }
        } else {
            conditionStr = "1=1";
        }
        // 移除查询列中最后多余的逗号
        sql.deleteCharAt(sql.length() - 1);
        // 追加SQL片段到querySQL中
        sql.append("\n").append("FROM ").append(tableName);
        // 追加SQL片段到countSQL中
        countSql.append("\n").append("FROM ").append(tableName);
        JSONObject result = new JSONObject();
        result.put("tableName", tableName);
        result.put("tableAliasName", tableAliasName);
        result.put("countSql", countSql.toString());
        result.put("querySql", sql.toString());
        result.put("joinTableNameObject", joinTableNameObject);
        result.put("canJoinTableNameObject", canJoinTableNameObject);
        result.put("selectColumn", selectColumnArray);
        result.put("orderBy", orderBy);
        result.put("condition", conditionStr);
        result.put("columnJson", showColumnArray);
        result.put("formJson", formItemArray);
        result.put("buttonState", buttonState);
        result.put("apiSlot", apiSlot);
        result.put("serviceName", serviceName);
        result.put("chineseQueryColumnArray", chineseQueryColumnArray);
        return result;
    }
}
