package com.af.v4.system.common.jpa.utils;

import com.af.v4.system.common.core.proxy.jpa.IQueryParamsProxy;
import org.json.JSONObject;

/**
 * 查询参数
 */
public class QueryParams implements IQueryParamsProxy {
    /**
     * 页数
     */
    private final Integer pageNo;

    /**
     * 页大小
     */
    private final Integer pageSize;

    /**
     * 查询数据源
     */
    private final String dataSource;

    /**
     * 预编译查询参数
     */
    private final JSONObject parameterList;

    // 私有构造函数，防止直接实例化
    private QueryParams(Builder builder) {
        this.pageNo = builder.pageNo;
        this.pageSize = builder.pageSize;
        this.dataSource = builder.dataSource;
        this.parameterList = builder.parameterList;
    }

    // Getters
    public Integer getPageNo() {
        return pageNo;
    }

    public Integer getPageSize() {
        return pageSize;
    }

    public String getDataSource() {
        return dataSource;
    }

    public JSONObject getParameterList() {
        return parameterList;
    }

    @Override
    public String toString() {
        return "pageNo = " + pageNo +
                ", pageSize = " + pageSize +
                ", dataSource = " + dataSource +
                ", params = " + parameterList;
    }

    // Builder 静态内部类
    public static class Builder implements IQueryParamsProxy.Builder {
        private Integer pageNo;
        private Integer pageSize;
        private String dataSource;
        private JSONObject parameterList;

        public Builder() {
        }

        /**
         * 静态工厂方法，通过 JSONObject 构造 QueryParams
         *
         * @param jsonObject 参数
         * @return 构造器
         */
        public static Builder fromJSONObject(JSONObject jsonObject) {
            Builder builder = new Builder();
            if (jsonObject == null) {
                return builder;
            }
            if (jsonObject.has("pageNo")) {
                builder.pageNo(jsonObject.getInt("pageNo"));
            }
            if (jsonObject.has("pageSize")) {
                builder.pageSize(jsonObject.getInt("pageSize"));
            }
            if (jsonObject.has("dataSource")) {
                builder.dataSource(jsonObject.getString("dataSource"));
            }
            if (jsonObject.has("params")) {
                builder.parameterList(jsonObject.getJSONObject("params"));
            }
            return builder;
        }

        /**
         * 页码
         */
        public Builder pageNo(Integer pageNo) {
            this.pageNo = pageNo;
            return this;
        }

        /**
         * 页数
         */
        public Builder pageSize(Integer pageSize) {
            this.pageSize = pageSize;
            return this;
        }

        /**
         * 数据源
         */
        public Builder dataSource(String dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        /**
         * 添加参数化查询参数
         */
        public Builder addParameter(String key, Object value) {
            if (this.parameterList == null) {
                this.parameterList = new JSONObject();
            }
            this.parameterList.put(key, value.toString());
            return this;
        }

        public boolean hasParameter(String key) {
            return this.parameterList != null && this.parameterList.has(key);
        }

        /**
         * 参数化查询参数
         */
        public Builder parameterList(JSONObject parameterList) {
            this.parameterList = parameterList;
            return this;
        }

        public IQueryParamsProxy build() {
            return new QueryParams(this);
        }
    }
}
