package com.af.v4.system.common.jpa.service;

import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.config.DynamicDataSourceConfig;
import com.af.v4.system.common.jpa.enums.ColumnTypeEnum;
import com.af.v4.system.common.jpa.enums.IDTypeEnum;
import com.af.v4.system.common.jpa.session.SessionPool;
import com.af.v4.system.common.jpa.types.MetaData;
import com.af.v4.system.common.jpa.types.Pair;
import com.af.v4.system.common.jpa.types.SubClasses;
import com.af.v4.system.common.plugins.io.IOTools;
import jakarta.annotation.PostConstruct;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.hibernate.UnknownEntityTypeException;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.generator.Generator;
import org.hibernate.id.*;
import org.hibernate.id.enhanced.SequenceStyleGenerator;
import org.hibernate.id.uuid.UuidGenerator;
import org.hibernate.metamodel.MappingMetamodel;
import org.hibernate.persister.entity.AbstractEntityPersister;
import org.hibernate.type.CollectionType;
import org.hibernate.type.ManyToOneType;
import org.hibernate.type.OneToOneType;
import org.hibernate.type.Type;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.io.InputStream;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 实体元数据服务
 *
 * @author Mr.river
 * @since 2.9.11
 */
@Service
public class MetaDataService {
    private static final Logger LOGGER = LoggerFactory.getLogger(MetaDataService.class);
    /**
     * 相当于线程锁,用于线程安全
     */
    private final static Object SYNC_LOCK = new Object();
    private static final Map<String, Map<String, MetaData>> META_MAP = new ConcurrentHashMap<>();
    private static final Map<String, Map<String, String>> ENTITY_LIFT_MAP = new ConcurrentHashMap<>();
    /**
     * 是否已加载所有元数据
     */
    private static volatile Boolean isAllMetaDataLoaded = false;
    private final DynamicDataSourceConfig dynamicDataSourceConfig;
    private final SessionPool sessionPool;

    public MetaDataService(DynamicDataSourceConfig dynamicDataSourceConfig, SessionPool sessionPool) {
        this.dynamicDataSourceConfig = dynamicDataSourceConfig;
        this.sessionPool = sessionPool;
    }

    public static Map<String, String> getEntityLiftMap() {
        String dataSourceName = DynamicDataSource.getDataSource();
        return ENTITY_LIFT_MAP.computeIfAbsent(dataSourceName, k -> new ConcurrentHashMap<>());
    }

    public static Map<String, MetaData> getMetaMap() {
        String dataSourceName = DynamicDataSource.getDataSource();
        return META_MAP.computeIfAbsent(dataSourceName, k -> new ConcurrentHashMap<>());
    }

    public static String getEntityLiftValueByKey(String entityName) {
        return getEntityLiftMap().get(entityName);
    }

    private static void putEntityLiftValue(String key, String value) {
        getEntityLiftMap().put(key, value);
    }

    @PostConstruct
    public void init() {
        // 加载subclassMap
        loadSubclassMap();
    }

    public MetaData getMetaMapItemByKey(String entityName) {
        MetaData result = getMetaMap().computeIfAbsent(entityName, this::getMetaData);
        if (result == null) {
            String dataSourceName = DynamicDataSource.getDataSource();
            throw new ServiceException("数据源[" + dataSourceName + "]未注册实体[" + entityName + "]，请检查Hibernate Mapping", HttpStatus.ERROR);
        }
        return result;
    }

    /**
     * 加载subclassMap
     */
    public void loadSubclassMap() {
        SAXReader reader = new SAXReader();
        // 加载根目录下的subclass
        IOTools.getStream("subclass.xml",
                stream -> loadSubclass(reader, stream));
        List<String> subClassList = dynamicDataSourceConfig.getSubClass();
        if (subClassList != null) {
            for (String moduleName : subClassList) {
                // module要添加路径分隔符
                if (!"".equals(moduleName)) {
                    moduleName += "/";
                }
                String finalModuleName = moduleName;
                IOTools.getStream("/" + moduleName + "subclass.xml",
                        stream -> loadSubclass(reader, stream),
                        notFindPath -> LOGGER.warn("{}模块下无subclass.xml文件", finalModuleName));
            }
        }
    }

    /**
     * 加载所有实体元数据
     */
    public void loadAllMetaData() {
        if (isAllMetaDataLoaded) {
            return;
        }
        synchronized (SYNC_LOCK) {
            if (isAllMetaDataLoaded) {
                return;
            }
            Set<String> dataSourceNameList = DynamicDataSource.getDataSourceMap().keySet();
            for (String dataSource : dataSourceNameList) {
                DynamicDataSource.withDataSource(dataSource, () -> {
                    //获取每个实体的元数据
                    sessionPool.getSessionFactory().getMetamodel()
                            .forEachEntityDescriptor(entityPersister -> this.getMetaMapItemByKey(entityPersister.getEntityName()));
                });
            }
            isAllMetaDataLoaded = true;
        }
    }

    private void loadSubclass(SAXReader reader, InputStream input) {
        Document document;
        try {
            document = reader.read(input);
        } catch (DocumentException e) {
            throw new RuntimeException(e);
        }
        Element root = document.getRootElement();
        for (Iterator<Element> it = root.elementIterator("entity"); it.hasNext(); ) {
            Element elm = it.next();
            String entityName = elm.attribute("name").getValue();
            String parentEntity = elm.attribute("parentEntity").getValue();
            String discProperty = elm.attribute("discProperty").getValue();
            String discriminator = elm.attribute("discriminator").getValue();
            MetaData em = getMetaMapItemByKey(parentEntity);
            Map<String, SubClasses> subclasses;
            if (em.hasSubclasses()) {
                subclasses = em.getSubclasses();
            } else {
                subclasses = new LinkedHashMap<>();
                em.setSubclasses(subclasses);
            }
            subclasses.put(entityName, new SubClasses(discProperty, discriminator));
            putEntityLiftValue(entityName, parentEntity);
        }
    }

    /**
     * 获取元数据
     *
     * @param entityName 实体名
     */
    private MetaData getMetaData(String entityName) {
        MetaData map = new MetaData();
        map.setEntityName(entityName);
        // 获取表名
        MappingMetamodel metamodel = sessionPool.getSessionFactory().getMappingMetamodel();
        AbstractEntityPersister aep;
        try {
            aep = (AbstractEntityPersister) metamodel.getEntityDescriptor(entityName);
        } catch (UnknownEntityTypeException e) {
            return null;
        }
        map.setTableName(aep.getTableName());
        // 获取主键属性名称
        String idPropName = aep.getIdentifierPropertyName();
        if (idPropName == null) {
            LOGGER.warn("停止处理【{}】的映射，可能含有不支持的映射结构", entityName);
            return null;
        }
        map.setIdName(idPropName);
        String idColName = aep.getPropertyColumnNames(idPropName)[0];
        map.setIdColName(idColName);
        // 获取主键类型
        ColumnTypeEnum idType = ColumnTypeEnum.toType(aep.getIdentifierType().getName());
        map.setIdType(idType);

        // 仅支持guid、sequence（oracle）、auto-increment、assigned、outside（id来自主实体的一对一）
        Generator idg = aep.getGenerator();
        if (idg instanceof UUIDGenerator ||
                idg instanceof UuidGenerator ||
                idg instanceof UUIDHexGenerator ||
                idg instanceof GUIDGenerator) {
            map.setIdGenerator(IDTypeEnum.ID_GUID);
        } else if (idg instanceof SequenceStyleGenerator generator) {
            String sequenceName = generator.getDatabaseStructure().getPhysicalName().render();
            map.setSequence(sequenceName);
            map.setIdGenerator(IDTypeEnum.ID_SEQ);
        } else if (idg instanceof IdentityGenerator) {
            map.setIdGenerator(IDTypeEnum.ID_AUTO);
        } else if (idg instanceof Assigned) {
            map.setIdGenerator(IDTypeEnum.ID_ASSIGNED);
        } else if (idg instanceof ForeignGenerator) {
            map.setIdGenerator(IDTypeEnum.ID_FOREIGNER);
        } else {
            throw new ServiceException("不支持的Hibernate Id生成器类型，请咨询研发组" + idg.getClass().getName());
        }

        // 判断该实体是否有乐观锁（即版本号）控制
        if (aep.isVersioned()) {
            String attr = aep.getPropertyNames()[aep.getVersionProperty()];
            map.setVerName(attr);
            map.setVerColName(aep.getPropertyColumnNames(attr)[0]);
            map.setVerType(ColumnTypeEnum.toType(aep.getPropertyType(attr).getName()));
        }

        // 列
        Map<String, Pair> columns = new LinkedHashMap<>(8);
        map.setColumns(columns);

        // 一对一，一对多关系映射
        Map<String, Pair> links = new HashMap<>(1);
        map.setLinks(links);

        // 处理反向多对一的关系
        Map<String, String> inverses = new HashMap<>(1);
        map.setInverses(inverses);
        Map<String, Pair> inverseMap = new HashMap<>(1);
        map.setInverseid(inverseMap);

        // 遍历实体属性集合
        for (String property : aep.getPropertyNames()) {
            //获取类型
            Type type = aep.getPropertyType(property);
            if (type instanceof CollectionType || type instanceof OneToOneType) {
                org.hibernate.persister.entity.Joinable ja;
                String linkEntity;
                SessionFactoryImplementor sf = sessionPool.getSessionFactory();
                if (type instanceof CollectionType st) {
                    ja = st.getAssociatedJoinable(sf);
                    linkEntity = st.getAssociatedEntityName(sf);
                } else {
                    OneToOneType st = (OneToOneType) type;
                    ja = st.getAssociatedJoinable(sf);
                    linkEntity = st.getAssociatedEntityName(sf);
                }
                String foreignKey = ja.getKeyColumnNames()[0];
                String link;
                if (type instanceof OneToOneType) {
                    link = property;
                } else {
                    link = ja.getName().substring(entityName.length() + 1);

                    MetaData linkedMap;
                    //prevent infinite recursion
                    if (linkEntity.equals(entityName)) {
                        linkedMap = map;
                    } else {
                        linkedMap = getMetaMapItemByKey(linkEntity);
                    }
                    Map<String, Pair> pkfk;
                    assert linkedMap != null;
                    if (linkedMap.hasAssociations()) {
                        pkfk = linkedMap.getAssociations();
                        pkfk.put(entityName, new Pair(foreignKey, idType));
                    } else {
                        pkfk = new HashMap<>();
                        pkfk.put(entityName, new Pair(foreignKey, idType));
                        linkedMap.setAssociations(pkfk);
                    }
                }
                links.put(link, new Pair(linkEntity, foreignKey));
            } else if (type instanceof ManyToOneType mto) {
                // 处理 ManyToOneType
                String linkedEntity = mto.getAssociatedEntityName();
                String colName = mto.getAssociatedJoinable(sessionPool.getSessionFactory()).getKeyColumnNames()[0];
                inverses.put(property, linkedEntity);
                inverseMap.put(property, new Pair(colName, getMetaMapItemByKey(linkedEntity).getIdType()));
            } else {
                String columnName = aep.getPropertyColumnNames(property)[0];
                columns.put(property, new Pair(columnName, ColumnTypeEnum.toType(type.getName())));
            }
        }
        return map;
    }

    /**
     * 解析元数据
     */
    public void enchanceMetaData(MetaData pmd, String entityName) {
        MetaData md = getMetaMapItemByKey(entityName);

        Map<String, Pair> links = md.getLinks();
        Map<String, Pair> plinks = pmd.getLinks();
        plinks.putAll(links);

        Map<String, SubClasses> sub;
        if (md.hasSubclasses()) {
            sub = md.getSubclasses();
        } else {
            sub = new LinkedHashMap<>();
        }
        Map<String, SubClasses> psub;
        if (pmd.hasSubclasses()) {
            psub = pmd.getSubclasses();
        } else {
            psub = new LinkedHashMap<>();
        }
        for (String key : sub.keySet()) {
            psub.put(key, sub.get(key).copy());
        }

        Map<String, String> inv = md.getInverses();
        Map<String, String> pinv = pmd.getInverses();
        for (String key : inv.keySet()) {
            pinv.put(key, inv.get(key));
        }

        Map<String, Pair> invid;
        if (md.hasInverseIds()) {
            invid = md.getInverseid();
        } else {
            invid = new HashMap<>(0);
        }
        Map<String, Pair> pinvid;
        if (pmd.hasInverseIds()) {
            pinvid = pmd.getInverseid();
        } else {
            pinvid = new HashMap<>(invid.size());
        }

        for (String key : invid.keySet()) {
            pinvid.put(key, invid.get(key).copy());
        }
    }
}
