package com.af.v4.system.common.jpa.action;

import com.af.v4.system.common.core.constant.CodeNormsConstants;
import com.af.v4.system.common.core.exception.CheckedException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.jpa.session.SessionPool;
import com.af.v4.system.common.jpa.transformer.StandardAliasTransformer;
import org.hibernate.Session;
import org.hibernate.exception.SQLGrammarException;
import org.hibernate.query.NativeQuery;
import org.hibernate.query.TupleTransformer;
import org.json.JSONArray;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.function.Supplier;

/**
 * SQL执行器
 *
 * @author Mr.river
 */
@Component
public class SqlAction {

    private static final Logger LOGGER = LoggerFactory.getLogger(SqlAction.class);
    private final SessionPool sessionPool;

    public SqlAction(SessionPool sessionPool) {
        this.sessionPool = sessionPool;
    }

    public JSONArray queryAll(String name, String sql, String dataSource) {
        return query(name, sql, null, null, dataSource);
    }

    public JSONArray query(String name, String sql, String dataSource) {
        return query(name, sql, 1, 1000, dataSource);
    }

    public JSONArray query(String name, String sql, Integer pageNo, Integer pageSize, String dataSource) {
        return (JSONArray) sqlMonitor(() -> {
            InnerSQLCall sqlCall = new InnerSQLCall(sql, pageNo, pageSize, StandardAliasTransformer.INSTANCE);
            JSONArray array = sqlCall.doInHibernate(sessionPool.getSession());
            if (array != null) {
                writeSqlResult(name, array);
            }
            return array;
        }, name, dataSource);
    }

    public Integer exec(String name, String sql, String dataSource) {
        return (int) sqlMonitor(() -> {
            Session session = sessionPool.getSession();
            NativeQuery<?> queryObject = session.createNativeQuery(sql);
            return queryObject.executeUpdate();
        }, name, dataSource);
    }

    private Object sqlMonitor(Supplier<Object> supplier, String name, String dataSource) {
        if (StringUtils.isEmpty(name)) {
            throw new CheckedException(CodeNormsConstants.E_01);
        }
        LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "执行SQL[{}]", name);
        long begin = System.currentTimeMillis();
        try {
            return DynamicDataSource.withDataSource(dataSource, supplier);
        } finally {
            long end = System.currentTimeMillis();
            LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "SQL[{}]耗时：{}ms", name, end - begin);
        }
    }

    /**
     * 打印SQL执行结果
     *
     * @param name  SQL名称
     * @param array 执行结果
     */
    private void writeSqlResult(String name, JSONArray array) {
        String result = array.toString();
        int length = result.length();
        if (length > 2000) {
            result = result.substring(0, 2000) + "...后续还有" + (length - 2000) + "个字符";
        }
        LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "SQL[{}]查询结果: \n{}", name, result);
    }

    private record InnerSQLCall(String sql, Integer page, Integer rows,
                                TupleTransformer<?> transformer) implements HibernateCallback<JSONArray> {

        @Override
        public JSONArray doInHibernate(Session session) {
            NativeQuery<?> q = session.createNativeQuery(sql);
            q.setTupleTransformer(transformer);
            try {
                List<?> list;
                if (page == null || rows == null || rows <= 0) {
                    list = q.list();
                } else {
                    int realPage = page <= 0 ? 0 : page - 1;
                    list = q.setFirstResult(realPage * rows).setMaxResults(rows).list();
                }
                if (transformer instanceof StandardAliasTransformer) {
                    return new JSONArray().putAll(list);
                } else {
                    return new JSONArray(list);
                }
            } catch (SQLGrammarException ex) {
                // 把sql语句添加到异常信息中
                String msg = "sql:\n" + sql + "\n" + ex.getMessage();
                throw new SQLGrammarException(msg, ex.getSQLException());
            }
        }
    }
}
