package com.af.v4.system.common.jpa.transformer;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.date.TemporalAccessorUtil;
import com.af.v4.system.common.jpa.service.MetaDataService;
import com.af.v4.system.common.jpa.utils.ClobUtil;
import com.af.v4.system.common.plugins.security.SensitiveDataEnhancer;
import com.af.v4.system.common.plugins.security.SubstitutionCipherUtil;
import com.af.v4.system.common.security.auth.AuthUtil;
import org.hibernate.query.TupleTransformer;
import org.json.JSONObject;
import org.json.JSONParserConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.sql.Clob;
import java.time.temporal.TemporalAccessor;
import java.util.Date;
import java.util.Map;

/**
 * 标准结果转换器
 */
public class StandardAliasTransformer implements TupleTransformer<JSONObject> {

    private static final Logger log = LoggerFactory.getLogger(StandardAliasTransformer.class);

    public static final StandardAliasTransformer INSTANCE = new StandardAliasTransformer();

    private StandardAliasTransformer() {
    }

    @SuppressWarnings("all")
    protected static void runTrans(Object obj, Object[] tuple, String[] aliases) {
        for (int i = 0; i < aliases.length; i++) {
            String alias = aliases[i];
            Object value = tuple[i];
            switch (value) {
                case null -> value = JSONObject.NULL;
                case BigDecimal bigDecimal -> {
                    if (isIntegerValue(bigDecimal)) {
                        long longValue = bigDecimal.longValue();
                        if (longValue < Integer.MAX_VALUE) {
                            value = (int) longValue;
                        } else {
                            value = longValue;
                        }
                    }
                }
                case Date date ->
                    // 处理日期格式
                        value = DateUtil.formatDateTime(date);
                case TemporalAccessor temporalAccessor ->
                    // 处理日期格式
                        value = TemporalAccessorUtil.format(temporalAccessor, "yyyy-MM-dd HH:mm:ss");
                case Clob clob -> value = ClobUtil.getClobString(clob);
                default -> {
                }
            }

            // 脱敏字段出库处理：敏感字段根据是否加密和权限进行处理
            if (value instanceof String str) {
                String lowerAlias = (alias != null) ? alias.toLowerCase() : null;
                Map<String, String> maskMap = MetaDataService.getSensitiveMaskMap();

                // 判断是否为敏感字段
                if (lowerAlias != null && maskMap.containsKey(lowerAlias)) {
                    String plainText = str;

                    // 1. 如果是加密数据，先解密
                    if (SubstitutionCipherUtil.isStrictCipher(str)) {
                        try {
                            plainText = SubstitutionCipherUtil.decrypt(str);
                        } catch (Exception e) {
                            // 解密失败时记录日志并保持原值
                            log.warn("敏感字段解密异常: field={}, error={}", lowerAlias, e.getMessage());
                            // 解密失败，保持密文原值，不进行后续脱敏处理
                            plainText = null;
                        }
                    }

                    // 2. 根据权限决定返回明文还是脱敏值
                    if (plainText != null) {
                        if (AuthUtil.hasSensitiveViewPermission(lowerAlias)) {
                            // 有权限：返回明文
                            value = plainText;
                        } else {
                            // 无权限：返回脱敏值
                            String strategy = maskMap.get(lowerAlias);
                            value = SensitiveDataEnhancer.applyMaskingStrategy(lowerAlias, plainText, strategy);
                        }
                    }
                }
            }
            String key = (alias != null) ? alias.toLowerCase() : ("col" + i);
            if (obj instanceof JSONObject object) {
                object.put(key, value);
            } else if (obj instanceof Map objectMap) {
                objectMap.put(key, value);
            }
        }
    }

    private static boolean isIntegerValue(BigDecimal bd) {
        return bd.stripTrailingZeros().scale() <= 0;
    }

    @Override
    public JSONObject transformTuple(Object[] tuple, String[] aliases) {
        JSONObject result = new JSONObject(new JSONParserConfiguration().setOrdered(true));
        runTrans(result, tuple, aliases);
        return result;
    }

}
