package com.af.v4.system.common.jpa.utils;

import com.af.v4.system.common.datasource.config.AfDataSourceConfig;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.common.datasource.wrapper.AfDataSourceWrapper;
import com.af.v4.system.common.jpa.action.MonitorInterceptor;
import com.af.v4.system.common.jpa.dialect.MyDialectResolver;
import org.springframework.context.ApplicationContext;
import org.springframework.core.io.Resource;
import org.springframework.orm.hibernate5.LocalSessionFactoryBean;
import org.springframework.orm.hibernate5.SpringSessionContext;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.List;
import java.util.Properties;

/**
 * SessionFactoryBeanBuilder
 *
 * @author Mr.river
 * @since 2.9.0
 */
@Component
public class SessionFactoryBeanBuilder {

    private final ApplicationContext context;

    public SessionFactoryBeanBuilder(ApplicationContext context) {
        this.context = context;
    }

    private static Properties getProperties(AfDataSourceWrapper wrapper) {
        AfDataSourceConfig value = wrapper.getConfig();
        Properties properties = new Properties();
        String hbm2dll = "none";
        // 所有数据源均不允许自动修改表结构
        properties.setProperty("hibernate.hbm2ddl.auto", hbm2dll);
        // 日志配置
        properties.setProperty("hibernate.show_sql", "false");
        properties.setProperty("hibernate.format_sql", "false");
        properties.setProperty("hibernate.current_session_context_class", SpringSessionContext.class.getName());
        // 注册StatementInspector
        properties.setProperty("hibernate.session_factory.statement_inspector", MonitorInterceptor.class.getName());
        // 序列不一样时 处理方式 FIX / LOG / EXCEPTION / NONE org.hibernate.id.enhanced.SequenceStyleGenerator.adjustIncrementSize
        properties.setProperty("hibernate.id.sequence.increment_size_mismatch_strategy", "LOG");
        // 数据库连接配置
        // 方言解析器配置
        if (value.getDialect() != null) {
            properties.setProperty("hibernate.dialect", value.getDialect());
        } else {
            properties.setProperty("hibernate.dialect_resolvers", MyDialectResolver.class.getCanonicalName());
        }
        boolean isClickhouse = wrapper.getDbType() == DbType.clickhouse;
        if (isClickhouse) {
            properties.setProperty("hibernate.allow_update_outside_transaction", "true");
        }
        return properties;
    }

    public LocalSessionFactoryBean buildSessionFactory(String dataSourceName, AfDataSourceWrapper dataSourceWrapper) throws IOException {
        LocalSessionFactoryBean sessionFactoryBean = new LocalSessionFactoryBean();
        sessionFactoryBean.setDataSource(dataSourceWrapper.getDataSource());
        sessionFactoryBean.setPackagesToScan(STR."com.af.v4.entity.\{dataSourceName}");
        Properties properties = getProperties(dataSourceWrapper);
        sessionFactoryBean.setHibernateProperties(properties);
        // configLocations
        List<String> cfgLocations = dataSourceWrapper.getConfig().getCfgLocations();
        if (cfgLocations != null && !cfgLocations.isEmpty()) {
            Resource[] cfgLocationResources = cfgLocations.stream()
                    .map(context::getResource).toList()
                    .toArray(new Resource[cfgLocations.size()]);
            sessionFactoryBean.setConfigLocations(cfgLocationResources);
        }
        sessionFactoryBean.afterPropertiesSet();
        return sessionFactoryBean;
    }
}
