package com.af.v4.system.common.jpa.service;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Tuple;
import cn.hutool.core.util.StrUtil;
import com.af.v4.system.api.RemoteEntityService;
import com.af.v4.system.api.factory.DynamicFeignClientFactory;
import com.af.v4.system.common.core.constant.CodeNormsConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.proxy.jpa.IEntityServiceProxy;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.config.AfDataSourceConfig;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.common.jpa.action.SqlAction;
import com.af.v4.system.common.jpa.enums.ColumnTypeEnum;
import com.af.v4.system.common.jpa.enums.IDTypeEnum;
import com.af.v4.system.common.jpa.enums.TableShardingStrategyEnum;
import com.af.v4.system.common.jpa.session.SessionPool;
import com.af.v4.system.common.jpa.types.MetaData;
import com.af.v4.system.common.jpa.types.Pair;
import com.af.v4.system.common.jpa.types.SubClasses;
import com.af.v4.system.common.jpa.utils.Condition;
import com.af.v4.system.common.jpa.utils.QueryParams;
import com.af.v4.system.common.plugins.core.CommonTools;
import com.microsoft.sqlserver.jdbc.SQLServerBulkCopy;
import com.microsoft.sqlserver.jdbc.SQLServerBulkCopyOptions;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.sql.rowset.CachedRowSet;
import javax.sql.rowset.RowSetProvider;
import java.io.Serializable;
import java.math.BigDecimal;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;

/**
 * 实体操作服务
 *
 * @author Mr.river
 * @since 1.0.0
 */
@Primary
@Service
@Transactional(rollbackFor = Exception.class)
public class EntityService implements IEntityServiceProxy {
    private static final Logger LOGGER = LoggerFactory.getLogger(EntityService.class);
    private final SessionPool sessionPool;
    private final SqlAction sqlAction;

    private final DynamicFeignClientFactory<RemoteEntityService> dynamicFeignClientFactory;
    private final ClickhouseService clickhouseService;
    private final MetaDataService metaDataService;

    public EntityService(SessionPool sessionPool, SqlAction sqlAction, DynamicFeignClientFactory<RemoteEntityService> dynamicFeignClientFactory, ClickhouseService clickhouseService, MetaDataService metaDataService) {
        this.sessionPool = sessionPool;
        this.sqlAction = sqlAction;
        this.dynamicFeignClientFactory = dynamicFeignClientFactory;
        this.clickhouseService = clickhouseService;
        this.metaDataService = metaDataService;
    }

    /**
     * 根据分表策略获取表名
     *
     * @param baseTableName 基础表名
     * @param shardingStrategy 分表策略
     * @return 处理后的表名
     */
    private String getTableNameWithSharding(String baseTableName, String shardingStrategy) {
        if (StrUtil.isBlank(shardingStrategy)) {
            return baseTableName;
        }

        TableShardingStrategyEnum strategy = TableShardingStrategyEnum.fromFormat(shardingStrategy);
        if (strategy == TableShardingStrategyEnum.NONE) {
            return baseTableName;
        }

        String suffix = DateUtil.format(new Date(), strategy.getFormat());
        return baseTableName + "_" + suffix;
    }

    public static Condition getCondition() {
        return Condition.build();
    }

    /**
     * 根据ID查询数据
     *
     * @param columns        列字段
     * @param entityName     实体名
     * @param id             主键值
     * @param dataSourceName 数据源名称
     * @return 实体
     */
    public JSONObject getById(String columns, String entityName, Object id, String dataSourceName) {
        MetaData map = metaDataService.getMetaMapItemByKey(entityName);
        String sql = "SELECT " + columns + " FROM " + map.getTableName() + " WHERE " + map.getIdColName() + " = '" + id + "'";
        JSONArray result = sqlAction.query("EntityService@getById", sql, new QueryParams.Builder().dataSource(dataSourceName).build());
        if (result.isEmpty()) {
            return null;
        } else {
            return result.getJSONObject(0);
        }
    }

    public JSONObject getById(String columns, String entityName, Object id) {
        return getById(columns, entityName, id, null);
    }

    public JSONObject getById(String entityName, Object id) {
        return getById("*", entityName, String.valueOf(id));
    }

    /**
     * 根据条件表达式查询数据
     *
     * @param columns        列字段
     * @param entityName     实体名
     * @param condition      条件表达式对象
     * @param dataSourceName 数据源名称
     * @return 实体
     */
    public JSONArray getByCondition(String columns, String entityName, Condition condition, String order, String dataSourceName) {
        MetaData map = metaDataService.getMetaMapItemByKey(entityName);
        String sql = "SELECT " + columns + " FROM " + map.getTableName() + " WHERE " + condition.getValue();
        if (order != null) {
            sql += " ORDER BY " + order;
        }
        return sqlAction.query("EntityService@getByCondition", sql, new QueryParams.Builder().dataSource(dataSourceName).build());
    }

    public JSONArray getByCondition(String columns, String entityName, Condition condition, String order) {
        return getByCondition(columns, entityName, condition, order, null);
    }

    public JSONArray getByCondition(String columns, String entityName, Condition condition) {
        return getByCondition(columns, entityName, condition, null);
    }

    public JSONArray getByCondition(String entityName, Condition condition) {
        return getByCondition("*", entityName, condition, null);
    }

    public JSONArray getByCondition(String entityName, Condition condition, String order) {
        return getByCondition("*", entityName, condition, order);
    }

    /**
     * 删除单个实体
     *
     * @param entityName     实体名
     * @param id             主键值
     * @param dataSourceName 数据源名称
     * @return 影响行数
     */
    public Integer delete(String entityName, Object id, String dataSourceName) {
        MetaData map = metaDataService.getMetaMapItemByKey(entityName);
        String sql = "DELETE FROM " + map.getTableName() + " WHERE " + map.getIdColName() + " = '" + id + "'";
        return sqlAction.exec("EntityService@Delete", sql, dataSourceName);
    }

    public Integer delete(String entityName, Object id) {
        return delete(entityName, id, null);
    }

    /**
     * 批量删除实体
     *
     * @param entityName     实体名
     * @param ids            主键值集合
     * @param dataSourceName 数据源名称
     * @return 影响行数
     */
    public Integer deleteAllByIds(String entityName, JSONArray ids, String dataSourceName) {
        MetaData map = metaDataService.getMetaMapItemByKey(entityName);
        String idStr = CommonTools.union(ids);
        String sql = "DELETE FROM " + map.getTableName() + " WHERE " + map.getIdColName() + " IN (" + idStr + " )";
        return sqlAction.exec("EntityService@deleteAllByIds", sql, dataSourceName);
    }

    public Integer deleteAllByIds(String entityName, JSONArray ids) {
        return deleteAllByIds(entityName, ids, null);
    }

    /**
     * 根据ID集合查询所有数据
     *
     * @param columns        列字段
     * @param entityName     实体名
     * @param ids            主键值集合
     * @param dataSourceName 数据源名称
     * @return 结果集
     */
    public JSONArray findAllByIds(String columns, String entityName, JSONArray ids, String dataSourceName) {
        MetaData map = metaDataService.getMetaMapItemByKey(entityName);
        String idStr = CommonTools.union(ids);
        String sql = "SELECT " + columns + " FROM " + map.getTableName() + " WHERE " + map.getIdColName() + " IN (" + idStr + " )";
        return sqlAction.query("EntityService@findAllByIds", sql, new QueryParams.Builder().dataSource(dataSourceName).build());
    }

    public JSONArray findAllByIds(String columns, String entityName, JSONArray ids) {
        return findAllByIds(columns, entityName, ids, null);
    }

    public JSONArray findAllByIds(String entityName, JSONArray ids) {
        return findAllByIds("*", entityName, ids);
    }

    /**
     * 查询实体的总数量
     *
     * @param entityName     实体名
     * @param dataSourceName 数据源名称
     * @return 总数
     */
    public Long getCount(String entityName, String dataSourceName) {
        MetaData map = metaDataService.getMetaMapItemByKey(entityName);
        String sql = "SELECT COUNT(0) count FROM " + map.getTableName();
        JSONArray result = sqlAction.query("EntityService@getCount", sql, new QueryParams.Builder().dataSource(dataSourceName).build());
        return !result.isEmpty() ? result.getJSONObject(0).getLong("count") : 0;
    }

    public Long getCount(String entityName) {
        return getCount(entityName, null);
    }

    /**
     * 查询所有数据
     *
     * @param columns        列字段
     * @param entityName     实体名
     * @param dataSourceName 数据源名称
     * @return 结果集
     */
    public JSONArray findAll(String columns, String entityName, String dataSourceName) {
        MetaData map = metaDataService.getMetaMapItemByKey(entityName);
        String sql = "SELECT " + columns + " FROM " + map.getTableName();
        return sqlAction.query("EntityService@findAll", sql, new QueryParams.Builder().dataSource(dataSourceName).build());
    }

    public JSONArray findAll(String columns, String entityName) {
        return findAll(columns, entityName, null);
    }

    public JSONArray findAll(String entityName) {
        return findAll("*", entityName);
    }

    /**
     * 根据指定键保存实体
     *
     * @param entityName     实体名
     * @param row            数据实体
     * @param keyArray       指定键集合
     * @param isCover        是否覆盖已有数据
     * @param dataSourceName 数据源名称
     * @return 结果
     */
    public JSONObject saveByKey(String entityName, JSONObject row, JSONArray keyArray, Boolean isCover, String dataSourceName) {
        // 获取条件表达式
        Condition condition = Condition.build();
        int keyArrayLength = keyArray.length();
        for (int i = 0; i < keyArrayLength; i++) {
            String keyName = keyArray.getString(i);
            condition = condition.eq(keyName, row.get(keyName));
            if (i < keyArrayLength - 1) {
                condition.and();
            }
        }
        // 查询已有数据
        MetaData map = metaDataService.getMetaMapItemByKey(entityName);
        String idName = map.getIdColName();
        JSONArray entityDataList = getByCondition(idName, entityName, condition);
        int entityDataLength = entityDataList.length();
        if (entityDataLength == 1) {
            if (!isCover) {
                throw new ServiceException("实体[" + entityName + "]对应的数据已存在", HttpStatus.BAD_REQUEST);
            } else {
                // 实体更新
                row.put(idName, entityDataList.getJSONObject(0).get(idName));
            }
        } else if (entityDataLength > 1) {
            throw new ServiceException("实体[" + entityName + "]对应的数据条数有误，预期：1，实际：" + entityDataLength, HttpStatus.BAD_REQUEST);
        } else {
            // 实体保存
            row.remove(idName);
        }
        return partialSave(entityName, row, dataSourceName);
    }

    public JSONObject saveByKey(String entityName, JSONObject row, JSONArray keyArray, Boolean isCover) {
        return saveByKey(entityName, row, keyArray, isCover, null);
    }

    public JSONObject saveByKey(String entityName, JSONObject row, JSONArray keyArray) {
        return saveByKey(entityName, row, keyArray, true);
    }

    /**
     * 根据提供的外键名和ID删除指定实体的已有数据，并保存提供的实体数据
     *
     * @param entityName     实体名
     * @param rowArray       数据实体集合
     * @param foreignName    外键名
     * @param foreignId      外键ID
     * @param dataSourceName 数据源名称
     * @return 结果
     */
    public JSONArray overrideSave(String entityName, JSONArray rowArray, String foreignName, Object foreignId, String dataSourceName) {
        sqlAction.exec("EntityService@doDelete",
                "DELETE FROM " + entityName + " WHERE" + foreignName + " = " + foreignId, dataSourceName);
        return partialSave(entityName, rowArray, dataSourceName);
    }

    public JSONArray overrideSave(String entityName, JSONArray rowArray, String foreignName, Object foreignId) {
        return overrideSave(entityName, rowArray, foreignName, foreignId, null);
    }

    /**
     * 获取实体主键名
     *
     * @param entityName 实体
     * @return 主键名
     */
    public String getTablePrimaryKey(String entityName) {
        return metaDataService.getMetaMapItemByKey(entityName).getIdColName();
    }

    /**
     * 实体保存
     *
     * @param entityName     实体名
     * @param row            数据实体
     * @param dataSourceName 数据源名称
     * @return 结果
     */
    public JSONObject partialSave(String entityName, JSONObject row, String dataSourceName) {
        return partialSave(entityName, new JSONArray().put(row), dataSourceName).getJSONObject(0);
    }

    public JSONObject partialSave(String entityName, JSONObject row) {
        return partialSave(entityName, row, null);
    }

    /**
     * 实体批量保存
     *
     * @param entityName     实体名
     * @param rowArray       数据实体集合
     * @param dataSourceName 数据源名称
     * @return 结果
     */
    public JSONArray partialSave(String entityName, JSONArray rowArray, String dataSourceName) {
        DbType dbType = DynamicDataSource.getDbType();
        JSONArray result;
        if (dbType == DbType.clickhouse) {
            result = new JSONArray();
            clickhouseService.save(entityName, rowArray);
        } else {
            MetaData md;
            // lift possible child entity
            if (MetaDataService.getEntityLiftMap().containsKey(entityName)) {
                String pEntityName = MetaDataService.getEntityLiftValueByKey(entityName);
                md = metaDataService.getMetaMapItemByKey(pEntityName).copy();
                metaDataService.enchanceMetaData(md, entityName);
            } else {
                md = metaDataService.getMetaMapItemByKey(entityName);
            }

            List<Serializable> aid = partialSave(rowArray, md, null, null, dataSourceName);
            result = new JSONArray(aid.size());
            aid.forEach(item -> {
                JSONObject obj = new JSONObject();
                obj.put(md.getIdColName(), item.toString());
                result.put(obj);
            });
        }
        return result;
    }

    public JSONArray partialSave(String entityName, JSONArray rowArray) {
        return partialSave(entityName, rowArray, null);
    }

    private void partialSave(JSONObject row, MetaData md, Object pIdValue, String pEntityName, String dataSourceName) {
        JSONArray array = new JSONArray(1);
        array.put(row);
        partialSave(array, md, pIdValue, pEntityName, dataSourceName);
    }

    private List<Serializable> partialSave(JSONArray rowArray, MetaData md, Object pIdValue, String pEntityName, String dataSourceName) {
        List<Serializable> result = new ArrayList<>(rowArray.length());
        rowArray.forEach(item -> {
            JSONObject row = (JSONObject) item;
            Object idValue = null;
            String idName = md.getIdColName();
            ColumnTypeEnum idType = md.getIdType();
            if (row.has(idName)) {
                idValue = row.get(idName);
            }

            Map<String, Pair> columns = md.getColumns();
            Map<String, Pair> links = md.getLinks();

            boolean isInsert = false;
            int affectedRows;

            if (idValue == null || idValue == JSONObject.NULL) {
                affectedRows = doInsert(row, md, pIdValue, pEntityName, idName, idType, columns, dataSourceName);
            } else {
                if (!(md.getIdGenerator() == IDTypeEnum.ID_ASSIGNED)) {
                    affectedRows = doUpdate(row, md, pIdValue, pEntityName, idValue, idType, columns, dataSourceName);
                } else {
                    if (hasRow(md.getTableName(), md.getIdColName(), idValue, idType, dataSourceName)) {
                        affectedRows = doUpdate(row, md, pIdValue, pEntityName, idValue, idType, columns, dataSourceName);
                    } else {
                        isInsert = true;
                        affectedRows = doInsert(row, md, pIdValue, pEntityName, idName, idType, columns, dataSourceName);
                    }
                }
            }

            //stale object
            if (affectedRows == 0) {
                throw new ServiceException("修改数据失败，返回的影响行数为:0", HttpStatus.BAD_REQUEST);
            }

            //possible modified idValue
            idValue = row.get(idName);


            //remove idName in row
            if (isInsert) {
                row.remove(idName);
            }

            //save or update one-to-many
            Iterator<String> itr = row.keys();
            while (itr.hasNext()) {
                String attr = itr.next();
                if (links.containsKey(attr) && row.has(attr)) {
                    String col = links.get(attr).col;
                    MetaData cmd = metaDataService.getMetaMapItemByKey(col);
                    String cEntityName = md.getEntityName();
                    //如果为空，如果是OneToOne，删除子
                    if (row.isNull(attr)) {
                        if (col.equals(attr))
                        //如果是一对一，子没传值，则删除子。
                        {
                            delete(cmd.getTableName(), row.get(idName));
                        }
                    } else {
                        doSave(row, idValue, attr, cmd, cEntityName, dataSourceName);
                    }
                }
            }
            result.add((Serializable) idValue);
        });
        return result;
    }

    private void doSave(JSONObject row, Object idValue, String attr, MetaData cmd, String pEntityName, String dataSourceName) {
        Object obj = row.get(attr);
        if (obj instanceof JSONArray) {
            JSONArray rows = row.getJSONArray(attr);
            for (int i = 0; i < rows.length(); i++) {
                partialSave(rows.getJSONObject(i), cmd, idValue, pEntityName, dataSourceName);
            }
        } else {
            partialSave((JSONObject) obj, cmd, idValue, pEntityName, dataSourceName);
        }
    }

    /**
     * hasRow
     */
    private boolean hasRow(String tableName, String idColName, Object idValue, ColumnTypeEnum idType, String dataSourceName) {
        return !sqlAction.query(
                "EntityService@hasRow",
                "select 1 from " + tableName + " where " + idColName + "=" + normalizeValue(idValue, idType),
                new QueryParams.Builder().dataSource(dataSourceName).build()
        ).isEmpty();
    }

    private int doUpdate(JSONObject row, MetaData md,
                         Object pIdValue, String pEntityName,
                         Object idValue, ColumnTypeEnum idType, Map<String, Pair> columns, String dataSourceName) {
        // 如果只有一个id号，不用更新，返回-1，避免认为更新失败
        if (row.keySet().size() == 1) {
            return -1;
        }
        Object verValue = null;
        String verName = md.getVerName();
        if (verName != null && row.has(verName)) {
            verValue = row.get(verName);
        }

        StringBuilder sb = new StringBuilder();

        String foreignkey = null;

        //update possible foreign key column value
        if (pEntityName != null) {
            Map<String, Pair> associations = md.getAssociations();
            if (associations != null && associations.containsKey(pEntityName)) {
                Pair pair = associations.get(pEntityName);
                foreignkey = pair.col;
                sb.append(" ").append(foreignkey).append("=").append(normalizeValue(pIdValue, pair.type)).append(", ");
            }
        }

        if (verName != null) {
            ColumnTypeEnum verType = md.getVerType();
            sb.append(" ").append(verName).append("=").append(normalizeVer(verValue, verType)).append(", ");
        }

        //enumerate every property in the row
        Iterator<String> itr = row.keys();
        while (itr.hasNext()) {
            String attr = itr.next();
            //average column, maybe foreign key column, but foreign key column is ignored since it is handled already
            if (columns.containsKey(attr) && !attr.equals(foreignkey) && !attr.equals(verName)) {
                Pair pair = columns.get(attr);
                sb.append(" ").append(pair.col).append("=").append(row.isNull(attr) ? "null, " : normalizeValue(row.get(attr), pair.type) + ", ");
            }
        }

        Integer affectedRow = null;
        // 保存主表
        if (!sb.isEmpty()) {
            sb.delete(sb.length() - 2, sb.length());
            sb.insert(0, "UPDATE " + md.getTableName() + " SET");
            sb.append(" WHERE ").append(md.getIdColName()).append(" = ").append(normalizeValue(idValue, idType));

            if (md.getVerName() != null) {
                ColumnTypeEnum verType = md.getVerType();
                sb.append(" AND ").append(md.getVerColName()).append(" = ").append(normalizeValue(verValue, verType));
            }
            affectedRow = sqlAction.exec("EntityService@doUpdate", sb.toString(), dataSourceName);
        }

        // 保存子表
        if (md.hasSubclasses()) {
            Map<String, SubClasses> subclasses = md.getSubclasses();
            for (String sentity : subclasses.keySet()) {
                SubClasses p = subclasses.get(sentity);
                //if discriminator is not found
                if (row.isNull(p.col())) {
                    break;
                }
                //if discriminator matches
                String dv = row.getString(p.col());
                String pTypeStr = p.type();
                if (pTypeStr.equals(dv) || pTypeStr.contains(dv + ",") || pTypeStr.contains("," + dv)) {
                    StringBuilder buf = new StringBuilder();
                    //update row
                    MetaData smd = metaDataService.getMetaMapItemByKey(sentity);
                    String keyAttr = smd.getIdColName();
                    Map<String, Pair> attrs = smd.getColumns();
                    for (String attr : attrs.keySet()) {
                        //skip key attr
                        if (attr.equals(keyAttr)) {
                            continue;
                        }
                        if (row.has(attr)) {
                            Pair a = attrs.get(attr);
                            buf.append(a.col).append("=").append(normalizeValue(row.isNull(attr) ? null : row.get(attr), a.type)).append(", ");
                        }
                    }

                    buf.delete(buf.length() - 2, buf.length());
                    buf.insert(0, "UPDATE " + smd.getTableName() + " SET ");
                    buf.append(" WHERE ").append(smd.getIdColName()).append(" = ").append(normalizeValue(idValue, idType));

                    sqlAction.exec("EntityService@doUpdate-child", buf.toString(), dataSourceName);

                    // 此处赋值用于解决实体保存时仅传入主表 ID 和子表集合的场景，即无需对主表进行操作，因此不会产生影响行数的情况
                    if (affectedRow == null) {
                        affectedRow = 1;
                    }

                    // 继续检查子类的关联，否则就断了链了
                    Map<String, Pair> links = smd.getLinks();
                    Map<String, Pair> pLinks = md.getLinks();
                    Iterator<String> itrLink = row.keys();
                    while (itrLink.hasNext()) {
                        String attr = itrLink.next();
                        if (!pLinks.containsKey(attr) && links.containsKey(attr) && row.has(attr)) {
                            String col = links.get(attr).col;
                            MetaData cmd = metaDataService.getMetaMapItemByKey(col);
                            String cEntityName = smd.getEntityName();
                            //如果为空，如果是OneToOne，删除子
                            if (row.isNull(attr)) {
                                if (col.equals(attr))
                                //如果是一对一，子没传值，则删除子。
                                {
                                    delete(cmd.getTableName(), idValue);
                                }
                            } else {
                                doSave(row, idValue, attr, cmd, cEntityName, dataSourceName);
                            }
                        }
                    }
                    break;
                }
            }
        }

        handleInverses(row, md, dataSourceName);

        return affectedRow == null ? 0 : affectedRow;
    }

    private void handleInverses(JSONObject row, MetaData md, String dataSourceName) {
        //inverse relation
        Map<String, String> inverses = md.getInverses();
        for (String inverse : inverses.keySet()) {
            if (row.isNull(inverse)) {
                continue;
            }
            Object arow = row.get(inverse);
            if (arow instanceof JSONObject) {
                MetaData amd = metaDataService.getMetaMapItemByKey(inverses.get(inverse));
                partialSave((JSONObject) arow, amd, null, null, dataSourceName);
            }
        }
    }

    private int doInsert(JSONObject row, MetaData md,
                         Object pIdValue, String pEntityName,
                         String idName, ColumnTypeEnum idType, Map<String, Pair> columns, String dataSourceName) {
        Object verValue = null;
        String verName = md.getVerName();
        if (verName != null && row.has(verName)) {
            verValue = row.get(verName);
        }

        String idColName = md.getIdColName();

        StringBuilder sbCols = new StringBuilder();
        StringBuilder sbValues = new StringBuilder();

        String foreignkey = null;
        //update possible foreign key column value
        if (pEntityName != null) {
            Map<String, Pair> associations = md.getAssociations();
            if (associations != null) {
                //处理一对多的关系
                for (String ass : associations.keySet()) {
                    if (ass.equals(pEntityName) || hasParent(ass, pEntityName)) {
                        Pair pair = associations.get(ass);
                        foreignkey = pair.col;
                        if (sbCols.indexOf(foreignkey + ",") == -1) {
                            sbCols.append(foreignkey).append(", ");
                            sbValues.append(normalizeValue(pIdValue, pair.type)).append(", ");
                        }
                    }
                }
            }
        }

        if (verName != null) {
            sbCols.append(verName).append(", ");
            ColumnTypeEnum verType = md.getVerType();
            sbValues.append(normalizeVer(verValue, verType)).append(", ");
        }

        //enumerate every property in the row
        Iterator<String> itr = row.keys();
        while (itr.hasNext()) {
            String attr = itr.next();
            //average column, maybe foreign key column, but foreign key column is ignored since it is handled already
            if (columns.containsKey(attr) && !attr.equals(foreignkey) && !attr.equals(verName)) {
                Pair pair = columns.get(attr);
                sbCols.append(pair.col).append(", ");
                sbValues.append(row.isNull(attr) ? "null, " : normalizeValue(row.get(attr), pair.type) + ", ");
            }
        }


        //handle id problems
        IDTypeEnum idTypeEnum = md.getIdGenerator();
        switch (idTypeEnum) {
            case ID_GUID -> {
                sbCols.append(idColName).append(", ");
                String guid = UUID.randomUUID().toString().replace("-", "");
                sbValues.append("'").append(guid).append("', ");
                row.put(idName, guid);
            }
            case ID_SEQ -> {
                sbCols.append(idColName).append(", ");
                String aid = getLastSeqId(md.getSequence(), dataSourceName);
                sbValues.append(aid).append(", ");
                row.put(idName, aid);
            }
            case ID_ASSIGNED -> {
                sbCols.append(idColName).append(", ");
                sbValues.append(normalizeValue(row.get(idName), idType)).append(", ");
                row.put(idName, row.get(idName));
            }
            case ID_FOREIGNER -> {
                sbCols.append(idColName).append(", ");
                String aid = normalizeValue(pIdValue, idType);
                sbValues.append(aid).append(", ");
                row.put(idName, aid);
            }
            default -> {
            }
        }

        if (md.hasInverses()) {
            //handle inverse problems
            Map<String, String> inverses = md.getInverses();
            Map<String, Pair> inversesid = md.getInverseid();
            for (String inverse : inverses.keySet()) {
                if (row.isNull(inverse)) {
                    continue;
                }
                Object arow = row.get(inverse);
                //LOGGER.debug("发现多对一属性赋值:");
                if (!(arow instanceof JSONObject json)) {
                    Pair pair = inversesid.get(inverse);
                    sbCols.append(pair.col).append(", ");
                    sbValues.append(normalizeValue(arow, pair.type)).append(", ");
                    //LOGGER.debug("发现多对一属性赋值:属性:" + pair.col + "，值：" + arow);
                } else {
                    // 多对一对象关系，属性名与列名必须一致
                    sbCols.append(inverse).append(", ");
                    // 值从给定的对象中取
                    Pair pair = inversesid.get(inverse);
                    arow = json.get(pair.col);
                    sbValues.append(normalizeValue(arow, pair.type)).append(", ");
                    //LOGGER.debug("发现多对一属性赋值:属性:" + pair.col + "，值：" + arow);
                }
            }
        }

        if (sbCols.isEmpty()) {
            throw new ServiceException("传入的实体内容无效：" + row + "，相关实体名：" + md.getTableName(), HttpStatus.BAD_REQUEST);
        }
        sbCols.delete(sbCols.length() - 2, sbCols.length());
        sbCols.insert(0, "INSERT INTO " + md.getTableName() + " (");
        sbValues.delete(sbValues.length() - 2, sbValues.length());
        sbCols.append(") VALUES (");
        sbCols.append(sbValues);
        sbCols.append(")");

        int affectedRows;

        if (idTypeEnum == IDTypeEnum.ID_AUTO) {
            Tuple tuple;
            try {
                tuple = rawJdbcUpdate(sbCols.toString(), dataSourceName);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            row.put(idName, tuple.get(1).toString());
            affectedRows = tuple.get(0);
        } else {
            affectedRows = sqlAction.exec("EntityService@doInsert", sbCols.toString(), dataSourceName);
        }

        //insert possible sub-entity
        if (md.hasSubclasses()) {
            Map<String, SubClasses> subclasses = md.getSubclasses();
            for (String sentity : subclasses.keySet()) {
                SubClasses p = subclasses.get(sentity);
                //if discriminator is not found
                if (row.isNull(p.col())) {
                    break;
                }
                String dv = row.getString(p.col());
                String pTypeStr = p.type();
                if (pTypeStr.equals(dv) || pTypeStr.contains(dv + ",") || pTypeStr.contains("," + dv)) {
                    StringBuilder fields = new StringBuilder();
                    StringBuilder vals = new StringBuilder();

                    //update row
                    MetaData smd = metaDataService.getMetaMapItemByKey(sentity);
                    String keyCol = smd.getIdColName();

                    //handle it
                    fields.append(keyCol).append(", ");
                    vals.append(normalizeValue(row.get(idName), idType)).append(", ");

                    Map<String, Pair> attrs = smd.getColumns();
                    for (String attr : attrs.keySet()) {
                        if (row.has(attr)) {
                            Pair a = attrs.get(attr);
                            // 外键不当属性加入
                            if (!(a.col.equals(foreignkey))) {
                                fields.append(a.col).append(", ");
                                vals.append(normalizeValue(row.isNull(attr) ? null : row.get(attr), a.type)).append(", ");
                            }
                        }
                    }

                    //处理反向查id的情况
                    if (smd.hasInverseIds()) {
                        Map<String, Pair> lookup = smd.getInverseid();
                        for (String property : lookup.keySet()) {
                            if (row.has(property) && !row.isNull(property)) {
                                Object obj = row.get(property);
                                Pair pair = lookup.get(property);
                                if (obj instanceof JSONObject record) {
                                    if (record.has(pair.col) && !record.isNull(pair.col)) {
                                        fields.append(pair.col).append(", ");
                                        vals.append(normalizeValue(record.get(pair.col), pair.type)).append(", ");
                                    }
                                } else {
                                    fields.append(pair.col).append(", ");
                                    vals.append(normalizeValue(row.get(property), pair.type)).append(", ");
                                }
                            }
                        }
                    }

                    //处理子类的外键
                    Map<String, Pair> associations = md.getAssociations();
                    if (associations != null) {
                        //处理一对多的关系
                        for (String ass : associations.keySet()) {
                            //如果子类的定义包括此列
                            Pair pair = associations.get(ass);
                            foreignkey = pair.col;
                            if (attrs.containsKey(foreignkey) && (pIdValue != null) && (fields.indexOf(foreignkey + ", ") == -1)) {
                                fields.append(foreignkey).append(", ");
                                vals.append(normalizeValue(pIdValue, pair.type)).append(", ");
                            }
                        }
                    }

                    fields.delete(fields.length() - 2, fields.length());
                    vals.delete(vals.length() - 2, vals.length());

                    fields.insert(0, "INSERT INTO " + smd.getTableName() + " (");
                    fields.append(") VALUES (");
                    fields.append(vals);
                    fields.append(")");

                    sqlAction.exec("EntityService@doInsert-child", fields.toString(), dataSourceName);

                    //继续检查子类的关联，否则就断了链了
                    Map<String, Pair> links = smd.getLinks();
                    Map<String, Pair> pLinks = md.getLinks();
                    Iterator<String> itrLink = row.keys();
                    while (itrLink.hasNext()) {
                        String attr = itrLink.next();
                        if (!pLinks.containsKey(attr) && links.containsKey(attr) && row.has(attr)) {
                            String col = links.get(attr).col;
                            MetaData cmd = metaDataService.getMetaMapItemByKey(col);
                            String cEntityName = smd.getEntityName();
                            //如果为空，如果是OneToOne，删除子
                            if (row.isNull(attr)) {
                                if (col.equals(attr))
                                //如果是一对一，子没传值，则删除子。
                                {
                                    delete(cmd.getTableName(), row.get(idName));
                                }
                            } else {
                                doSave(row, row.get(idName), attr, cmd, cEntityName, dataSourceName);
                            }
                        }
                    }
                    break;
                }
            }
        }

        handleInverses(row, md, dataSourceName);

        return affectedRows;
    }

    /**
     * 检查继承关系
     */
    private boolean hasParent(String entity, String pEntityName) {
        // 处理边界条件
        if (entity == null || pEntityName == null) {
            return false;
        }

        // 获取实体映射
        Map<String, String> entityLiftMap = MetaDataService.getEntityLiftMap();

        // 使用集合来跟踪已访问的实体，防止循环引用
        Set<String> visitedEntities = new HashSet<>();

        while (entityLiftMap.containsKey(entity)) {
            if (!visitedEntities.add(entity)) {
                // 检测到循环引用，退出循环
                return false;
            }

            String pEntity = entityLiftMap.get(entity);
            if (pEntity.equals(pEntityName)) {
                return true;
            }
            entity = pEntity;
        }

        return false;
    }


    private Tuple rawJdbcUpdate(String sql, String dataSourceName) {
        AtomicReference<Tuple> tuple = new AtomicReference<>();
        return DynamicDataSource.withDataSource(dataSourceName, () -> {
            sessionPool.getSession().doWork(connection -> {
                LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "执行原生SQL：{\n{}\n}", sql);
                PreparedStatement statement = connection.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
                ResultSet rs;
                int affectedRows = statement.executeUpdate();
                if (affectedRows != 0) {
                    rs = statement.getGeneratedKeys();
                    if (rs.next()) {
                        String idValue = String.valueOf(rs.getLong(1));
                        tuple.set(new Tuple(affectedRows, idValue));
                    } else {
                        throw new RuntimeException("得到插入记录id失败。");
                    }
                }
            });
            return tuple.get();
        });
    }

    private String getLastSeqId(String seq, String dataSourceName) {
        DbType dbType = DynamicDataSource.getDbType();
        String sql;
        if (dbType == DbType.postgresql) {
            sql = "SELECT nextval('" + seq + "') newid";
        } else {
            sql = "select " + seq + ".nextval newid from dual";
        }
        return sqlAction.query("EntityService@getLastSeqId", sql, new QueryParams.Builder().dataSource(dataSourceName).build())
                .getJSONObject(0).get("newid").toString();
    }

    private String normalizeVer(Object verValue, ColumnTypeEnum verType) {
        DbType dbType = DynamicDataSource.getDbType();
        if (verType == ColumnTypeEnum.COL_TIME) {
            if (dbType == DbType.oracle) {
                return "current_timestamp";
            } else {
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                return "'" + sdf.format(new Date()) + "'";
            }
        } else if (verValue == null) {
            return "1";
        } else {
            return String.valueOf(Integer.parseInt(String.valueOf(verValue)) + 1);
        }
    }

    /**
     * normalizeValue
     */
    private String normalizeValue(Object value, ColumnTypeEnum valType) {
        if (value == null) {
            return "null";
        }
        DbType dbType = DynamicDataSource.getDbType();
        switch (valType) {
            case COL_STRING -> {
                return "'" + value + "'";
            }
            case COL_INTEGER, COL_DOUBLE, COL_LONG, COL_DECIMAL -> {
                if ("".equals(value)) {
                    return "null";
                }
                return String.valueOf(value);
            }
            case COL_BOOLEAN -> {
                if (dbType == DbType.oracle) {
                    return "'" + ((Boolean) value ? "Y" : "N") + "'";
                } else {
                    //该if是为了解决在SQLServer下保存数据时可能会导致Byte转Boolean时失败
                    if (value instanceof Byte) {
                        return value.toString();
                    } else if (value instanceof Integer) {
                        return value.toString();
                    } else {
                        return (Boolean) value ? "1" : "0";
                    }
                }
            }
            case COL_DATE -> {
                if (dbType == DbType.oracle) {
                    return "TO_DATE(SUBSTR('" + value + "', 1, 10), 'YYYY-MM-DD')";
                } else {
                    return "'" + value + "'";
                }
            }
            case COL_TIME -> {
                if (dbType == DbType.oracle) {
                    return "TO_TIMESTAMP('" + value + "', 'YYYY-MM-DD HH24:MI:SS')";
                } else {
                    return "'" + value + "'";
                }
            }

            case COL_CLOB -> {
                if (dbType == DbType.oracle) {
                    StringBuilder sql = new StringBuilder();
                    String[] parts = StrUtil.split(value.toString(), 2000);
                    for (String part : parts) {
                        sql.append("TO_CLOB('").append(part).append("')||");
                    }
                    int length = sql.length();
                    sql.delete(length - 2, length);
                    return sql.toString();
                } else {
                    return "'" + value + "'";
                }
            }
            default -> throw new ServiceException("传递数据类型出错", HttpStatus.CONFIG_ERROR);
        }
    }

    /**
     * sqlserver大数据批量插入
     * <p>
     * 大数据批量插入有一些限制，使用必须满足以下条件
     * <ul>
     *     <li>数据表主键不自增，插入的数据集合中必须包含主键列</li>
     *     <li>大数据批量插入必须包含每个数据列，本方法会对null值列自动赋默认值</li>
     *     <li>hibernateMapping中的数据列必须与实际表结构完全一致（字段顺序，字段类型）</li>
     * </ul>
     *
     * @param entityName     实体名称
     * @param rowArray       数据集合
     * @param dataSourceName 数据源名称
     */
    public void sqlserverBulkInsert(String entityName, JSONArray rowArray, String dataSourceName) {
        MetaData map = metaDataService.getMetaMapItemByKey(entityName);
        String tableName = map.getTableName();
        String idColName = map.getIdColName();
        ColumnTypeEnum typeEnum = map.getIdType();
        Map<String, Pair> normalColumns = map.getColumns();
        Map<String, Pair> allColumns = new HashMap<>(normalColumns.size() + 1);
        allColumns.put(idColName, new Pair(idColName, typeEnum));
        allColumns.putAll(map.getColumns());
        DynamicDataSource.withDataSource(dataSourceName, () -> sessionPool.getSession().doWork(connection -> {
            PreparedStatement statement = connection.prepareStatement("SELECT * FROM " + tableName + " WHERE 1 = 0");
            try (ResultSet rs = statement.executeQuery();
                 CachedRowSet crs = RowSetProvider.newFactory().createCachedRowSet()) {
                crs.populate(rs);
                //既然是批量插入肯定是需要循环
                for (Object o : rowArray) {
                    //更新虚拟行的数据
                    JSONObject params = (JSONObject) o;
                    //移动指针到"插入行"，插入行是一个虚拟行
                    crs.moveToInsertRow();
                    for (Map.Entry<String, Pair> entry : allColumns.entrySet()) {
                        String key = entry.getKey();
                        Object value = params.opt(key);
                        Pair pair = entry.getValue();
                        ColumnTypeEnum columnTypeEnum = pair.type;
                        switch (columnTypeEnum) {
                            case COL_STRING -> crs.updateString(key, value == null ? "" : value.toString());
                            case COL_INTEGER ->
                                    crs.updateInt(key, Integer.parseInt(value == null ? "0" : value.toString()));
                            case COL_LONG ->
                                    crs.updateLong(key, Long.parseLong(value == null ? "0" : value.toString()));
                            case COL_DOUBLE ->
                                    crs.updateDouble(key, Double.parseDouble(value == null ? "0" : value.toString()));
                            case COL_TIME ->
                                    crs.updateTimestamp(key, new Timestamp((value == null ? DateTime.now() : DateUtil.parse(value.toString())).toSqlDate().getTime()));
                            case COL_DATE ->
                                    crs.updateDate(key, (value == null ? DateTime.now() : DateUtil.parse(value.toString())).toSqlDate());
                            case COL_BOOLEAN ->
                                    crs.updateBoolean(key, Boolean.parseBoolean(value == null ? "false" : value.toString()));
                            case COL_DECIMAL ->
                                    crs.updateBigDecimal(key, new BigDecimal(value == null ? "0" : value.toString()));
                        }
                    }
                    // 插入虚拟行
                    crs.insertRow();
                    //移动指针到当前行
                    crs.moveToCurrentRow();
                }
                //进行批量插入
                SQLServerBulkCopyOptions copyOptions = new SQLServerBulkCopyOptions();
                copyOptions.setKeepIdentity(true);
                copyOptions.setBatchSize(8000);
                copyOptions.setUseInternalTransaction(true);
                AfDataSourceConfig dataSource = DynamicDataSource.getWrapper().getConfig();
                String jdbcUrl = dataSource.getUrl();
                if (!jdbcUrl.endsWith(";")) {
                    jdbcUrl += ";";
                }
                jdbcUrl += "username=" + dataSource.getUsername()
                        + ";password=" + dataSource.getPassword() + ";";
                try (SQLServerBulkCopy bulkCopy = new SQLServerBulkCopy(jdbcUrl)) {
                    bulkCopy.setBulkCopyOptions(copyOptions);
                    bulkCopy.setDestinationTableName(tableName);
                    bulkCopy.writeToServer(crs);
                }
            }
        }));
    }

    public void sqlserverBulkInsert(String entityName, JSONArray rowArray) {
        sqlserverBulkInsert(entityName, rowArray, null);
    }

    /**
     * 远程调用删除实体
     *
     * @param serviceName 服务名
     * @param entityName  实体名
     * @param id          主键值
     * @return 执行结果
     */
    public JSONObject remoteDelete(String serviceName, String entityName, Object id) {
        RemoteEntityService remoteEntityService = dynamicFeignClientFactory.getFeignClient(RemoteEntityService.class, serviceName);
        R<Map<String, Object>> result = remoteEntityService.deleteById(entityName, String.valueOf(id), SecurityConstants.INNER);
        return result.parseResponseJson();
    }

    /**
     * 远程调用批量删除实体
     *
     * @param serviceName 服务名
     * @param entityName  实体名
     * @param ids         主键值集合
     * @return 执行结果
     */
    public JSONObject remoteDeleteAllByIds(String serviceName, String entityName, JSONArray ids) {
        JSONObject params = new JSONObject();
        params.put("ids", ids);
        RemoteEntityService remoteEntityService = dynamicFeignClientFactory.getFeignClient(RemoteEntityService.class, serviceName);
        R<Map<String, Object>> result = remoteEntityService.deleteAllByIds(entityName, params.toString(), SecurityConstants.INNER);
        return result.parseResponseJson();
    }

    /**
     * 远程调用根据ID集合查询所有数据
     *
     * @param serviceName 服务名
     * @param entityName  实体名
     * @param ids         主键值集合
     * @return 执行结果
     */
    public JSONObject remoteFindAllByIds(String serviceName, String columns, String entityName, JSONArray ids) {
        JSONObject params = new JSONObject();
        params.put("columns", columns);
        params.put("ids", ids);
        RemoteEntityService remoteEntityService = dynamicFeignClientFactory.getFeignClient(RemoteEntityService.class, serviceName);
        R<List<Object>> result = remoteEntityService.findAllByIds(entityName, params.toString(), SecurityConstants.INNER);
        return result.parseResponseJson();
    }

    public JSONObject remoteFindAllByIds(String serviceName, String entityName, JSONArray ids) {
        return remoteFindAllByIds(serviceName, null, entityName, ids);
    }

    /**
     * 远程调用查询实体的总数量
     *
     * @param serviceName 服务名
     * @param entityName  实体名
     * @return 执行结果
     */
    public JSONObject remoteGetCount(String serviceName, String entityName) {
        RemoteEntityService remoteEntityService = dynamicFeignClientFactory.getFeignClient(RemoteEntityService.class, serviceName);
        R<Map<String, Object>> result = remoteEntityService.getCount(entityName, SecurityConstants.INNER);
        return result.parseResponseJson();
    }

    /**
     * 远程调用查询所有数据
     *
     * @param serviceName 服务名
     * @param columns     列字段
     * @param entityName  实体名
     * @return 执行结果
     */
    public JSONObject remoteFindAll(String serviceName, String columns, String entityName) {
        JSONObject params = new JSONObject();
        params.put("columns", columns);
        RemoteEntityService remoteEntityService = dynamicFeignClientFactory.getFeignClient(RemoteEntityService.class, serviceName);
        R<List<Object>> result = remoteEntityService.findAll(entityName, params.toString(), SecurityConstants.INNER);
        return result.parseResponseJson();
    }

    public JSONObject remoteFindAll(String serviceName, String entityName) {
        return remoteFindAll(serviceName, "*", entityName);
    }

    /**
     * 远程调用根据ID查询数据
     *
     * @param serviceName 服务名
     * @param columns     列字段
     * @param entityName  实体名
     * @param id          主键值
     * @return 实体
     */
    public JSONObject remoteGetById(String serviceName, String columns, String entityName, Object id) {
        JSONObject params = new JSONObject();
        params.put("columns", columns);
        RemoteEntityService remoteEntityService = dynamicFeignClientFactory.getFeignClient(RemoteEntityService.class, serviceName);
        R<Map<String, Object>> result = remoteEntityService.getById(entityName, String.valueOf(id), params.toString(), SecurityConstants.INNER);
        return result.parseResponseJson();
    }

    public JSONObject remoteGetById(String serviceName, String entityName, Object id) {
        return remoteGetById(serviceName, "*", entityName, String.valueOf(id));
    }

    /**
     * 远程根据指定键保存实体
     *
     * @param entityName 实体名
     * @param row        数据实体
     * @param keyArray   指定键集合
     * @param isCover    是否覆盖已有数据
     * @return 结果
     */
    public JSONObject remoteSaveByKey(String serviceName, String entityName, JSONObject row, JSONArray keyArray, Boolean isCover) {
        JSONObject params = new JSONObject();
        params.put("entity", row);
        params.put("keyArray", keyArray);
        params.put("isCover", isCover);
        RemoteEntityService remoteEntityService = dynamicFeignClientFactory.getFeignClient(RemoteEntityService.class, serviceName);
        R<Map<String, Object>> result = remoteEntityService.saveByKey(entityName, params.toString(), SecurityConstants.INNER);
        return result.parseResponseJson();
    }

    public JSONObject remoteSaveByKey(String serviceName, String entityName, JSONObject row, JSONArray keyArray) {
        return remoteSaveByKey(serviceName, entityName, row, keyArray, true);
    }

    /**
     * 远程调用实体保存
     *
     * @param serviceName 服务名
     * @param entityName  实体名
     * @param row         数据实体
     * @return 结果
     */
    public JSONObject remotePartialSave(String serviceName, String entityName, JSONObject row) {
        RemoteEntityService remoteEntityService = dynamicFeignClientFactory.getFeignClient(RemoteEntityService.class, serviceName);
        R<Map<String, Object>> result = remoteEntityService.save(entityName, row.toString(), SecurityConstants.INNER);
        return result.parseResponseJson();
    }

    /**
     * 远程调用实体批量保存
     *
     * @param serviceName 服务名
     * @param entityName  实体名
     * @param rowArray    数据实体集合
     * @return 结果
     */
    public JSONObject remotePartialSave(String serviceName, String entityName, JSONArray rowArray) {
        RemoteEntityService remoteEntityService = dynamicFeignClientFactory.getFeignClient(RemoteEntityService.class, serviceName);
        R<List<Object>> result = remoteEntityService.saveBatch(entityName, rowArray.toString(), SecurityConstants.INNER);
        return result.parseResponseJson();
    }

    /**
     * 实体保存（支持分表策略）
     *
     * @param entityName 实体名
     * @param row 数据实体
     * @param shardingStrategy 分表策略
     * @param dataSourceName 数据源名称
     * @return 结果
     */
    public JSONObject partialSaveWithSharding(String entityName, JSONObject row, String shardingStrategy, String dataSourceName) {
        return partialSaveWithSharding(entityName, new JSONArray().put(row), shardingStrategy, dataSourceName).getJSONObject(0);
    }

    /**
     * 实体保存（支持分表策略）
     *
     * @param entityName 实体名
     * @param row 数据实体
     * @param shardingStrategy 分表策略
     * @return 结果
     */
    public JSONObject partialSaveWithSharding(String entityName, JSONObject row, String shardingStrategy) {
        return partialSaveWithSharding(entityName, row, shardingStrategy, null);
    }

    /**
     * 实体批量保存（支持分表策略）
     *
     * @param entityName 实体名
     * @param rowArray 数据实体集合
     * @param shardingStrategy 分表策略
     * @param dataSourceName 数据源名称
     * @return 结果
     */
    public JSONArray partialSaveWithSharding(String entityName, JSONArray rowArray, String shardingStrategy, String dataSourceName) {
        DbType dbType = DynamicDataSource.getDbType();
        JSONArray result;
        if (dbType == DbType.clickhouse) {
            result = new JSONArray();
            clickhouseService.save(entityName, rowArray);
        } else {
            MetaData md;
            // lift possible child entity
            if (MetaDataService.getEntityLiftMap().containsKey(entityName)) {
                String pEntityName = MetaDataService.getEntityLiftValueByKey(entityName);
                md = metaDataService.getMetaMapItemByKey(pEntityName).copy();
                metaDataService.enchanceMetaData(md, entityName);
            } else {
                md = metaDataService.getMetaMapItemByKey(entityName);
            }

            // 应用分表策略
            if (StrUtil.isNotBlank(shardingStrategy)) {
                String originalTableName = md.getTableName();
                String shardedTableName = getTableNameWithSharding(originalTableName, shardingStrategy);
                md.setTableName(shardedTableName);
                LOGGER.info("使用分表策略 [{}] 将表名 [{}] 转换为 [{}]", shardingStrategy, originalTableName, shardedTableName);
            }

            List<Serializable> aid = partialSave(rowArray, md, null, null, dataSourceName);
            result = new JSONArray(aid.size());
            aid.forEach(item -> {
                JSONObject obj = new JSONObject();
                obj.put(md.getIdColName(), item.toString());
                result.put(obj);
            });
        }
        return result;
    }

    /**
     * 实体批量保存（支持分表策略）
     *
     * @param entityName 实体名
     * @param rowArray 数据实体集合
     * @param shardingStrategy 分表策略
     * @return 结果
     */
    public JSONArray partialSaveWithSharding(String entityName, JSONArray rowArray, String shardingStrategy) {
        return partialSaveWithSharding(entityName, rowArray, shardingStrategy, null);
    }
}
