package com.af.v4.system.common.jpa.config;

import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.config.AfDataSourceConfig;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.common.datasource.wrapper.AfDataSourceWrapper;
import com.af.v4.system.common.jpa.action.MonitorInterceptor;
import com.af.v4.system.common.jpa.dialect.MyDialectResolver;
import com.af.v4.system.common.jpa.dynamic.DynamicSessionFactory;
import com.af.v4.system.common.jpa.dynamic.DynamicSessionFactoryImpl;
import com.af.v4.system.common.jpa.dynamic.DynamicTransactionManager;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.zaxxer.hikari.HikariDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.Resource;
import org.springframework.orm.hibernate5.LocalSessionFactoryBean;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Hibernate配置
 *
 * @author Mr.river
 */
@Configuration
@ConfigurationProperties(prefix = "spring.datasource")
public class HibernateConfig {

    private static final Logger LOGGER = LoggerFactory.getLogger(HibernateConfig.class);
    public final String currentSessionContextClass = "org.springframework.orm.hibernate5.SpringSessionContext";
    private final ApplicationContext context;
    private final MonitorInterceptor monitorInterceptor;
    private final ApplicationUtils applicationUtils;

    public HibernateConfig(ApplicationContext context) {
        this.context = context;
        this.applicationUtils = context.getBean(ApplicationUtils.class);
        this.monitorInterceptor = context.getBean(MonitorInterceptor.class);
        DataSourceProperties dataSourceProperties = context.getBean(DataSourceProperties.class);
        // 强制设置数据源，对旧版本的druid数据源配置兼容
        dataSourceProperties.setType(HikariDataSource.class);
    }

    @Bean
    public DynamicSessionFactory dynamicSessionFactory() {
        Map<String, AfDataSourceWrapper> dataSourceList = DynamicDataSource.getDataSourceMap();
        Map<Object, LocalSessionFactoryBean> sessionFactoryBeanMap = new HashMap<>(dataSourceList.size());
        dataSourceList.forEach((key, value) -> {
            LocalSessionFactoryBean sessionFactoryBean = new LocalSessionFactoryBean();
            sessionFactoryBean.setDataSource(value.getDataSource());
            sessionFactoryBean.setPackagesToScan("com.af.v4.entity." + key);
            sessionFactoryBean.setEntityInterceptor(monitorInterceptor);
            Properties properties = getProperties(value);
            sessionFactoryBean.setHibernateProperties(properties);
            // configLocations
            List<String> cfgLocations = value.getConfig().getCfgLocations();
            if (cfgLocations != null && !cfgLocations.isEmpty()) {
                Resource[] cfgLocationResources = cfgLocations.stream()
                        .map(context::getResource).toList()
                        .toArray(new Resource[cfgLocations.size()]);
                sessionFactoryBean.setConfigLocations(cfgLocationResources);
            }
            try {
                sessionFactoryBean.afterPropertiesSet();
            } catch (IOException e) {
                LOGGER.error("数据源连接失败！", e);
            }
            sessionFactoryBeanMap.put(key, sessionFactoryBean);
        });
        return new DynamicSessionFactoryImpl(sessionFactoryBeanMap, sessionFactoryBeanMap.get(DynamicDataSource.DEFAULT_DATASOURCE_NAME));
    }

    private Properties getProperties(AfDataSourceWrapper wrapper) {
        AfDataSourceConfig value = wrapper.getConfig();
        Properties properties = new Properties();
        EnvType envType = applicationUtils.getEnvType();
        if (envType == EnvType.DEV) {
            properties.setProperty("hibernate.hbm2ddl.auto", "update");
        } else {
            properties.setProperty("hibernate.hbm2ddl.auto", "none");
        }
        properties.setProperty("hibernate.show_sql", "false");
        properties.setProperty("hibernate.format_sql", "false");
        properties.setProperty("hibernate.current_session_context_class", currentSessionContextClass);
        // 方言解析器配置
        if (value.getDialect() != null) {
            properties.setProperty("hibernate.dialect", value.getDialect());
        } else {
            properties.setProperty("hibernate.dialect_resolvers", MyDialectResolver.class.getCanonicalName());
        }
        boolean isClickhouse = wrapper.getDbType() == DbType.clickhouse;
        if (isClickhouse) {
            properties.setProperty("hibernate.allow_update_outside_transaction", "true");
        }
        return properties;
    }

    @Bean
    public DynamicTransactionManager transactionManager() {
        DynamicTransactionManager transactionManager = new DynamicTransactionManager();
        // 注入sessionFactory
        transactionManager.setSessionFactory(dynamicSessionFactory());
        return transactionManager;
    }
}
