package com.af.v4.system.common.jpa.utils;

import cn.hutool.core.util.StrUtil;
import org.json.JSONArray;

import java.util.stream.Stream;

/**
 * 查询表达式生成
 */
public class Condition {
    private final StringBuilder conditionValue = new StringBuilder();

    private Condition() {

    }

    public static Condition build() {
        return new Condition();
    }

    private Condition appendCondition(QueryType queryType, String column, Object value) {
        if (queryType != QueryType.IS_NULL && queryType != QueryType.IS_NOT_NULL) {
            conditionValue.append(column)
                    .append(" ")
                    .append(queryType.value)
                    .append(" '")
                    .append(value)
                    .append("'");
        } else {
            conditionValue.append(column)
                    .append(" ")
                    .append(queryType.value);
        }
        return this;
    }

    public Condition and() {
        conditionValue.append(" AND ");
        return this;
    }

    public Condition or() {
        conditionValue.append(" OR ");
        return this;
    }

    public Condition eq(String column, Object value) {
        QueryType queryType;
        if (StrUtil.equalsIgnoreCase(value == null ? "NULL" : value.toString(), "NULL")) {
            queryType = QueryType.IS_NULL;
        } else {
            queryType = QueryType.EQUALS;
        }
        return appendCondition(queryType, column, value);
    }

    public Condition neq(String column, Object value) {
        QueryType queryType;
        if (StrUtil.equalsIgnoreCase(value == null ? "NULL" : value.toString(), "NULL")) {
            queryType = QueryType.IS_NOT_NULL;
        } else {
            queryType = QueryType.NOT_EQUALS;
        }
        return appendCondition(queryType, column, value);
    }

    public Condition gt(String column, Object value) {
        return appendCondition(QueryType.GREATER_THEN, column, value);
    }

    public Condition gtEquals(String column, Object value) {
        return appendCondition(QueryType.GREATER_EQUALS_THEN, column, value);
    }

    public Condition lt(String column, Object value) {
        return appendCondition(QueryType.LESS_THEN, column, value);
    }

    public Condition ltEquals(String column, Object value) {
        return appendCondition(QueryType.LESS_EQUALS_THEN, column, value);
    }

    public Condition between(String column, Object beginValue, Object endValue) {
        conditionValue.append(column)
                .append(" BETWEEN '")
                .append(beginValue)
                .append("' AND '")
                .append(endValue)
                .append("'");
        return this;
    }

    public Condition like(String column, Object value) {
        conditionValue.append(column)
                .append(" LIKE '%")
                .append(value)
                .append("%'");
        return this;
    }

    public Condition rightLike(String column, Object value) {
        conditionValue.append(column)
                .append(" LIKE '")
                .append(value)
                .append("%'");
        return this;
    }

    public Condition leftLike(String column, Object value) {
        conditionValue.append(column)
                .append(" LIKE '%")
                .append(value)
                .append("'");
        return this;
    }

    public Condition notIn(String column, String... values) {
        return notIn(column, new JSONArray(values));
    }

    public Condition notIn(String column, JSONArray values) {
        conditionValue.append(column).append(" NOT IN (");
        for (Object value : values) {
            conditionValue.append("'").append(value).append("',");
        }
        conditionValue.deleteCharAt(conditionValue.length() - 1).append(")");
        return this;
    }

    public Condition in(String column, String... values) {
        return in(column, new JSONArray(values));
    }

    public Condition in(String column, JSONArray values) {
        conditionValue.append(column).append(" IN (");
        for (Object value : values) {
            conditionValue.append("'").append(value).append("',");
        }
        conditionValue.deleteCharAt(conditionValue.length() - 1).append(")");
        return this;
    }

    public Condition orderBy(String column) {
        conditionValue.append(" ORDER BY ").append(column);
        return this;
    }

    public String getValue() {
        return conditionValue.toString();
    }

    public enum QueryType {
        EQUALS("="),
        NOT_EQUALS("!="),
        IS_NULL("IS NULL"),
        IS_NOT_NULL("IS NOT NULL"),
        GREATER_THEN(">"),
        GREATER_EQUALS_THEN(">="),
        LESS_THEN("<"),
        LESS_EQUALS_THEN("<=");

        private final String value;

        QueryType(String value) {
            this.value = value;
        }

        public static QueryType toType(String value) {
            return Stream.of(QueryType.values())
                    .filter(p -> p.value.equals(value))
                    .findAny()
                    .orElse(null);
        }

        public static boolean is(String value) {
            return toType(value) != null;
        }

        public String getValue() {
            return value;
        }
    }

    public enum WhereType {
        AND("AND"),
        OR("OR");

        private final String value;

        WhereType(String value) {
            this.value = value;
        }

        public static WhereType toType(String value) {
            return Stream.of(WhereType.values())
                    .filter(p -> p.value.equals(value))
                    .findAny()
                    .orElse(null);
        }

        public static boolean is(String value) {
            return toType(value) != null;
        }

        public String getValue() {
            return value;
        }
    }
}
