package com.af.v4.system.common.jpa.service;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.expression.Expression;
import com.af.v4.system.common.jpa.action.SqlAction;
import com.af.v4.system.common.jpa.utils.SQLRuntimeSupport;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.resource.mapper.SqlMapper;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Iterator;

@Service
@Transactional(rollbackFor = Exception.class)
public class SqlService {
    private static final Logger LOGGER = LoggerFactory.getLogger(SqlService.class);

    private final SqlMapper sqlMapper;

    private final SqlAction sqlAction;

    private final ApplicationUtils applicationUtils;

    public SqlService(SqlMapper sqlMapper, SqlAction sqlAction, ApplicationUtils applicationUtils) {
        this.sqlMapper = sqlMapper;
        this.sqlAction = sqlAction;
        this.applicationUtils = applicationUtils;
    }

    /**
     * 获取SQL文件的编译后SQL
     *
     * @param name   SQL文件映射名
     * @param params 参数
     * @return SQL语句
     */
    public String getSQL(String name, JSONObject params) {
        return this.getSQLByFile(name, params);
    }


    /**
     * 获取合计执行结果
     *
     * @param name   SQL文件映射名
     * @param params 参数
     * @return 结果集
     */
    public JSONObject queryTotal(String name, JSONObject params) {
        // 获取求和字段等内容
        Object sums = null;
        if (params.has("sums")) {
            sums = params.get("sums");
        }
        // 产生sql语句编译后的结果
        String sql = this.getSQLByFile(name, params);

        // 求和时，order by会导致sql错误，过滤掉order by部分。
        sql = filterOutOrderBy(sql, sums);
        JSONArray array = sqlAction.query(name, sql);
        return array.getJSONObject(0);
    }

    public JSONObject queryTotal(String name, String str) {
        return queryTotal(name, StringUtils.isEmpty(str) ? new JSONObject() : new JSONObject(str));
    }

    /**
     * 根据SQL文件执行查询
     *
     * @param name     SQL文件映射名
     * @param params   参数
     * @param pageNo   页码
     * @param pageSize 每页大小
     * @return 结果集
     */
    public JSONArray query(String name, JSONObject params, Integer pageNo, Integer pageSize) {
        // 产生SQL语句编译后的结果
        String sql = this.getSQLByFile(name, params);

        return sqlAction.query(name, sql, pageNo, pageSize);
    }

    public JSONArray query(String name, JSONObject params, Integer pageSize) {
        return query(name, params, 1, pageSize);
    }

    public JSONArray query(String name, JSONObject params) {
        return query(name, params, 1, 1000);
    }

    /**
     * 根据指定名称的SQL语句执行查询
     *
     * @param name SQL查询标识名
     * @param sql  SQL语句
     * @return 结果集
     * @apiNote 注意：不推荐依赖该方法作为SQL查询，因为该方法没有校验name参数的唯一性，这会给排查SQL问题的同事带来麻烦
     */
    public JSONArray querySQL(String name, String sql) {
        return querySQL(name, sql, 1, 1000);
    }

    public JSONArray querySQL(String name, String sql, Integer pageSize) {
        return querySQL(name, sql, 1, pageSize);
    }

    public JSONArray querySQL(String name, String sql, Integer pageNo, Integer pageSize) {
        return sqlAction.query(name, sql, pageNo, pageSize);
    }

    /**
     * 根据SQL文件执行增删改
     *
     * @param name   SQL文件映射名
     * @param params 参数
     * @return 执行结果
     */
    public Integer exec(String name, JSONObject params) {
        // 产生sql语句编译后的结果
        String sql = this.getSQLByFile(name, params);
        return execSQL(name, sql);
    }

    /**
     * 根据SQL语句执行增删改
     *
     * @param name SQL查询标识名
     * @param sql  SQL语句
     * @return 影响行数
     * @apiNote 注意：不推荐依赖该方法作为SQL查询，因为该方法没有校验name参数的唯一性，这会给排查SQL问题的同事带来麻烦
     */
    public Integer execSQL(String name, String sql) {
        return sqlAction.exec(name, sql);
    }

    /**
     * 从SQL文件中获取SQL语句
     */
    private String getSQLByFile(String sqlName, JSONObject params) {
        // 获取原始sql语句
        SqlMapper.SqlResource resource = sqlMapper.getResource(sqlName);
        String source = resource.getSource();
        // 附加用户注册的对象到SQL中
        params.put("ENV", applicationUtils.getValues());
        params.put("RESOURCE", resource.getJsonParams());
        return Expression.run(source, params.toMap()).toString();
    }

    /**
     * 过滤order by子句，产生求和结果
     */
    private String filterOutOrderBy(String source, Object sums) {
        int idx = source.toLowerCase().lastIndexOf("order by");
        if (idx == -1) {
            idx = source.length() - 1;
        }
        StringBuilder sql = new StringBuilder("select ");
        // 如果有求和部分，产生求和部分的语句
        if (sums != null) {
            if (sums instanceof JSONArray arraySums) {
                for (int i = 0; i < arraySums.length(); i++) {
                    String name = (String) arraySums.get(i);
                    sql.append("sum(").append(name).append(") ").append(name).append(", ");
                }
            } else if (sums instanceof JSONObject objSums) {
                Iterator<String> keys = objSums.keys();
                while (keys.hasNext()) {
                    String name = keys.next();
                    String value = objSums.getString(name);
                    sql.append("sum(").append(value).append(") ").append(name).append(", ");
                }
            }
        }
        if (idx != -1) {
            sql.append("count(*) n, 1 placeholder from ( ").append(source, 0, idx).append(") t_");
        }
        return sql.toString();
    }

    /**
     * 获取数据源类型名称
     *
     * @return 数据源类型
     * @see com.alibaba.druid.DbType
     */
    public String getDbTypeName() {
        return DynamicDataSource.getDbType().name();
    }

    /**
     * 解析符合数据源的SQL
     *
     * @return 符合
     */
    public String resolveSQL(JSONObject sqlList) {
        String dbType = getDbTypeName();
        String sqlStr = sqlList.optString(dbType, null);
        if (sqlStr == null) {
            throw new ServiceException("未找到符合数据源[" + dbType + "]的SQL");
        }
        return sqlStr;
    }

    public void openStandardTransformerSupport() {
        SQLRuntimeSupport.openStandardTransformer();
    }

    public void closeStandardTransformerSupport() {
        SQLRuntimeSupport.closeStandardTransformer();
    }

    public void disableLogPrint() {
        SQLRuntimeSupport.disableLogPrint();
    }

    public void enableLogPrint() {
        SQLRuntimeSupport.enableLogPrint();
    }
}
