package com.af.v4.system.common.jpa.action;

import com.af.v4.system.common.core.constant.CodeNormsConstants;
import com.af.v4.system.common.core.exception.CheckedException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.jpa.session.SessionPool;
import com.af.v4.system.common.jpa.transformer.AliasTransformer;
import com.af.v4.system.common.jpa.transformer.StandardAliasTransformer;
import com.af.v4.system.common.jpa.utils.SQLRuntimeSupport;
import org.hibernate.Session;
import org.hibernate.exception.SQLGrammarException;
import org.hibernate.query.NativeQuery;
import org.hibernate.transform.ResultTransformer;
import org.json.JSONArray;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * SQL执行器
 *
 * @author Mr.river
 */
@Component
public class SqlAction {

    private static final Logger LOGGER = LoggerFactory.getLogger(SqlAction.class);
    private final SessionPool sessionPool;

    public SqlAction(SessionPool sessionPool) {
        this.sessionPool = sessionPool;
    }

    public JSONArray queryAll(String name, String sql) {
        return query(name, sql, null, null);
    }

    public JSONArray query(String name, String sql) {
        return query(name, sql, 1, 1000);
    }

    public JSONArray query(String name, String sql, Integer pageNo, Integer pageSize) {
        return (JSONArray) sqlMonitor(() -> {
            ResultTransformer resultTransformer;
            boolean userStandardTrans = SQLRuntimeSupport.getStandardTransformerValue();
            if (userStandardTrans) {
                resultTransformer = StandardAliasTransformer.INSTANCE;
            } else {
                resultTransformer = AliasTransformer.INSTANCE;
            }
            InnerSQLCall sqlCall = new InnerSQLCall(sql, pageNo, pageSize, resultTransformer);
            JSONArray array = sqlCall.doInHibernate(sessionPool.getSession());
            if (array != null) {
                writeSqlResult(name, array);
            }
            return array;
        }, name);
    }

    public Integer exec(String name, String sql) {
        return (int) sqlMonitor(() -> {
            Session session = sessionPool.getSession();
            NativeQuery<?> queryObject = session.createNativeQuery(sql);
            return queryObject.executeUpdate();
        }, name);
    }

    private Object sqlMonitor(Process process, String name) {
        if (StringUtils.isEmpty(name)) {
            throw new CheckedException(CodeNormsConstants.E_01);
        }
        LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "执行SQL[{}]", name);
        long begin = System.currentTimeMillis();
        try {
            return process.apply();
        } finally {
            long end = System.currentTimeMillis();
            LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "SQL[{}]耗时：{}ms", name, end - begin);
        }
    }

    /**
     * 打印SQL执行结果
     *
     * @param name  SQL名称
     * @param array 执行结果
     */
    private void writeSqlResult(String name, JSONArray array) {
        String result = array.toString();
        int length = result.length();
        if (length > 2000) {
            result = result.substring(0, 2000) + "...后续还有" + (length - 2000) + "个字符";
        }
        LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "SQL[{}]查询结果: \n{}", name, result);
    }

    @FunctionalInterface
    interface Process {
        Object apply();
    }

    private record InnerSQLCall(String sql, Integer page, Integer rows,
                                ResultTransformer transformer) implements HibernateCallback<JSONArray> {

        @Override
        public JSONArray doInHibernate(Session session) {
            NativeQuery<?> q = session.createNativeQuery(sql);
            q.setResultTransformer(transformer);
            try {
                List<?> list;
                if (page == null || rows == null || rows <= 0) {
                    list = q.list();
                } else {
                    int realPage = page <= 0 ? 0 : page - 1;
                    list = q.setFirstResult(realPage * rows).setMaxResults(rows).list();
                }
                if (transformer instanceof StandardAliasTransformer) {
                    return new JSONArray().putAll(list);
                } else {
                    return new JSONArray(list);
                }
            } catch (SQLGrammarException ex) {
                // 把sql语句添加到异常信息中
                String msg = "sql:\n" + sql + "\n" + ex.getMessage();
                throw new SQLGrammarException(msg, ex.getSQLException());
            }
        }
    }
}
