package com.af.v4.system.common.job;

import akka.protobufv3.internal.ServiceException;
import jakarta.annotation.PostConstruct;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import tech.powerjob.client.PowerJobClient;
import tech.powerjob.common.enums.ExecuteType;
import tech.powerjob.common.enums.ProcessorType;
import tech.powerjob.common.enums.TimeExpressionType;
import tech.powerjob.common.exception.PowerJobException;
import tech.powerjob.common.request.http.SaveJobInfoRequest;
import tech.powerjob.common.response.JobInfoDTO;
import tech.powerjob.common.response.ResultDTO;

@Component
public class PowerJobClientStarter {

    private static final Logger LOGGER = LoggerFactory.getLogger(PowerJobClientStarter.class);

    @Value("${powerjob.worker.app-name}")
    private String appName;
    @Value("${powerjob.worker.server-address}")
    private String serverAddress;
    @Value("${powerjob.worker.password}")
    private String password;
    private PowerJobClient powerJobClient;

    @PostConstruct
    public void init() {
        try {
            powerJobClient = new PowerJobClient(serverAddress, appName, password);
        } catch (PowerJobException e) {
            LOGGER.warn("PowerJob服务端未启动，客户端OpenAPI无法使用");
        }
    }

    public PowerJobClient getPowerJobClient() {
        return powerJobClient;
    }

    /**
     * 创建一个简单单机任务
     *
     * @param jobName            任务名称
     * @param jobDescription     任务描述
     * @param jobParams          任务参数
     * @param timeExpressionType 调度方式
     * @param timeExpression     调度值（CRON,固定值）
     * @param jobClass           任务实例
     * @return 是否创建成功
     */
    public ResultDTO<Long> createSimpleJob(String jobName, String jobDescription, JSONObject jobParams,
                                           TimeExpressionType timeExpressionType, String timeExpression, Class<?> jobClass) {
        SaveJobInfoRequest saveJobInfoRequest = new SaveJobInfoRequest();
        saveJobInfoRequest.setJobName(jobName);
        saveJobInfoRequest.setJobDescription(jobDescription);
        saveJobInfoRequest.setJobParams(jobParams.toString());
        saveJobInfoRequest.setTimeExpressionType(timeExpressionType);
        saveJobInfoRequest.setTimeExpression(timeExpression);
        saveJobInfoRequest.setExecuteType(ExecuteType.STANDALONE);
        saveJobInfoRequest.setProcessorType(ProcessorType.BUILT_IN);
        saveJobInfoRequest.setProcessorInfo(jobClass.getCanonicalName());
        saveJobInfoRequest.setMaxInstanceNum(1);
        return powerJobClient.saveJob(saveJobInfoRequest);
    }

    public boolean hasJob(Long powerJobId) {
        ResultDTO<JobInfoDTO> resultDTO = powerJobClient.fetchJob(powerJobId);
        return resultDTO.isSuccess() && resultDTO.getData() != null && resultDTO.getData().getStatus() != 99;
    }

    public boolean deleteJob(Long powerJobId) {
        if (!hasJob(powerJobId)) {
            LOGGER.error("powerJob [{}] 不存在", powerJobId);
            return false;
        }
        try {
            ResultDTO<Void> resultDTO = powerJobClient.deleteJob(powerJobId);
            return resultDTO.isSuccess();
        } catch (Exception e) {
            LOGGER.error("删除任务失败: {}", e.getMessage());
            return false;
        }
    }
}
