package com.af.v4.system.common.job.processer;

import com.af.v4.system.common.log.service.LogResolveService;
import com.af.v4.system.common.logic.service.LogicService;
import com.af.v4.system.common.plugins.concurrent.LogicBatchTaskProcessor;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import tech.powerjob.worker.core.processor.ProcessResult;

import java.util.concurrent.TimeUnit;

/**
 * 分页分批执行定时调度业务的简单封装
 *
 * @author Mr.river
 */
@Component
public abstract class GlazeBatchPagingTimer extends GlazeBasicTimer {

    private final Logger LOGGER = LoggerFactory.getLogger(GlazeBatchPagingTimer.class);

    protected final LogicService logicService;

    public GlazeBatchPagingTimer(LogResolveService logResolveService, LogicService logicService) {
        super(logResolveService);
        this.logicService = logicService;
    }

    @Override
    public ProcessResult exec(JSONObject jobParams) throws InterruptedException {
        JSONArray batchResult = new JSONArray();
        LogicBatchTaskProcessor processor = new LogicBatchTaskProcessor();
        int pageNo = 1;
        int failCount = 0;
        int maxFail = 10;

        while (!Thread.currentThread().isInterrupted()) {
            JSONArray resultArray;
            try {
                resultArray = queryBatchData(jobParams, pageNo);
                failCount = 0;
            } catch (Exception e) {
                LOGGER.error("查询数据失败", e);
                failCount++;
                if (failCount >= maxFail) break;
                try {
                    TimeUnit.SECONDS.sleep(60);
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw ie;
                }
                continue;
            }
            if (resultArray == null || resultArray.isEmpty()) break;
            LOGGER.info("======开始执行业务，第{}批次，本批次共{}条=====", pageNo, resultArray.length());
            JSONObject result = processor.process(resultArray, (JSONObject item) -> handle(jobParams, item));
            batchResult.put(result);
            pageNo++;
            LOGGER.info("======第{}批次执行完成=====", pageNo);
        }
        LOGGER.info("分页批量任务处理完成，结果：{}", batchResult);
        return new ProcessResult(true, batchResult.toString());
    }

    /**
     * 查询一批数据
     * @param jobParams 任务参数
     * @param pageNo 页码
     * @return 查询结果
     */
    public abstract JSONArray queryBatchData(JSONObject jobParams, int pageNo);

    /**
     * 处理业务
     * @param jobParams 任务参数
     * @param dataItem 单条数据
     * @return 处理结果
     */
    public abstract Object handle(JSONObject jobParams, JSONObject dataItem);
}
