package com.af.v4.system.common.file.utils;

import com.af.v4.system.common.core.constant.FileConstants;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.liuli.utils.enums.OSTypeEnum;
import com.af.v4.system.common.resource.config.ResourceConfig;
import org.springframework.stereotype.Component;

import java.io.File;

/**
 * 文件路径构建工具
 */
@Component
public final class FilePathUtil {

    private final ApplicationUtils applicationUtils;

    private final ResourceConfig resourceConfig;

    public FilePathUtil(ApplicationUtils applicationUtils, ResourceConfig resourceConfig) {
        this.applicationUtils = applicationUtils;
        this.resourceConfig = resourceConfig;
    }

    /**
     * 获取文件扩展名
     *
     * @param filename 文件名
     * @return 扩展名
     */
    public static String fileExt(String filename) {
        String[] array = filename.split("\\.");
        return array[array.length - 1].toLowerCase();
    }

    /**
     * 获取服务器的文件存储根路径
     *
     * @return 服务器的文件存储根路径
     */
    public String getFileRootPathByServer() {
        String rootPath = resourceConfig.getFileRootPath();
        if (rootPath == null) {
            // 生成保存的相对路径
            OSTypeEnum osType = ApplicationUtils.getOSType();
            if (osType == OSTypeEnum.WINDOWS) {
                rootPath = "D:" + File.separatorChar + "resources";
            } else {
                rootPath = File.separatorChar + "usr" + File.separatorChar + "local" + File.separatorChar + "tomcat" + File.separatorChar + "files";
            }
        }
        return rootPath;
    }

    /**
     * 获取服务器的文件存储扩展路径
     *
     * @return 服务器的文件存储扩展路径
     */
    public String getFileExtraPathByServer(String resourceType) {
        return File.separatorChar + applicationUtils.getApplicationName()
                + File.separatorChar + FileConstants.RESOURCE_TYPE_2_DIR.get(resourceType);
    }

    /**
     * 获取服务器的文件存储完整路径
     *
     * @return 服务器的文件存储完整路径
     */
    public String getFileFullPathByServer(String resourceType, String fileName) {
        return getFileRootPathByServer() + getFileExtraPathByServer(resourceType) + fileName;
    }

    /**
     * 获取服务器的文件下载路径
     *
     * @return 服务器的文件下载路径
     */
    public String getFileDownloadUrlByServer(String resourceType, String fileName) {
        return "/resource" + getFileExtraPathByServer(resourceType) + fileName;
    }

    /**
     * 获取minio的文件存储扩展路径
     *
     * @return minio的文件存储扩展路径
     */
    public String getFileExtraPathBySSO(String resourceType) {
        return FileConstants.RESOURCE_TYPE_2_DIR.get(resourceType);
    }

    /**
     * 获取minio的文件存储完整路径
     *
     * @return minio的文件存储完整路径
     */
    public String getFileFullPathBySSO(String resourceType, String fileName) {
        return getFileExtraPathBySSO(resourceType) + fileName;
    }

    /**
     * 获取minio的文件下载路径
     *
     * @return minio的文件下载路径
     */
    public String getFileDownloadUrlBySSO(String resourceType, String fileName) {
        return "/minio/" + getFileFullPathBySSO(resourceType, fileName);
    }
}
