package com.af.v4.system.common.file.service;

import com.af.v4.system.common.core.constant.FileConstants;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.file.entity.UploadFileInfo;
import com.af.v4.system.common.file.enums.ResUploadModeEnum;
import com.af.v4.system.common.file.utils.FilePathUtil;
import com.af.v4.system.common.jpa.service.EntityService;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.plugins.core.CalculateUtil;
import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.plugins.core.ConvertTools;
import org.apache.commons.io.FileUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * 文件处理服务
 *
 * @author Mr.river
 */
@Service
public class FileService {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileService.class);
    private final EntityService entityService;
    private final MinioService minioService;
    private final ApplicationUtils applicationUtils;
    private final FilePathUtil filePathUtil;

    public FileService(EntityService entityService, MinioService minioService, ApplicationUtils applicationUtils, FilePathUtil filePathUtil) {
        this.entityService = entityService;
        this.minioService = minioService;
        this.applicationUtils = applicationUtils;
        this.filePathUtil = filePathUtil;
    }

    /**
     * 文件上传
     *
     * @param file          文件
     * @param resUploadMode 资源上传模式
     * @param operator      操作人
     * @param formType      表单类型
     * @param useType       附件用途
     * @return 响应
     */
    public FileUploadResult uploadFile(MultipartFile file, String resUploadMode, String operator, String formType, String useType
    ) {
        ResUploadModeEnum resUploadTypeEnum = ResUploadModeEnum.toType(resUploadMode);
        if (resUploadTypeEnum == null) {
            return new FileUploadResult(false, "不支持该资源上传类型：" + resUploadMode);
        }
        // 文件名称
        String fileName = file.getOriginalFilename();
        assert fileName != null;

        UploadFileInfo fileInfo = new UploadFileInfo();
        // 文件大小
        fileInfo.setSize(CalculateUtil.div(file.getSize(), 1024 * 1024, 4));
        // 文件后缀名
        fileInfo.setExtension(FilePathUtil.fileExt(fileName));
        // 解析扩展名称对应的系统资源类型
        fileInfo.setResourceType(FileConstants.RESOURCE_EXT_2_TYPE.get(fileInfo.getExtension()));
        // 检测是否为系统不支持的资源类型
        if (fileInfo.getResourceType() == null) {
            throw new ServiceException("当前资源类型不支持上传：" + fileInfo.getExtension());
        }
        // 上传原文件的文件名
        fileInfo.setOriginalName(fileName.replaceAll("(?i)." + fileInfo.getExtension(), ""));
        // 自定义新的存储文件名
        fileInfo.setSaveName(CommonTools.getUUID(true) + "." + fileInfo.getExtension());
        FileUploadResult result;
        // 根据不同的文件上传模式，上传文件
        switch (resUploadTypeEnum) {
            case SERVER -> {
                fileInfo.setExtraPath(filePathUtil.getFileExtraPathByServer(fileInfo.getResourceType()));
                fileInfo.setSavePath(filePathUtil.getFileFullPathByServer(fileInfo.getResourceType(), fileInfo.getSaveName()));
                Map<String, Object> entity = uploadFileToServer(file, fileInfo, operator, formType, useType);
                result = new FileUploadResult(true, entity);
            }
            case BASE64 -> {
                Map<String, Object> entity = uploadFileToBase64(file);
                result = new FileUploadResult(true, entity);
            }
            default -> {
                fileInfo.setExtraPath(filePathUtil.getFileExtraPathBySSO(fileInfo.getResourceType()));
                fileInfo.setSavePath(filePathUtil.getFileFullPathBySSO(fileInfo.getResourceType(), fileInfo.getSaveName()));
                Map<String, Object> entity = uploadFileToOSS(file, fileInfo, operator, formType, useType);
                result = new FileUploadResult(true, entity);
            }
        }
        return result;
    }

    /**
     * 上传文件到服务器
     *
     * @param file     文件对象
     * @param fileInfo 文件实体
     * @param operator 操作人
     * @param formType 表单类型
     * @param useType  用途
     * @return t_files实体
     */
    public Map<String, Object> uploadFileToServer(MultipartFile file, UploadFileInfo fileInfo, String operator, String formType, String useType) {
        String path = fileInfo.getSavePath();
        File fileDir = new File(path);
        // 写入文件
        try {
            FileUtils.copyInputStreamToFile(file.getInputStream(), fileDir);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        String downloadPath = filePathUtil.getFileDownloadUrlByServer(fileInfo.getResourceType(), fileInfo.getSaveName());
        try {
            return saveFileEntity(ResUploadModeEnum.SERVER, fileInfo, downloadPath, path, operator, formType, useType);
        } catch (IOException | ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 上传文件为base64串
     *
     * @param file 文件
     * @return 结果
     */
    public Map<String, Object> uploadFileToBase64(MultipartFile file) {
        try {
            Map<String, Object> message = new HashMap<>(1);
            message.put("base64", ConvertTools.base64Encode(file.getBytes()));
            return message;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 上传文件到OSS
     *
     * @param file     文件
     * @param fileInfo 文件实体
     * @param operator 操作人
     * @param formType 表单类型
     * @param useType  用途
     * @return 结果
     */
    public Map<String, Object> uploadFileToOSS(MultipartFile file, UploadFileInfo fileInfo, String operator, String formType, String useType) {
        String bucketName = applicationUtils.getApplicationName();
        try {
            minioService.upload(file, fileInfo.getSavePath(), bucketName);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        String downloadPath = filePathUtil.getFileDownloadUrlBySSO(fileInfo.getResourceType(), fileInfo.getSaveName());
        try {
            return saveFileEntity(ResUploadModeEnum.SERVER, fileInfo, downloadPath, fileInfo.getSavePath(), operator, formType, useType);
        } catch (IOException | ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    public FileUploadResult deleteFile(Integer id, String resDeleteMode) {
        ResUploadModeEnum resUploadTypeEnum = ResUploadModeEnum.toType(resDeleteMode);
        FileUploadResult result;
        if (resUploadTypeEnum == null) {
            return new FileUploadResult(false, "不支持该资源上传类型：" + resDeleteMode);
        }
        JSONObject row = new JSONObject();
        row.put("id", id);
        row.put("f_state", "删除");
        try {
            JSONObject entityResult = entityService.partialSave("t_files", row);
            result = new FileUploadResult(true, entityResult);
        } catch (IOException | ClassNotFoundException e) {
            throw new RuntimeException("实体t_files不存在, {}", e);
        }
        // 根据不同的文件上传模式，删除文件
        switch (resUploadTypeEnum) {
            case SERVER -> {
                // TODO
            }
            case BASE64 -> {
                // TODO
            }
            default -> {
                // TODO
            }
        }
        return result;
    }

    private Map<String, Object> saveFileEntity(ResUploadModeEnum resUploadModeEnum, UploadFileInfo fileInfo, String downloadPath, String realPath, String operator, String formType, String useType) throws IOException, ClassNotFoundException {
        // 存储记录
        JSONObject filesEntity = new JSONObject();

        filesEntity.put("f_filename", fileInfo.getSaveName());
        filesEntity.put("f_stock_id", 0);
        filesEntity.put("f_filetype", fileInfo.getExtension());
        filesEntity.put("f_downloadpath", downloadPath);
        filesEntity.put("f_realpath", realPath);
        filesEntity.put("f_file_size", fileInfo.getSize());
        filesEntity.put("f_operator", operator);
        filesEntity.put("f_form_type", formType);
        filesEntity.put("f_type", fileInfo.getResourceType());
        filesEntity.put("f_use_type", useType);
        filesEntity.put("f_upload_mode", resUploadModeEnum.getValue());
        filesEntity.put("fusetype", useType);
        // 保存记录
        filesEntity.put("id", entityService.partialSave("t_files", filesEntity).getString("id"));
        return filesEntity.toMap();
    }

    /**
     * 文件上传结果
     */
    public static class FileUploadResult {

        /**
         * 是否成功
         */
        private final Boolean isSuccess;
        /**
         * 响应数据
         */
        private final Object data;

        private FileUploadResult(boolean isSuccess, Object data) {
            this.isSuccess = isSuccess;
            this.data = data;
        }


        public Boolean getSuccess() {
            return isSuccess;
        }

        public Object getData() {
            return data;
        }
    }
}
