package com.af.v4.system.common.file.service;

import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.file.enums.ResUploadModeEnum;
import com.af.v4.system.common.jpa.service.EntityService;
import com.af.v4.system.common.plugins.core.CalculateUtil;
import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.plugins.core.ConvertTools;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.resource.config.ResourceConfig;
import org.apache.commons.io.FileUtils;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 文件处理服务
 *
 * @author Mr.river
 */
@Service
public class FileService {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileService.class);
    private static final String STATIC_RES_URL = "/resource";
    private final EntityService entityService;
    private final MinioService minioService;
    private final ResourceConfig resourceConfig;

    public FileService(EntityService entityService, MinioService minioService, ResourceConfig resourceConfig) {
        this.entityService = entityService;
        this.minioService = minioService;
        this.resourceConfig = resourceConfig;
    }


    /**
     * 文件上传
     *
     * @param file          文件
     * @param resUploadMode 资源上传模式
     * @param stockId       文件所属仓库id
     * @param operator      操作人
     * @param pathKey       扩展目录
     * @param formType      表单类型
     * @param useType       附件用途
     * @param bucketName    OSS BucketName
     * @return 响应
     */
    public FileUploadResult uploadFile(MultipartFile file, String resUploadMode, int stockId,
                                       String operator, String pathKey,
                                       String formType, String useType,
                                       String bucketName
    ) {
        ResUploadModeEnum resUploadTypeEnum = ResUploadModeEnum.toType(resUploadMode);
        if (resUploadTypeEnum == null) {
            return new FileUploadResult(false, "不支持该资源上传类型：" + resUploadMode);
        }
        JSONObject stock = new JSONObject();
        // 查询出仓库信息
        if (stockId > 0 ) {
            JSONObject response = entityService.remoteGetById(ServiceNameConstants.SYSTEM_SERVICE, "f_root_path, f_alias, f_lower_path_json", "t_files_manager", stockId);
            if (response.getInt("status") != HttpStatus.SUCCESS) {
                String msg = response.optString("msg", "unknown error");
                return new FileUploadResult(false, "查询仓库信息失败：" + msg);
            }
            stock = response.getJSONObject("data");
        }
        // 获取目录
        String rootPath = stock.optString("f_root_path",resourceConfig.getFileRootPath());
        String path = rootPath + stock.optString("f_alias",bucketName);
        // 获取扩展目录
        String extraPathKey = null;
        if (pathKey != null) {
            JSONArray extraPathKeys =
                    new JSONArray(new JSONArray(stock.getString("f_lower_path_json")).toList().stream()
                            .filter(item -> Objects.equals(((Map<String, Object>) item).get("label"), pathKey))
                            .collect(Collectors.toList()));
            if (!extraPathKeys.isEmpty()) {
                extraPathKey = extraPathKeys.getJSONObject(0).getString("value");
            }
        }
        // 获取文件名称和大小
        String fileName = file.getOriginalFilename();
        BigDecimal fileSize = CalculateUtil.div(file.getSize(), 1024 * 1024, 4);
        FileUploadResult result;
        // 根据不同的文件上传模式，上传文件
        switch (resUploadTypeEnum) {
            case SERVER -> {
                assert fileName != null;
                Map<String, Object> entity = uploadFileToServer(file, stockId, rootPath, path, extraPathKey, fileName, fileSize, operator, formType, useType);
                result = new FileUploadResult(true, entity);
            }
            case BASE64 -> {
                Map<String, Object> entity = uploadFileToBase64(file);
                result = new FileUploadResult(true, entity);
            }
            case OSS -> {
                Map<String, Object> entity = uploadFileToOSS(file, stockId, rootPath, path, extraPathKey, fileName,
                        fileSize, operator, formType, useType, bucketName);
                result = new FileUploadResult(true, entity);
            }
            default ->
                    throw new ServiceException("不支持的上传模式：" + resUploadTypeEnum.getValue(), HttpStatus.BAD_REQUEST);
        }
        return result;
    }

    /**
     * 上传文件到服务器
     *
     * @param file         文件
     * @param f_stock_id   仓库id
     * @param rootPath     仓库根目录
     * @param path         文件目录
     * @param extraPathKey 文件上传扩展路径Key
     * @param filename     文件名称
     * @param filesize     文件大小 m为单位
     * @param operator     操作人
     * @return 结果 t_files 的id
     */
    public Map<String, Object> uploadFileToServer(MultipartFile file, int f_stock_id, String rootPath, String path,
                                                  String extraPathKey, String filename, BigDecimal filesize, String operator, String formType, String useType) {
        // 获取文件后缀名
        String fileSuffix = filename.substring(filename.lastIndexOf('.') + 1);
        // 如果有扩展目录则追加
        if (extraPathKey != null) {
            path += File.separator + extraPathKey;
        }
        // 生成文件名
        String fileName = CommonTools.getUUID(true) + "." + fileSuffix;
        // 生成最终的文件上传路径：path\{extraPath}\yyyyMMdd\UUID_filename
        path += File.separator + DateTools.getNow("yyyyMMdd") + File.separator + fileName;
        File fileDir = new File(path);
        // 写入文件
        try {
            FileUtils.copyInputStreamToFile(file.getInputStream(), fileDir);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        // 存储记录
        JSONObject files_entity = new JSONObject();
        String downloadPath = STATIC_RES_URL + "/" + path.replace(File.separator, "/").substring(rootPath.length());
        files_entity.put("f_filename", filename);
        files_entity.put("f_stock_id", f_stock_id);
        files_entity.put("f_filetype", fileSuffix);
        files_entity.put("f_downloadpath", downloadPath);
        files_entity.put("f_realpath", path);
        files_entity.put("f_file_size", filesize);
        files_entity.put("f_operator", operator);
        files_entity.put("f_form_type", formType);
        files_entity.put("f_use_type", useType);
        files_entity.put("fusetype", useType);
        // 保存记录
        try {
            files_entity.put("id", entityService.partialSave("t_files", files_entity).getString("id"));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return files_entity.toMap();
    }

    /**
     * 上传文件为base64串
     *
     * @param file 文件
     * @return 结果
     */
    public Map<String, Object> uploadFileToBase64(MultipartFile file) {
        try {
            Map<String, Object> message = new HashMap<>(1);
            message.put("base64", ConvertTools.base64Encode(file.getBytes()));
            return message;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 上传文件到OSS
     *
     * @param file 文件
     * @return 结果
     */
    public Map<String, Object> uploadFileToOSS(MultipartFile file, int f_stock_id, String rootPath, String path,
                                               String extraPathKey, String filename, BigDecimal filesize,
                                               String operator, String formType, String useType, String BucketName) {
        // 获取文件后缀名
        String fileSuffix = filename.substring(filename.lastIndexOf('.') + 1);
        // 生成文件名
        String fileName = CommonTools.getUUID(true) + "." + fileSuffix;

        if (formType.equals("image")) {
            fileName = "images/" + fileName;
        }

        try {
            minioService.upload(file, fileName, BucketName);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        // 存储记录
        JSONObject files_entity = new JSONObject();
        String downloadPath = "/oss/" + BucketName + "/" + fileName;
        String realPath = "Minio/" + BucketName + "/" + fileName;
        files_entity.put("f_filename", filename);
        files_entity.put("f_stock_id", f_stock_id);
        files_entity.put("f_filetype", fileSuffix);
        files_entity.put("f_downloadpath", downloadPath);
        files_entity.put("f_realpath", realPath);
        files_entity.put("f_file_size", filesize);
        files_entity.put("f_operator", operator);
        files_entity.put("f_form_type", formType);
        files_entity.put("f_use_type", useType);
        files_entity.put("fusetype", useType);
        // 保存记录
        try {
            files_entity.put("id", entityService.partialSave("t_files", files_entity).getString("id"));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return files_entity.toMap();
    }

    public FileUploadResult deleteFile(Integer id, String resDeleteMode) {
        ResUploadModeEnum resUploadTypeEnum = ResUploadModeEnum.toType(resDeleteMode);
        FileUploadResult result;
        if (resUploadTypeEnum == null) {
            return new FileUploadResult(false, "不支持该资源上传类型：" + resDeleteMode);
        }
        // 根据不同的文件上传模式，上传文件
        switch (resUploadTypeEnum) {
            case SERVER -> {
                JSONObject row = new JSONObject();
                row.put("id", id);
                row.put("f_state", "删除");
                try {
                    JSONObject jsonObject = entityService.partialSave("t_files", row);
                    result = new FileUploadResult(true, jsonObject);
                } catch (IOException | ClassNotFoundException e) {
                    throw new RuntimeException("实体t_files不存在, {}", e);
                }
            }
            case BASE64 -> {
                result = new FileUploadResult(true, "BASE64");
            }
            case OSS -> {
                result = new FileUploadResult(true, "OSS");
            }
            default ->
                    throw new ServiceException("不支持的上传模式：" + resUploadTypeEnum.getValue(), HttpStatus.BAD_REQUEST);
        }
        return result;
    }

    /**
     * 文件上传结果
     */
    public static class FileUploadResult {

        /**
         * 是否成功
         */
        private final Boolean isSuccess;
        /**
         * 响应数据
         */
        private final Object data;

        private FileUploadResult(boolean isSuccess, Object data) {
            this.isSuccess = isSuccess;
            this.data = data;
        }


        public Boolean getSuccess() {
            return isSuccess;
        }

        public Object getData() {
            return data;
        }
    }
}
