package com.af.v4.system.common.expression.report.core;

import org.dom4j.Element;
import org.json.JSONArray;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

// 一批单元格组成的可以复制的左边块，每个块对应一个数据集
// content - 重复快用于获取数据的表达式
public class LeftBlock extends Block {
    // 块头单元格，块可以没有头单元格
    public Cell headCell;

    // 块表达式中data代表的对象
    public Object data;

    // 重复数据的名称
    public String name;

    // 从xml元素构造
    public LeftBlock(Element ele) {
        super(ele);
        // 加载子块
        for (Element obj : ele.elements("leftblock")) {
            this.cells.add(new LeftBlock(obj));
        }
        // 获取重复数据的名称
        this.name = ele.attributeValue("name");

        // 加载头
        Element eHead = ele.element("head");
        if (eHead != null) {
            this.headCell = new Cell(eHead);
        }
    }

    public LeftBlock(int row, int column, int rowSpan, int columnSpan,
                     String content) {
        super(row, column, rowSpan, columnSpan, content);
    }

    // 重置块的起始行
    public void setRow(int row) {
        // 对块中的每一个单元格，更改列号，如果是块，要递归做
        for (Cell cell : cells) {
            cell.row = cell.row + row;
            if (cell instanceof LeftBlock block) {
                block.setRow(row);
            }
        }
        if (this.headCell != null) {
            this.headCell.row += row;
        }
    }

    @Override
    public void compile() {
        super.compile();
        if (this.headCell != null) {
            this.headCell.compile();
        }
    }

    @Override
    public void setReport(Report report) {
        super.setReport(report);
        if (this.headCell != null) {
            this.headCell.setReport(report);
        }
    }

    // 根据块的数据，产生可以使用的单元格
    // result - 创建的单元格存放在result里
    // return - 复制了多少行
    public int createCell(List<Cell> result) {
        // 对单元格按行列进行排序，把同一行放在一起
        cells.sort(Comparator.comparingInt(o -> (o.row * 10000 + o.column)));

        // 复制了多少行
        int copyRow = 0;
        // 对所有排序了的单元格（包括块），重新计算列
        for (Cell cell : cells) {
            // 如果是block
            if (cell instanceof LeftBlock block) {
                // 如果是可复制的block
                if (block.content != null && !block.content.trim().isEmpty()) {
                    int i = 0;
                    Object v = block.invoke();
                    if (v instanceof JSONArray) {
                        for (Object item : (JSONArray) v) {
                            LeftBlock newBlock = block.clone();
                            // 第一块只加增加的行数，第二块要加上块本身的行数
                            if (i != 0) {
                                copyRow += block.rowSpan;
                            }
                            // 设置块的起始列
                            newBlock.setRow(copyRow);
                            // 让新的块产生单元格
                            List<Cell> cells = new ArrayList<>();

                            newBlock.data = item;
                            this.report.vars.put(newBlock.name, item);

                            copyRow += newBlock.createCell(cells);
                            result.addAll(cells);

                            i++;
                        }
                    } else {
                        throw new RuntimeException("必须返回JSONArray，行："
                                + block.row + ", 列：" + block.column + ", 内容："
                                + block.content);
                    }
                } else {
                    // 不可复制的block，直接把block内容拿处理即可
                    LeftBlock newBlock = block.clone();
                    // 设置块的起始列
                    newBlock.setRow(copyRow);
                    // 让新的块产生单元格
                    List<Cell> cells = new ArrayList<>();
                    copyRow += newBlock.createCell(cells);
                    result.addAll(cells);
                }
            } else {
                cell.row = cell.row + copyRow;
                cell.content = cell.invoke().toString();
                result.add(cell);
            }
        }

        // 如果有头部，返回头部单元格扩展的结果
        if (this.headCell != null) {
            this.headCell.rowSpan += copyRow;
            this.headCell.content = this.headCell.invoke().toString();
            result.add(this.headCell);
        }

        return copyRow;
    }

    // 复制
    public LeftBlock clone() {
        LeftBlock result = new LeftBlock(this.row, this.column, this.rowSpan,
                this.columnSpan, this.content);
        if (this.delegate != null) {
            result.delegate = this.delegate;
        }
        if (this.report != null) {
            result.report = this.report;
        }

        // 复制单元格
        for (Cell cell : cells) {
            result.cells.add(cell.clone());
        }

        // 复制表头
        if (this.headCell != null) {
            result.headCell = this.headCell.clone();
        }
        if (this.name != null) {
            result.name = this.name;
        }

        return result;
    }
}
