package com.af.v4.system.common.expression.report.core;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.json.JSONObject;

import java.io.StringReader;
import java.util.*;

// 产生一个实际报表。
// 分两步，首先记录各单元格行、列、跨行数、跨列数，然后根据记录对所有单元个进行整理，产生table表格。
public class Report {
    // 报表中用到的sql查询以及查询参数
    public final List<Sql> sqls = new ArrayList<>();
    // 报表所有单元格以及块
    public final List<Cell> cells = new ArrayList<>();
    // 报表所对应的全局变量，包括sql语句执行结果
    public Map<String, Object> vars = new HashMap<>();

    // 用xml文件内容以及json格式参数构造报表
    public Report(String source, JSONObject vars) {
        this.load(source);
        this.vars = vars.toMap();
    }

    // 产生HTML的table形式的报表结果
    public String render() {
        // 编译
        this.compile();
        // 把report注入各单元格，以便执行时，从report中获取参数
        for (Cell cell : cells) {
            cell.setReport(this);
        }

        // 产生单元格
        List<Cell> result = this.createCells();
        Document doc = DocumentHelper.createDocument();
        Element table = this.layout(doc, result);
        return table.asXML();
    }

    // 加载xml格式报表
    private void load(String source) {
        SAXReader reader = new SAXReader();
        try {
            Document document = reader.read(new StringReader(source));
            Element root = document.getRootElement();
            // 加载所有sql语句
            for (Element obj : root.elements("sql")) {
                this.sqls.add(new Sql(obj));
            }
            // 加载所有独立单元格
            for (Element obj : root.elements("cell")) {
                this.cells.add(new Cell(obj));
            }
            // 加载所有报表块
            for (Element obj : root.elements("reportblock")) {
                this.cells.add(new ReportBlock(obj));
            }
        } catch (DocumentException ex) {
            throw new RuntimeException(ex);
        }
    }

    // 对所有单元格的表达式进行编译
    public void compile() {
        for (Cell cell : cells) {
            cell.compile();
        }
    }

    // 产生单元格，根据绘制的、排好顺序的块，按顺序一块块处理。
    public List<Cell> createCells() {
        List<Cell> result = new ArrayList<>();

        // 对单元格按行列进行排序，把同一行放在一起
        cells.sort(Comparator.comparingInt(o -> (o.row * 10000 + o.column)));
        int copyRow = 0;
        for (Cell cell : cells) {
            // 如果是报表块，让报表块产生单元格
            if (cell instanceof ReportBlock block) {
                block.setRow(copyRow);
                copyRow += block.createCell(result);
            }
            // 纯单元格
            else {
                if (!("false".equals(cell.isDown))) {
                    cell.row += copyRow;
                }
                // 执行单元格内容表达式
                cell.content = cell.invoke().toString();
                result.add(cell);
            }
        }

        return result;
    }

    // 对所有单元格进行整理，产生table表格
    public Element layout(Document doc, List<Cell> cells) {
        // 产生table
        Element eTable = doc.addElement("table");
        eTable.addAttribute("class", "tableprint");
        // 对单元格按行列进行排序，把同一行放在一起
        cells.sort(Comparator.comparingInt(o -> (o.row * 10000 + o.column)));
        int currentRow = -1;
        // 正在添加的当前行
        Element eTr = null;
        for (Cell cell : cells) {
            // 当行号发生变化时，产生新的tr元素
            if (cell.row != currentRow) {
                eTr = eTable.addElement("tr");
                currentRow = cell.row;
            }
            // 添加td，设置内容
            assert eTr != null;
            Element eTd = eTr.addElement("td");
            if (cell.css != null && !cell.css.isEmpty()) {
                eTd.addAttribute("class", cell.css);
            }
            if (cell.width != null && !cell.width.isEmpty()) {
                eTd.addAttribute("width", cell.width);
            }
            if (cell.height != null && !cell.height.isEmpty()) {
                eTd.addAttribute("height", cell.height);
            }
            if (cell.columnSpan > 1) {
                eTd.addAttribute("colspan", cell.columnSpan + "");
            }
            if (cell.rowSpan > 1) {
                eTd.addAttribute("rowspan", cell.rowSpan + "");
            }
            // 是html类型，输出html
            if (cell.type != null && cell.type.equals("html")) {
                SAXReader reader = new SAXReader();
                try {
                    Document document = reader.read(new StringReader(cell.content));
                    Element root = document.getRootElement();
                    eTd.add(root);
                } catch (RuntimeException e) {
                    throw e;
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            } else {
                eTd.addText(cell.content);
            }
        }
        return eTable;
    }
}
