package com.af.v4.system.common.expression.core;

import com.af.v4.system.common.expression.exception.ReturnWayException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * 执行单元，保存所有虚参，Expression编译后的结果
 *
 * @author 何宁社
 */
public class Delegate implements Function<JSONObject, Object>, Consumer<JSONObject>, Supplier<Object> {
    /**
     * 执行时的Expression
     */
    private final Expression exp;
    /**
     * 执行的源代码
     */
    private final String source;
    /**
     * lambda外层参数名列表
     */
    private final List<String> lambdaOutProps = new ArrayList<>();
    /**
     * 参数值及变量值存放处
     */
    private Map<String, Object> objectNames = new HashMap<>(32);

    /**
     * 用Expression构造，执行时最顶层Expression
     *
     * @param exp 表达式
     */
    Delegate(Expression exp, String source) {
        this.exp = exp;
        this.source = source;
    }

    /**
     * 用Expression构造，执行时最顶层Expression
     *
     * @param exp 表达式
     */
    Delegate(Expression exp, String source, Map<String, Object> objectNames) {
        this.exp = exp;
        this.source = source;
        this.objectNames = objectNames;
    }

    /**
     * 无参执行过程
     *
     * @return 结果
     */
    public Object invoke() {
        return this.invoke(new HashMap<>(32));
    }

    /**
     * 执行程序，执行前，参数必须实例化
     *
     * @param params 执行时，所带参数值
     * @return 执行结果
     */
    public Object invoke(Map<String, Object> params) {
        // 把初始参数给参数表
        this.objectNames = params;
        // 沿根Expression节点遍历，把delegate传递下去
        putDelegate(this.exp);
        // 调用exp的执行过程
        try {
            return exp.invoke();
        } catch (ReturnWayException returnWay) {
            return returnWay.getReturnObject();
        }
    }

    /**
     * 沿根节点递归，传递delegate的过程
     *
     * @param parent 表达式
     */
    private void putDelegate(Expression parent) {
        for (Expression child : parent.children) {
            // 有些节点会放空的子节点
            if (child == null) {
                continue;
            }
            //try特殊处理
            if ("Try".equals(child.type.name())) {
                Expression[] exps = (Expression[]) child.value;
                assert exps != null;
                if (exps[0] != null) {
                    exps[0].setDelegate(this);
                    putDelegate(exps[0]);
                }
                if (exps[1] != null) {
                    exps[1].setDelegate(this);
                    putDelegate(exps[1]);
                }
            }
            child.setDelegate(this);
            putDelegate(child);
        }
    }

    Expression getExp() {
        return this.exp;
    }

    Map<String, Object> getObjectNames() {
        return objectNames;
    }

    List<String> getLambdaOutProps() {
        return lambdaOutProps;
    }

    void put(String key, Object value) {
        objectNames.put(key, value);
    }

    Object get(String key) {
        return objectNames.get(key);
    }

    boolean containsKey(String key) {
        return objectNames.containsKey(key);
    }

    String getSource() {
        return source;
    }

    public Object apply() {
        return apply(null);
    }

    @Override
    public Object apply(JSONObject params) {
        if (params == null) {
            params = new JSONObject();
        }
        Map<String, Object> map = this.objectNames;
        if (lambdaOutProps.isEmpty()) {
            lambdaOutProps.addAll(map.keySet());
        }
        Map<String, Object> lambdaMap = new HashMap<>(map);
        lambdaMap.put("data", params);
        return this.invoke(lambdaMap);
    }

    @Override
    public void accept(JSONObject params) {
        apply(params);
    }

    @Override
    public Object get() {
        return get(null);
    }
}
