package com.af.v4.system.common.expression.util;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class MethodSignatureMatcher {

    /**
     * Method缓存
     */
    private static final Map<String, Method> CACHE_METHOD_MAP = new ConcurrentHashMap<>(32);

    /**
     * 判断方法method的参数和params指定的参数是否兼容
     *
     * @param method 方法
     * @param args   参数列表
     * @return 是否兼容
     */
    public static boolean isCompatible(final Method method, final Object[] args) {
        final Class<?>[] targetTypes = method.getParameterTypes();
        if (args.length != targetTypes.length) {
            return false;
        }
        for (int i = 0; i < args.length; i++) {
            final Class<?> targetType = targetTypes[i];
            if (!isCompatible((Class<?>) args[i], targetType)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 两个类型是否兼容
     *
     * @param srcType    源类型
     * @param targetType 目标类型
     * @return 是否兼容
     */
    private static boolean isCompatible(Class<?> srcType,
                                        final Class<?> targetType) {
        if (srcType == null || targetType == srcType) {
            return true;
        }
        // int与Integer兼容
        if (
                (targetType == int.class && srcType == Integer.class) ||
                        (targetType == Integer.class && srcType == int.class)
        ) {
            return true;
        }

        // 如果传入的是接口
        if (srcType.isInterface()) {
            return hasInterface(srcType, targetType);
            // 如果传入的是类
        } else {
            if (targetType.isInterface()) {
                while (srcType != null) {
                    Class<?>[] facets = srcType.getInterfaces();
                    for (Class<?> facet : facets) {
                        // 看facet是否有
                        if (hasInterface(facet, targetType)) {
                            return true;
                        }
                    }
                    srcType = srcType.getSuperclass();
                }
                return false;
            } else {
                return isSubclass(srcType, targetType);
            }
        }
    }

    /**
     * srcType 实现了 targetType 接口中的一个
     *
     * @param srcType    源类型
     * @param targetType 目标类型
     * @return 是否实现
     */
    private static boolean hasInterface(Class<?> srcType, Class<?> targetType) {
        if (targetType.isInterface()) {
            if (targetType == srcType) {
                return true;
            }
            // 查看本身实现的接口
            Class<?>[] facets = srcType.getInterfaces();
            for (Class<?> facet : facets) {
                if (facet == targetType) {
                    return true;
                } else {
                    return hasInterface(facet, targetType);
                }
            }
        } else {
            while (targetType != null) {
                // 查看本身实现的接口
                Class<?>[] facets = targetType.getInterfaces();
                for (Class<?> facet : facets) {
                    if (facet == srcType) {
                        return true;
                    } else {
                        return hasInterface(srcType, facet);
                    }
                }
                targetType = targetType.getSuperclass();
            }
        }
        return false;
    }

    /**
     * srcType 是 targetType 的子类
     *
     * @param srcType    源类型
     * @param targetType 目标类型
     * @return 是否子类
     */
    private static boolean isSubclass(Class<?> srcType, Class<?> targetType) {
        Class<?> superClass = srcType.getSuperclass();

        while (superClass != null) {
            if (superClass == targetType) {
                return true;
            }
            superClass = superClass.getSuperclass();
        }
        // 如果是Object.class，再判断一次
        return null == targetType;
    }

    /**
     * 查找类clazz是否存在签名是args的method
     *
     * @param clazz      类
     * @param methodName 方法名
     * @param args       参数列表
     */
    public static Method getMatchingMethod(final Class<?> clazz,
                                           String methodName, final Object[] args) {
        String realMethodName = clazz.getName() + "@" + methodName + Arrays.toString(args);
        if (CACHE_METHOD_MAP.containsKey(realMethodName)) {
            return CACHE_METHOD_MAP.get(realMethodName);
        } else {
            for (final Method method : clazz.getMethods()) {
                if (method.getName().equals(methodName)
                        && isCompatible(method, args)) {
                    CACHE_METHOD_MAP.put(realMethodName, method);
                    return method;
                }
            }
            return null;
        }
    }
}
