package com.af.v4.system.common.excel.convert;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.excel.converters.Converter;
import com.alibaba.excel.enums.CellDataTypeEnum;
import com.alibaba.excel.metadata.GlobalConfiguration;
import com.alibaba.excel.metadata.data.ReadCellData;
import com.alibaba.excel.metadata.data.WriteCellData;
import com.alibaba.excel.metadata.property.ExcelContentProperty;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * BigDecimal转换
 * 确保BigDecimal数值导出时精度一致
 */
public class ExcelBigDecimalConvert implements Converter<BigDecimal> {

    /**
     * 小数位数精度
     */
    private static final int DECIMAL_SCALE = 4;

    /**
     * 舍入模式
     */
    private static final RoundingMode ROUNDING_MODE = RoundingMode.DOWN;

    @Override
    public Class<BigDecimal> supportJavaTypeKey() {
        return BigDecimal.class;
    }

    @Override
    public CellDataTypeEnum supportExcelTypeKey() {
        return CellDataTypeEnum.NUMBER;
    }

    @Override
    public BigDecimal convertToJavaData(ReadCellData<?> cellData, ExcelContentProperty contentProperty, GlobalConfiguration globalConfiguration) {
        String stringValue = cellData.getStringValue();
        if (stringValue != null) {
            try {
                return new BigDecimal(stringValue).setScale(DECIMAL_SCALE, ROUNDING_MODE);
            } catch (NumberFormatException e) {
                return BigDecimal.ZERO;
            }
        }
        return BigDecimal.ZERO;
    }

    @Override
    public WriteCellData<Object> convertToExcelData(BigDecimal value, ExcelContentProperty contentProperty, GlobalConfiguration globalConfiguration) {
        if (ObjectUtil.isNotNull(value)) {
            // 确保导出时使用一致的小数位数和舍入规则
            BigDecimal scaledValue = value.setScale(DECIMAL_SCALE, ROUNDING_MODE);
            WriteCellData<Object> cellData = new WriteCellData<>(scaledValue);
            cellData.setType(CellDataTypeEnum.NUMBER);
            return cellData;
        }
        WriteCellData<Object> cellData = new WriteCellData<>(BigDecimal.ZERO);
        cellData.setType(CellDataTypeEnum.NUMBER);
        return cellData;
    }
}
