package com.af.v4.system.common.excel.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.io.resource.ClassPathResource;
import cn.hutool.core.util.IdUtil;
import com.af.v4.system.common.excel.constant.ExcelConstant;
import com.af.v4.system.common.excel.convert.BaseDateConverter;
import com.af.v4.system.common.excel.convert.CustomStringStringConverter;
import com.af.v4.system.common.excel.convert.ExcelBigNumberConvert;
import com.af.v4.system.common.excel.core.CellColorSheetWriteHandler;
import com.af.v4.system.common.excel.core.DefaultExcelListener;
import com.af.v4.system.common.excel.core.ExcelListener;
import com.af.v4.system.common.excel.core.ExcelResult;
import com.af.v4.system.common.excel.template.ExcelTemplate_Grey;
import com.af.v4.system.common.excel.utils.FileUtil;
import com.af.v4.system.common.resource.config.ResourceConfig;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.exception.ExcelGenerateException;
import com.alibaba.excel.support.ExcelTypeEnum;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.excel.write.metadata.style.WriteCellStyle;
import com.alibaba.excel.write.metadata.style.WriteFont;
import com.alibaba.excel.write.style.HorizontalCellStyleStrategy;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.alibaba.excel.write.style.row.SimpleRowHeightStyleStrategy;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.util.*;


/**
 * Excel服务
 *
 * @author Eraser
 */
@Service
public class ExcelService {

    /**
     * 分页依据 每页最大数据数
     */
    private static int pageMaxSize;
    private final ResourceConfig resourceConfig;

    public ExcelService(ResourceConfig resourceConfig) {
        this.resourceConfig = resourceConfig;
    }

    /**
     * excel格式判断并设置分页依据
     */
    private static String getExcelType(int dataSize, int headSize) {
        String type;
        if (dataSize < 65536 - 6 && headSize < 256) {
            type = "XLS";
            pageMaxSize = 65536 - 6;
        } else {
            type = "XLSX";
            pageMaxSize = 1048576 - 6;
        }
        return type;
    }

    /**
     * 根据pageMaxSize进行数据分页
     *
     * @param sheetName 表名
     * @param writer    写入器
     * @param head      表头
     * @param writeData 写入数据
     * @param fillData  填充数据
     */
    private static void branchSheet(String sheetName, ExcelWriter writer, List<List<String>> head, List<List<Object>> writeData, List<Object> fillData) {
        int sheetCount;
        int start = 0;
        int dataSize = writeData == null ? fillData.size() : writeData.size();
        if (dataSize % pageMaxSize == 0) {
            sheetCount = dataSize / pageMaxSize;
        } else {
            sheetCount = (dataSize / pageMaxSize) + 1;
        }
        try {
            for (int pageNo = 0; pageNo < sheetCount; pageNo++) {
                WriteSheet writeSheet;
                int end = (pageNo + 1) * pageMaxSize;
                if (end > dataSize) {
                    end = dataSize;
                }
                if (head != null) {
                    writeSheet = EasyExcel.writerSheet(pageNo, sheetName + pageNo).head(head).build();
                    if (writeData != null) {
                        writer.write(writeData.subList(start, end), writeSheet);
                    }
                } else {
                    writeSheet = EasyExcel.writerSheet(pageNo, sheetName + pageNo).build();
                    writer.fill(fillData.subList(start, end), writeSheet);
                }
                start = end;
            }
        } catch (Exception e) {
            throw new ExcelGenerateException("分页导出错误！如是模板填充导出请确定是否根据文件限制大小提前设置模板页数！");
        }
    }

    /**
     * 重置响应体
     */
    private static void resetResponse(String sheetName, String type, HttpServletResponse response) {
        String filename = encodingFilename(sheetName, type);
        FileUtil.setAttachmentResponseHeader(response, filename);
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;charset=UTF-8");
    }

    /**
     * 编码文件名
     */
    private static String encodingFilename(String filename, String type) {
        if (type.equals("XLSX")) {
            return filename + "_" + IdUtil.fastSimpleUUID() + ".xlsx";
        } else if (type.equals("XLS")) {
            return filename + "_" + IdUtil.fastSimpleUUID() + ".xls";
        } else {
            return filename + "_" + IdUtil.fastSimpleUUID() + ".cvs";
        }
    }

    /**
     * 同步导入(适用于小数据量)
     *
     * @param is       输入流
     * @param headList 表头列表
     * @return 转换后集合
     */
    public static <T> List<T> importExcel(InputStream is, JSONArray headList) {
        return EasyExcel.read(is).head(assembleHead(headList)).autoCloseStream(false).sheet().doReadSync();
    }

    /**
     * 使用校验监听器 异步导入 同步返回
     *
     * @param is         输入流
     * @param headMap    表头列表  中文表头 --> 字段
     * @param isValidate 是否 Validator 检验 默认为是
     * @return 转换后集合
     */
    public static <T> ExcelResult<T> importExcel(InputStream is, Map<String, Object> headMap, boolean isValidate) {
        DefaultExcelListener<T> listener = new DefaultExcelListener<>(isValidate, headMap);
        EasyExcel.read(is, listener).sheet()
                .registerConverter(new BaseDateConverter.LocalDateTimeConverter())
                .registerConverter(new BaseDateConverter.LocalDateConverter())
                .registerConverter(new BaseDateConverter.LocalTimeConverter())
                .doRead();
        return listener.getExcelResult();
    }


    /**
     * 使用自定义监听器 异步导入 自定义返回
     *
     * @param is       输入流
     * @param listener 自定义监听器
     * @return 转换后集合
     */
    public static <T> ExcelResult<T> importExcel(InputStream is, ExcelListener<T> listener) {
        EasyExcel.read(is, listener).sheet()
                .registerConverter(new BaseDateConverter.LocalDateTimeConverter())
                .doRead();
        return listener.getExcelResult();
    }

    /**
     * 数据转换写入构建器
     *
     * @param type        Excel类型
     * @param os          输出流
     * @param constants   样式模板
     * @param hasStyle    是否存在样式
     * @param style       样式
     * @param hasAppoint  是否使用单元格样式设置
     * @param appointCell 单元格
     */
    private static ExcelWriterBuilder write(String type, OutputStream os, ExcelConstant constants, boolean hasStyle, List<Object> style, boolean hasAppoint, List<Object> appointCell) {
        //执行导出
        ExcelWriterBuilder builder = EasyExcel.write(os)
                //不自动关闭流
                .autoCloseStream(false)
                //大数值自动转换
                .registerConverter(new ExcelBigNumberConvert())
                //String转换
                .registerConverter(new CustomStringStringConverter())
                //Timestamp转换
                .registerConverter(new BaseDateConverter.LocalDateTimeConverter())
                //设置Excel类型
                .excelType(ExcelTypeEnum.valueOf(type))
                //自动适配
                .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                //设置头和内容行高
                .registerWriteHandler(new SimpleRowHeightStyleStrategy((short) 20, (short) 20));
        if (hasStyle) {
            //设置表头样式
            HorizontalCellStyleStrategy horizontalCellStyleStrategy = exportStyle(style, constants);
            builder.registerWriteHandler(horizontalCellStyleStrategy);
        }
        if (hasAppoint) {
            //设置表头样式
            HashMap<Integer, List<Integer>> map = (HashMap<Integer, List<Integer>>) appointCell.get(0);
            CellColorSheetWriteHandler writeHandler = new CellColorSheetWriteHandler(map, IndexedColors.RED.index, IndexedColors.YELLOW.index);
            builder.registerWriteHandler(writeHandler);
        }
        return builder;
    }

    /**
     * excel样式设定
     *
     * @param styleList 样式列表
     * @param constants 样式模板
     * @return 样式注册对象
     */
    private static HorizontalCellStyleStrategy exportStyle(List<Object> styleList, ExcelConstant constants) {
        try {
            if (styleList != null) {
                Map<String, Object> styleMap = (Map<String, Object>) styleList.get(0);
                short head_font_size = Short.parseShort(styleMap.get("EXCEL_HEAD_FONT_SIZE").toString());
                String head_align = styleMap.get("EXCEL_HEAD_ALIGN").toString().toUpperCase();
                String head_font_color = styleMap.get("EXCEL_HEAD_FONT_COLOR").toString().toUpperCase();
                String head_full_color = styleMap.get("EXCEL_HEAD_FULL_COLOR").toString().toUpperCase();

                //样式设定
                //表头
                constants.EXCEL_HEAD_FONT_SIZE = head_font_size;
                constants.EXCEL_HEAD_ALIGN = HorizontalAlignment.valueOf(head_align);
                constants.EXCEL_HEAD_FONT_COLOR = IndexedColors.valueOf(head_font_color).getIndex();
                constants.EXCEL_HEAD_FULL_COLOR = IndexedColors.valueOf(head_full_color).getIndex();
            }
            //设置单元格
            return new HorizontalCellStyleStrategy(getWriteCellStyle(constants), new ArrayList<>());
        } catch (Exception e) {
            throw new RuntimeException("Excel样式设定异常", e);
        }

    }

    private static WriteCellStyle getWriteCellStyle(ExcelConstant constants) {
        WriteCellStyle headWriteCellStyle = new WriteCellStyle();
        WriteFont headFont = new WriteFont();
        //--设置表头字体
        headFont.setColor(constants.EXCEL_HEAD_FONT_COLOR);
        headFont.setFontName(constants.EXCEL_HEAD_FONT_NAME);
        headFont.setFontHeightInPoints(constants.EXCEL_HEAD_FONT_SIZE);
        headWriteCellStyle.setWriteFont(headFont);
        //--设置表头居中对齐
        headWriteCellStyle.setHorizontalAlignment(constants.EXCEL_HEAD_ALIGN);
        //--设置表头填充颜色
        headWriteCellStyle.setFillForegroundColor(constants.EXCEL_HEAD_FULL_COLOR);
        headWriteCellStyle.setShrinkToFit(true);
        return headWriteCellStyle;
    }

    /**
     * Excel默认样式导出方法
     *
     * @param dataList  数据列表
     * @param headList  表头列表
     * @param sheetName 表名
     * @param response  响应值
     */
    public static void exportExcel(List<Object> dataList, JSONArray headList, String sheetName, HttpServletResponse response) {
        //默认导出模板为灰色
        exportExcel(dataList, headList, sheetName, null, true, new ExcelTemplate_Grey(), false, null, response);
    }

    /**
     * 直接导出Excel
     *
     * @param dataList    数据列表
     * @param headList    表头列表
     * @param sheetName   表名
     * @param style       样式
     * @param hasStyle    是否存在样式(存在模板也为存在样式)
     * @param constant    模板
     * @param hasAppoint  是否使用单元格样式设置
     * @param appointCell 单元格样式
     * @param response    响应值
     */
    public static void exportExcel(List<Object> dataList, JSONArray headList, String sheetName, List<Object> style, boolean hasStyle, ExcelConstant constant, boolean hasAppoint, List<Object> appointCell, HttpServletResponse response) {
        try {
            String type = getExcelType(dataList.size(), headList.length());
            resetResponse(sheetName, type, response);
            ServletOutputStream os = response.getOutputStream();
            //组装表格头
            List<List<String>> head = assembleHead(headList);
            //组装数据体
            List<List<Object>> data = assembleData(dataList);
            int dataSize = dataList.size();
            if (dataSize > pageMaxSize) {
                ExcelWriter writer = write(type, os, constant, hasStyle, style, hasAppoint, appointCell).build();
                branchSheet(sheetName, writer, head, data, null);
                writer.finish();
            } else {
                ExcelWriterBuilder builder = write(type, os, constant, hasStyle, style, hasAppoint, appointCell);
                builder.sheet(sheetName).head(head).doWrite(data);
            }
        } catch (IOException e) {
            throw new RuntimeException("导出Excel异常！", e);
        }
    }

    /**
     * 单表多数据模板导出 模板格式为 {.字段}
     * 因为在使用jdk17时导出xlsx格式出错且暂时未找到原因，不得不做出妥协使用xls，但考虑到xls有行数限制，会采用分页来应对
     * jdk8正常使用
     * ！！！！注意：设置模板时请提前设置多页防止系统报错！！！！
     *
     * @param fileName     文件名
     * @param templatePath 模板路径 resource 目录下的路径包括模板文件名
     *                     例如: excel/temp.xlsx
     *                     重点: 模板文件必须放置到启动类对应的 resource 目录下
     * @param data         模板需要的数据
     */
    //todo jdk17模板文件只支持XLS格式,XLSX报错 jdk8正常使用
    public static void exportTemplate(List<Object> data, String fileName, String templatePath, HttpServletResponse response) {
        try {
            if (CollUtil.isEmpty(data)) {
                throw new IllegalArgumentException("数据为空");
            }
            //强制设置导出格式为xls
            String type = getExcelType(1, 1);
            resetResponse(fileName, type, response);
            ServletOutputStream os = response.getOutputStream();
            ClassPathResource templateResource = new ClassPathResource(templatePath);
            ExcelWriter writer = write(type, os, null, false, null, false, null)
                    .withTemplate(templateResource.getStream()).build();
            int dataSize = data.size();
            if (dataSize > pageMaxSize) {
                branchSheet(fileName, writer, null, null, data);
            } else {
                WriteSheet writeSheet = EasyExcel.writerSheet().build();
                writer.fill(data, writeSheet);
            }
            writer.finish();
        } catch (IOException e) {
            throw new RuntimeException("导出Excel异常", e);
        }
    }

    /**
     * 组装表格表头
     *
     * @param headList 头列表
     */
    private static List<List<String>> assembleHead(JSONArray headList) {

        List<List<String>> list = new ArrayList<>();
        headList.forEach(str -> {
            List<String> heads = new ArrayList<>();
            heads.add(str.toString());
            list.add(heads);
        });
        return list;
    }

    /**
     * 组装表格数据
     */
    private static List<List<Object>> assembleData(List<Object> dataList) {
        //组装数据体
        List<List<Object>> lists = new ArrayList<>(dataList.size());
        dataList.forEach(item -> {
            Collection<Object> values = (((Map<String, Object>) item).values());
            List<Object> data = new ArrayList<>();
            values.forEach(v -> {
                if (JSONObject.NULL == v) {
                    v = "--";
                }
                data.add(v);
            });
            lists.add(data);
        });
        return lists;
    }
    /**
     * 组装表格数据
     */
    private static List<List<Object>> assembleData(List<Object> dataList, List<String> fields) {
        //组装数据体
        List<List<Object>> lists = new ArrayList<>(dataList.size());
        dataList.forEach(item -> {
            List<Object> data = new ArrayList<>();
            fields.forEach(
                    field -> {
                        Object v = ((Map<String, Object>) item).get(field);
                        if (JSONObject.NULL == v) {
                            v = "--";
                        }
                        data.add(v);
                    }
            );
            lists.add(data);
        });
        return lists;
    }

    /**
     * 生成Excel文件放到服务器（自定义样式）
     * 动态表头，动态数据，生成Excel文件
     *
     * @param dataList    数据列表
     * @param headList    表头列表
     * @param fields      自定义数据字段
     * @param sheetName   工作簿名
     * @param style       样式
     * @param hasStyle    是否存在样式(存在模板也为存在样式)
     * @param constant    模板
     * @param hasAppoint  是否使用单元格样式设置
     * @param appointCell 单元格样式
     * @return 文件路径
     */
    private String exportExcelToServer(List<Object> dataList, JSONArray headList, List<String> fields, String sheetName, List<Object> style, boolean hasStyle, ExcelConstant constant, boolean hasAppoint, List<Object> appointCell) {
        try {
            String type = getExcelType(dataList.size(), headList.length());
            sheetName = encodingFilename(sheetName, type);
            String filePath = getFilePath();
            String fileName = getFileName(sheetName);
            List<List<String>> head = assembleHead(headList);
            List<List<Object>> data = fields == null ? assembleData(dataList) : assembleData(dataList, fields);
            FileOutputStream fos = new FileOutputStream(filePath + fileName);
            int dataSize = dataList.size();
            if (dataSize > pageMaxSize) {
                ExcelWriter writer = write(type, fos, constant, hasStyle, style, hasAppoint, appointCell).build();
                branchSheet(sheetName, writer, head, data, null);
                writer.finish();
            } else {
                ExcelWriterBuilder builder = write(type, fos, constant, hasStyle, style, hasAppoint, appointCell);
                builder.sheet(sheetName).head(head).doWrite(data);
            }
            return fileName;
        } catch (Exception e) {
            throw new RuntimeException("创建Excel异常！", e);
        }
    }

    /**
     * 生成Excel文件放到服务器（默认样式）
     * 动态表头，动态数据，生成Excel文件
     *
     * @param dataList  数据列表
     * @param headList  表头列表
     * @param sheetName 工作簿名
     * @return 文件路径
     */
    public String exportExcelToServer(List<Object> dataList, JSONArray headList, String sheetName) {
        return exportExcelToServer(dataList, headList, null, sheetName, null, false, null, false, null);
    }

    public String exportExcelToServerByFieldList(List<Object> dataList, JSONArray fieldList, String sheetName) {
        JSONArray headList = new JSONArray();
        List<String> fields = new ArrayList<>(fieldList.length());
        for (Object temp : fieldList) {
            fields.add(((String) temp).split(":")[0]);
            headList.put(((String) temp).split(":")[1]);
        }
        return exportExcelToServer(dataList, headList, fields, sheetName, null, false, null, false, null);
    }

    /**
     * 获取文件路径
     */
    private String getFilePath() {
        String filePath = resourceConfig.getFileRootPath() + "/excel/export/";
        File file = new File(filePath);
        if (!file.exists()) {
            file.mkdirs();
        }
        return filePath;
    }

    /**
     * 获取文件名称
     */
    private String getFileName(String name) {
        return (name == null ? UUID.randomUUID() + ".xlsx" : name);
    }

}
