package com.af.v4.system.common.elasticsearch.utils;


import co.elastic.clients.elasticsearch.ElasticsearchClient;
import co.elastic.clients.elasticsearch._types.ElasticsearchException;
import co.elastic.clients.elasticsearch._types.mapping.TypeMapping;
import co.elastic.clients.elasticsearch.indices.*;
import co.elastic.clients.json.JsonpMapper;
import jakarta.json.Json;
import jakarta.json.stream.JsonParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.io.StringReader;
import java.util.Map;

/**
 * 为ElasticSearch封装基础API
 * 索引操作相关方法
 */
@Component
public class AFElasticSearchIndexUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(AFElasticSearchIndexUtils.class);

    /**
     * 注入配置文件
     */
    @Autowired
    private ElasticsearchClient client;


    /**
     * 添加索引，并为其设置结构
     *
     * @param indexName 索引名
     * @param mapping   映射字符串，该字符串要符合Json格式
     */
    public void addWithMapping(String indexName, String mapping) {
        boolean exist = isExist(indexName);
        if (exist) {
            LOGGER.error("索引：" + indexName + "已存在，无法创建！请检查后再试！");
            return;
        }
        JsonpMapper mapper = client._transport().jsonpMapper();
        JsonParser parser = Json.createParser(new StringReader(mapping));

        CreateIndexRequest createIndexRequest = new CreateIndexRequest.Builder()
                .index(indexName)
                .mappings(TypeMapping._DESERIALIZER.deserialize(parser, mapper))
                .build();

        CreateIndexResponse response = null;
        try {
            response = client.indices().create(createIndexRequest);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        LOGGER.info("创建ElasticSearch索引[" + indexName + "] 成功！");
        LOGGER.info("索引[" + indexName + "] 的结构为：" + mapping);
    }

    /**
     * 查询索引是否存在
     *
     * @param indexName 索引名
     * @return 是否存在
     */
    public boolean isExist(String indexName) {

        boolean exists = false;
        GetIndexResponse response = null;
        try {
            response = client.indices().get(GetIndexRequest.of(i -> i.index(indexName)));
            exists = true;
        } catch (ElasticsearchException | IOException e) {
            System.out.println(e);
        }
        return exists;
    }

    /**
     * 删除索引
     *
     * @param indexName 索引名
     */
    public void delete(String indexName) {
        boolean exist = isExist(indexName);
        if (!exist) {
            LOGGER.warn("索引[" + indexName + "] 不存在！");
            return;
        }

        DeleteIndexRequest deleteIndexRequest = DeleteIndexRequest.of(i -> i.index(indexName));
        try {
            client.indices().delete(deleteIndexRequest);
        } catch (IOException e) {
            LOGGER.error(e.toString());
        }

    }

    /**
     * 在控制台打印索引属性
     *
     * @param indexName 索引名
     */
    public void showInfo(String indexName) {
        boolean exist = isExist(indexName);
        if (!exist) {
            LOGGER.warn("索引[" + indexName + "] 不存在！");
            return;
        }

        GetIndexRequest getIndexRequest = GetIndexRequest.of(i -> i.index(indexName));
        GetIndexResponse response = null;
        try {
            response = client.indices().get(getIndexRequest);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        IndexState target = response.get(indexName);
        assert target != null;
        Map<String, Alias> aliases = target.aliases();
        TypeMapping mappings = target.mappings();
        IndexSettings settings = target.settings();
        LOGGER.info("aliases");
        LOGGER.info("aliases");
        LOGGER.info(aliases.toString());
        LOGGER.info("mappings");
        assert mappings != null;
        LOGGER.info(mappings.toString());
        LOGGER.info("settings");
        assert settings != null;
        LOGGER.info(settings.toString());
    }
}
