package com.af.v4.system.common.elasticsearch.utils;

import co.elastic.clients.elasticsearch.ElasticsearchClient;
import co.elastic.clients.elasticsearch.core.BulkResponse;
import co.elastic.clients.elasticsearch.core.SearchRequest;
import co.elastic.clients.elasticsearch.core.SearchResponse;
import co.elastic.clients.elasticsearch.core.bulk.BulkOperation;
import co.elastic.clients.elasticsearch.core.search.Hit;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 为ElasticSearch封装基础API
 * 文档操作相关方法（不包含查询）
 */
@Component
public class AFElasticSearchDocUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(AFElasticSearchDocUtils.class);

    /**
     * 注入配置文件
     */
    @Autowired
    private ElasticsearchClient client;


    /**
     * 增
     * 插入数据
     * 需要手动建立实体类，该实体类成员变量名需与ES Mapping中一致
     * List中的数据类型为BulkOperation，示例如下
     * list.add(BulkOperation.of(i -> i.create(e -> e.document(new ESTestEntity("赵","强强","学生","一班","游戏，音乐，抽烟",101,18)))));
     *
     * @param indexName 索引名
     * @param data      待插入数据
     */
    public void insert(String indexName, List<BulkOperation> data) {
        BulkResponse response = null;
        try {
            response = client.bulk(e -> e.index(indexName).operations(data));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        LOGGER.info("存储[" + data.size() + "]条数据成功！共耗时：" + response.took() + "ms");
    }

    /**
     * 将搜索返回值格式化为JsonArray的格式
     *
     * @param response ES的返回值
     * @return 格式化后的JsonArray
     */
    private JSONArray formatResult(SearchResponse<Object> response) {
        JSONArray result = new JSONArray();
        List<Hit<Object>> hits = response.hits().hits();
        for (Hit<Object> hit : hits) {
            String id = hit.id();
            Map source = (Map) hit.source();
            Map<String, List<String>> highlight = hit.highlight();
            JSONObject temp = new JSONObject();
            JSONArray highlightArr = new JSONArray();
            assert source != null;
            Set sourceKeySet = source.keySet();
            temp.put("id", id);
            for (Object o : sourceKeySet) {
                Object o1 = source.get(o);
                if (o1 instanceof Integer) {
                    temp.put(o.toString(), o1);
                } else {
                    temp.put(o.toString(), o1.toString());
                }
            }
            int size = highlight.size();
            if (size > 0) {
                Set<String> highlightKeySet = highlight.keySet();
                for (String s : highlightKeySet) {
                    List<String> list = highlight.get(s);
                    for (String s1 : list) {
                        highlightArr.put(s1);
                    }
                }
            }
            temp.put("highlight", highlightArr);

            result.put(temp);
        }
        return result;
    }

    /**
     * 删
     * 根据id删除文档
     * id的集合类型为BulkOperation
     * 例如：delete.add(new BulkOperation.Builder().delete(d -> d.index("aaa").id("BqsdoIUBWc-7CSjrrB0i")).build());
     *
     * @param indexName 索引名
     * @param ids       id的list集合
     */
    public void delete(String indexName, List<BulkOperation> ids) {
        try {
            BulkResponse response = client.bulk(e -> e.index("aaa").operations(ids));
            long took = response.took();
            LOGGER.info("已完成删除，用时：" + took + "ms");
        } catch (IOException e) {
            LOGGER.error(e.toString());
        }
    }

    /**
     * 执行查询
     *
     * @param request 组装完成的查询请求
     * @return 匹配到的结果
     */
    public JSONArray search(SearchRequest request) {
        try {
            SearchResponse<Object> search = client.search(request, Object.class);
            return formatResult(search);
        } catch (IOException e) {
            LOGGER.error("查询失败！");
            LOGGER.error(e.toString());
            return new JSONArray();
        }
    }
}
