package com.af.v4.system.common.elasticsearch.utils;


import com.af.v4.system.common.elasticsearch.config.ElasticSearchConfig;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.IndicesClient;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.indices.CreateIndexRequest;
import org.elasticsearch.client.indices.CreateIndexResponse;
import org.elasticsearch.client.indices.GetIndexRequest;
import org.elasticsearch.client.indices.GetIndexResponse;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.xcontent.XContentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.Map;
import java.util.Set;

/**
 * 为ElasticSearch封装基础API
 * 索引操作相关方法
 */
@Component
public class AFElasticSearchIndexUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(AFElasticSearchIndexUtils.class);

    /**
     * 注入配置文件
     */
    private final ElasticSearchConfig config;

    public AFElasticSearchIndexUtils(ElasticSearchConfig config) {
        this.config = config;
    }

    /**
     * 添加索引，没什么用
     * 请直接使用addWithMapping，添加索引的同时，创建结构
     *
     * @param indexName 索引名
     * @return ElasticSearch返回信息对象
     */
    public CreateIndexResponse add(String indexName) {
        boolean indexExist = isExist(indexName);
        if (indexExist) {
            LOGGER.warn("索引[{}] 已存在，无法创建！", indexName);
            return null;
        }
        // 获取操作索引的对象
        IndicesClient indices = config.getIndices();
        // 创建索引请求对象
        CreateIndexRequest createIndexRequest = new CreateIndexRequest(indexName);
        CreateIndexResponse response;
        try {
            response = indices.create(createIndexRequest, RequestOptions.DEFAULT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        LOGGER.info("创建ElasticSearch索引[{}] 成功！", indexName);
        return response;
    }

    /**
     * 添加索引，并为其设置结构
     *
     * @param indexName 索引名
     * @param mapping   映射字符串，该字符串要符合Json格式
     * @return ElasticSearch返回信息对象
     */
    public CreateIndexResponse addWithMapping(String indexName, String mapping) {
        CreateIndexResponse response;
        boolean indexExist = isExist(indexName);
        if (indexExist) {
            LOGGER.warn("索引[{}] 已存在，无法创建！", indexName);
            return null;
        }
        IndicesClient indices = config.getIndices();
        CreateIndexRequest request = new CreateIndexRequest(indexName);
        request.mapping(mapping, XContentType.JSON);
        try {
            response = indices.create(request, RequestOptions.DEFAULT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        LOGGER.info("创建ElasticSearch索引[{}] 成功！", indexName);
        LOGGER.info("索引[{}] 的结构为：{}", indexName, mapping);
        return response;
    }

    /**
     * 查询索引是否存在
     *
     * @param indexName 索引名
     * @return 是否存在
     */
    public boolean isExist(String indexName) {
        IndicesClient indices = config.getIndices();
        GetIndexRequest request = new GetIndexRequest(indexName);
        boolean exists;
        try {
            exists = indices.exists(request, RequestOptions.DEFAULT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return exists;
    }

    /**
     * 删除索引
     *
     * @param indexName 索引名
     * @return 是否删除成功
     */
    public boolean delete(String indexName) {
        boolean exist = isExist(indexName);
        if (!exist) {
            LOGGER.warn("索引[{}] 不存在！", indexName);
            return false;
        }
        IndicesClient indices = config.getIndices();
        DeleteIndexRequest request = new DeleteIndexRequest(indexName);
        AcknowledgedResponse delete;
        try {
            delete = indices.delete(request, RequestOptions.DEFAULT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        LOGGER.info("删除索引[{}]成功！", indexName);
        return delete.isAcknowledged();
    }

    /**
     * 在控制台打印索引设定
     *
     * @param indexName 索引名
     */
    public void showInfo(String indexName) {
        boolean exist = isExist(indexName);
        if (!exist) {
            LOGGER.warn("索引[{}] 不存在！", indexName);
            return;
        }
        IndicesClient indices = config.getIndices();
        GetIndexRequest request = new GetIndexRequest(indexName);
        GetIndexResponse response;
        try {
            response = indices.get(request, RequestOptions.DEFAULT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 获取索引的mapping
     *
     * @param indexName 索引名
     * @return mapping定义内容
     */
    public Map<String, Object> getMapping(String indexName) {
        boolean exist = isExist(indexName);
        if (!exist) {
            LOGGER.warn("索引[{}] 不存在！", indexName);
            return null;
        }
        IndicesClient indices = config.getIndices();
        GetIndexRequest request = new GetIndexRequest(indexName);
        GetIndexResponse response;
        try {
            response = indices.get(request, RequestOptions.DEFAULT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        Map<String, MappingMetadata> mappings = response.getMappings();
        Set<String> keys = mappings.keySet();
        for (String key : keys) {
            MappingMetadata mappingMetadata = mappings.get(key);
            Map<String, Object> sourceAsMap = mappingMetadata.getSourceAsMap();
            return (Map<String, Object>) sourceAsMap.get("properties");
        }
        return null;
    }
}
