package com.af.v4.system.common.elasticsearch.common;

import com.af.v4.system.common.elasticsearch.common.constance.AFLogstashTable;
import com.af.v4.system.common.elasticsearch.config.AFLogstashGlobalConfig;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.*;
import java.util.ArrayList;
import java.util.List;

/**
 * 该类用于自动生成Logstash配置文件
 */
@Component
public class AFLogstashGenerator {
    private final AFLogstashGlobalConfig globalConfig;


    @Value("${logstash.outputFolder}")
    private String outputFolder;

    public AFLogstashGenerator(AFLogstashGlobalConfig globalConfig) {
        this.globalConfig = globalConfig;
    }

    /**
     * 自动生成配置文件，并写入到指定目录
     *
     * @param tableList table实体类的List
     */
    public void saveTableConf(List<AFLogstashTable> tableList) {
        for (AFLogstashTable table : tableList) {
            List<String> fieldNames = table.getAllFieldName();
            String outputPath = outputFolder + "\\" + table.getTableName() + ".conf";
            File file = new File(outputPath);

            if (!file.exists()) {
                try {
                    boolean newFile = file.createNewFile();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }

            String content = getConfContent(fieldNames, table);

            try (
                    FileWriter fileWritter = new FileWriter(file, false)
            ) {
                fileWritter.write(content);
                fileWritter.flush();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        System.out.println("已完成输出！");
    }

    /**
     * 动态拼接配置文件中的字符
     *
     * @param fieldNames 所有字段名
     * @param table      对应的table实体类
     * @return 配置文件中的字符串
     */
    public String getConfContent(List<String> fieldNames, AFLogstashTable table) {
        StringBuilder sb = new StringBuilder();
        sb.append("input {");
        sb.append("\n");
        sb.append("\tjdbc {");
        sb.append("\n");
        sb.append("\t\tjdbc_driver_library => \"").append(globalConfig.getJDBCDriverPath()).append("\"");
        sb.append("\n");
        sb.append("\t\tjdbc_driver_class => \"").append(globalConfig.getJDBCDriverClassName()).append("\"");
        sb.append("\n");
        sb.append("\t\tjdbc_connection_string => \"").append(globalConfig.getDBurl()).append("\"");
        sb.append("\n");
        sb.append("\t\tjdbc_user => \"").append(globalConfig.getDBUsername()).append("\"");
        sb.append("\n");
        sb.append("\t\tjdbc_password => \"").append(globalConfig.getDBPassword()).append("\"");
        sb.append("\n");
        sb.append("\t\tschedule => \"").append(table.getSchedule()).append("\"");
        sb.append("\n");
        sb.append("\t\tstatement => \"SELECT * FROM ").append(table.getTableName()).append("\"");
        sb.append("\n");
        sb.append("\t\tuse_column_value => true");
        sb.append("\n");
        sb.append("\t\ttracking_column_type => \"").append(table.getUniqueFiledType()).append("\"");
        sb.append("\n");
        sb.append("\t\ttracking_column => \"").append(table.getUniqueFiled()).append("\"");
        sb.append("\n");
        sb.append("\t\tlast_run_metadata_path => \"syncpoint_table\"");
        sb.append("\n");
        sb.append("\t}");
        sb.append("\n");
        sb.append("}");
        sb.append("\n");
        sb.append("\n");
        sb.append("filter {");
        sb.append("\n");
        sb.append("\tmutate {");
        sb.append("\n");
        sb.append("\t\tadd_field => {\"").append(globalConfig.getAllFieldDefineName()).append("\" => \"");
        for (String name : fieldNames) {
            sb.append("%{").append(name).append("} ");
        }
        sb.append("\"}");
        sb.append("\n");
        sb.append("\t}");
        sb.append("\n");
        sb.append("\tmutate {");
        sb.append("\n");
        sb.append("\t\tadd_field => {\"tableName\" => \"").append(table.getTableName()).append("\"}");
        sb.append("\n");
        sb.append("\t}");
        sb.append("\n");
        sb.append("}");
        sb.append("\n");
        sb.append("\n");

        sb.append("output {");
        sb.append("\n");
        sb.append("\telasticsearch {");
        sb.append("\n");
        sb.append("\t\thosts => [\"").append(globalConfig.getESHost()).append("\"]");
        sb.append("\n");
        sb.append("\t\tindex => [\"").append(globalConfig.getESTargetIndex()).append("\"]");
        sb.append("\n");
        sb.append("\t\tdocument_id => \"%{").append(table.getIdField()).append("}\"");
        sb.append("\n");
        sb.append("\t}");
        sb.append("\tstdout {");
        sb.append("\n");
        sb.append("\t\tcodec => json_lines");
        sb.append("\n");
        sb.append("\t}");
        sb.append("\n");
        sb.append("}");


        return sb.toString();
    }

    /**
     * 获取所有字段名
     *
     * @param table table实体类
     * @return 所有字段名的List
     */
    public List<String> getAllFieldName(AFLogstashTable table) {
        String dbType = globalConfig.getDBType();
        Connection connect = getConnect();
        String sql = "";
        String tableName = table.getTableName();
        if (dbType.equals("sqlserver")) {
            sql = "select COLUMN_NAME from information_schema.COLUMNS where TABLE_NAME = ?";
        }
        if (dbType.equals("oracle")) {
            sql = "select column_name  from user_tab_columns  where table_name= ?";
            tableName = tableName.toUpperCase();
        }
        PreparedStatement preparedStatement;
        ResultSet resultSet;
        List<String> colName = new ArrayList<>();
        try {
            preparedStatement = connect.prepareStatement(sql);
            preparedStatement.setString(1, tableName);
            resultSet = preparedStatement.executeQuery();
            while (resultSet.next()) {
                String name = resultSet.getString(1);
                colName.add(name);
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }

        try {
            connect.close();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }

        return colName;
    }

    /**
     * 获取数据库连接
     *
     * @return 数据库连接
     */
    public Connection getConnect() {
        String url = globalConfig.getDBurl();
        String driver = globalConfig.getJDBCDriverClassName();
        String user = globalConfig.getDBUsername();
        String password = globalConfig.getDBPassword();
        Connection con;
        try {
            Class.forName(driver);
            con = DriverManager.getConnection(url, user, password);
        } catch (ClassNotFoundException | SQLException e) {
            throw new RuntimeException(e);
        }

        return con;
    }
}
