package com.af.v4.system.common.elasticsearch.common;

import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightBuilder;
import org.elasticsearch.search.sort.SortOrder;

/**
 * 构造ES SearchSourceBuilder 对象
 * 可以多次，手动指定条件
 * 支持链式调用
 */
public class ESConditionFactory {

    // 每个类只存在一个原始构造器，对齐进行属性赋值
    private final SearchSourceBuilder builder = new SearchSourceBuilder();

    // 多条件构造器
    private BoolQueryBuilder boolBuilder;

    // 高亮构造器
    private HighlightBuilder highlightBuilder;

    // 是否使用了should，如果使用了should，需要添加最小匹配条数，否则会失效
    private boolean useShould = false;

    /**
     * 初始化boolBuilder
     */
    private void boolBuilderInit(){
        if (boolBuilder == null){
            boolBuilder = new BoolQueryBuilder();
        }
    }

    /**
     * 初始化highlightBuilder
     */
    private void highlightInit(){
        if (highlightBuilder == null){
            highlightBuilder = new HighlightBuilder();
        }
    }

    /**
     * 高亮查询
     * ！！！注意！！！
     * 该方法必须明确指定查询哪一个字段，哪一个值
     * 否则不会生效，如与 allField，findAll一起使用会失效
     * @param preTag 高亮前缀
     * @param postTag 高亮后缀
     * @param highlightFields 需要高亮的字段
     * @return 当前实例对象
     */
    public ESConditionFactory highlight(String preTag,String postTag,String[] highlightFields){
        highlightInit();
        highlightBuilder.preTags(preTag);
        highlightBuilder.postTags(postTag);
        for (String highlightField : highlightFields) {
            highlightBuilder.field(highlightField);
        }
        return this;
    }

    /**
     * 全文检索，搜索所有字段
     * @param target 目标值
     * @return 当前
     */
    public ESConditionFactory allField(String target){
        builder.query(QueryBuilders.matchQuery("all_field_for_search",target));
        return this;
    }

    /**
     * 按范围查询
     * @param fieldName 字段名
     * @param rangeMode 范围条件
     * @param target 目标值
     * @return 当前实例对象
     */
    public ESConditionFactory range(String fieldName, String rangeMode, String target){
        boolBuilderInit();
        switch (rangeMode) {
            case "gt" -> boolBuilder.filter(QueryBuilders.rangeQuery(fieldName).gt(target));
            case "gte" -> boolBuilder.filter(QueryBuilders.rangeQuery(fieldName).gte(target));
            case "lt" -> boolBuilder.filter(QueryBuilders.rangeQuery(fieldName).lt(target));
            case "lte" -> boolBuilder.filter(QueryBuilders.rangeQuery(fieldName).lte(target));
        }
        return this;
    }

    /**
     * 应该是xxx，与or的作用相同
     * @param fieldName 字段
     * @param target 目标值
     * @return 当前实例对象
     */
    public ESConditionFactory should(String fieldName,String target){
        boolBuilderInit();
        useShould = true;
        boolBuilder.should(QueryBuilders.termQuery(fieldName,target));
        return this;
    }

    /**
     * 必须是
     * @param fieldName 字段名
     * @param target 目标值
     * @return 当前实例对象
     */
    public ESConditionFactory must(String fieldName,String target){
        boolBuilderInit();
        boolBuilder.must(QueryBuilders.matchQuery(fieldName,target));
        return this;
    }

    /**
     * 必须不是
     * @param fieldName 字段名
     * @param target 目标值
     * @return 当前实例对象
     */
    public ESConditionFactory mustNot(String fieldName,String target){
        boolBuilderInit();
        boolBuilder.mustNot(QueryBuilders.termQuery(fieldName,target));
        return this;
    }


    /**
     * 查找所有
     * @return 当前实例对象
     */
    public ESConditionFactory findAll(){
        builder.query(QueryBuilders.matchAllQuery());
        return this;
    }

    /**
     * 对搜索内容进行排序
     * @param fieldName 排序字段
     * @param order 排序类型，该值为 org.elasticsearch.search.sort.SortOrder 枚举类
     * @return 当前实例对象
     */
    public ESConditionFactory order(String fieldName, SortOrder order){
        builder.sort(fieldName, order);
        return this;
    }

    /**
     * 分页查询
     * @param from 从第几条开始
     * @param size 每页大小
     * @return 当前实例对象
     */
    public ESConditionFactory limit(int from,int size){
        builder.from(from);
        builder.size(size);
        return this;
    }

    /**
     * 获取最终builder对象
     * @return builder对象
     */
    public SearchSourceBuilder result(){
        if (useShould){
            boolBuilder.minimumShouldMatch(1);
        }
        if (boolBuilder != null){
            builder.query(boolBuilder);
        }
        if (highlightBuilder != null){
            builder.highlighter(highlightBuilder);
        }
        return builder;
    }
}
