package com.af.v4.system.common.elasticsearch.utils;

import com.af.v4.system.common.elasticsearch.config.ElasticSearchConfig;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightField;
import org.elasticsearch.xcontent.XContentType;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.*;

/**
 * 为ElasticSearch封装基础API
 * 文档操作相关方法（不包含查询）
 */
@Component
public class AFElasticSearchDocUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(AFElasticSearchDocUtils.class);

    /**
     * 注入配置文件
     */
    private final ElasticSearchConfig config;

    public AFElasticSearchDocUtils(ElasticSearchConfig config) {
        this.config = config;
    }

    /**
     * 增
     * 插入数据
     *
     * @param indexName 索引名
     * @param data      待插入数据，格式：[{key:value,key:value},{key:value,key:value}...]
     */
    public void insert(String indexName, JSONArray data) {
        RestHighLevelClient client = config.getClient();
        BulkRequest bulkRequest = new BulkRequest();
        int sum = 0;
        for (int i = 0; i < data.length(); i++) {
            sum++;
            JSONObject item = data.getJSONObject(i);
            Set<String> keySet = item.keySet();
//            String uuid = getUUID();
            Map<String, Object> map = new HashMap<>();
            for (String key : keySet) {
                map.put(key, item.get(key));
            }
            bulkRequest.add(new IndexRequest(indexName).source(map, XContentType.JSON));
        }
        BulkResponse response;
        try {
            response = client.bulk(bulkRequest, RequestOptions.DEFAULT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        LOGGER.info("存储[" + sum + "]条数据成功！供耗时：" + response.getTook());
    }

    /**
     * 删
     * 根据id删除文档
     *
     * @param indexName 索引名
     * @param ids       id的list集合
     * @return ES响应对象
     */
    public BulkResponse delete(String indexName, List<String> ids) {
        RestHighLevelClient client = config.getClient();
        BulkRequest bulkRequest = new BulkRequest();
        int sum = 0;
        for (String id : ids) {
            sum++;
            bulkRequest.add(new DeleteRequest().index(indexName).id(id));
        }
        BulkResponse response;
        try {
            response = client.bulk(bulkRequest, RequestOptions.DEFAULT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        LOGGER.info("共删除文档[" + sum + "]个，执行成功！");
        return response;
    }

    /**
     * 执行查询
     *
     * @param indexName 查询的索引
     * @param builder   条件构造器
     * @return 匹配到的结果
     */
    public List<List<Map<String, Object>>> search(String indexName, SearchSourceBuilder builder) {
        RestHighLevelClient client = config.getClient();
        SearchRequest request = new SearchRequest(indexName);

        request.source(builder);
        SearchResponse response;
        try {
            response = client.search(request, RequestOptions.DEFAULT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        SearchHits hits = response.getHits();

        List<List<Map<String, Object>>> hitResponse = new ArrayList<>();
        for (SearchHit hit : hits) {
            List<Map<String, Object>> temp = new ArrayList<>();
            Map<String, Object> hitResult = hit.getSourceAsMap();
            Map<String, HighlightField> highlightFields = hit.getHighlightFields();
            Set<String> keySet = highlightFields.keySet();
            Map<String, String> highlightItems = new HashMap<>();
            for (String key : keySet) {
                String original = highlightFields.get(key).toString();
                String modified = original.substring(original.lastIndexOf("[") + 1, original.length() - 2);
                highlightItems.put(key, modified);
            }
//            String hitHighLightResultStr = hit.getHighlightFields().toString();
//            hitHighLightResultStr = hitHighLightResultStr.substring(hitHighLightResultStr.lastIndexOf("[") + 1,hitHighLightResultStr.length() -5);
            Map<String, Object> hitHighLightResult = new HashMap<>();
            hitHighLightResult.put("highlight", highlightItems);
            temp.add(hitResult);
            temp.add(hitHighLightResult);
            hitResponse.add(temp);
        }

        return hitResponse;
    }
}
