package com.af.v4.system.common.datasource;

import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.common.datasource.wrapper.AfDataSourceWrapper;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;
import org.springframework.lang.NonNull;

import javax.sql.DataSource;
import java.util.*;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * 动态数据源工具类（基于Java 21 ScopedValue实现）
 */
public class DynamicDataSource extends AbstractRoutingDataSource {
    public static final String DEFAULT_DATASOURCE_NAME = "master";
    private static final Logger LOGGER = LoggerFactory.getLogger(DynamicDataSource.class);

    /**
     * 使用ScopedValue替代ThreadLocal
     */
    private static final ScopedValue<Deque<String>> LOOKUP_KEY_HOLDER = ScopedValue.newInstance();

    private static Map<String, AfDataSourceWrapper> dataSourceInfo = new HashMap<>(3);

    public DynamicDataSource(Map<String, AfDataSourceWrapper> dataSources) {
        dataSourceInfo = dataSources;
        super.setTargetDataSources(new HashMap<>(1));
        super.afterPropertiesSet();
    }

    public DynamicDataSource() {
    }

    /**
     * 获取当前数据源名称
     */
    public static String getDataSource() {
        Deque<String> deque = LOOKUP_KEY_HOLDER.orElse(new ArrayDeque<>());
        return deque.peek() != null ? deque.peek() : DEFAULT_DATASOURCE_NAME;
    }

    /**
     * 核心方法：设置数据源并执行业务
     * 由于ScopedValue不可变，必须创建新的Deque副本
     * 并通过ScopedValue.where()开启作用域执行业务
     */
    public static <T> T withDataSource(String dataSource, Supplier<T> supplier) {
        if (dataSource == null) {
            return supplier.get();
        }
        Deque<String> currentStack = LOOKUP_KEY_HOLDER.orElse(new ArrayDeque<>());
        // 如果当前栈顶就是目标数据源，直接执行业务
        if (!currentStack.isEmpty() && currentStack.peek().equals(dataSource)) {
            return supplier.get();
        }
        // 复制当前栈，push新数据源
        Deque<String> newStack = new ArrayDeque<>(currentStack);
        String dataSourceStr = StringUtils.isEmpty(dataSource) ? DEFAULT_DATASOURCE_NAME : dataSource;
        newStack.push(dataSourceStr);

        // 作用域内执行，自动传递ScopedValue
        try {
            return ScopedValue.where(LOOKUP_KEY_HOLDER, newStack).call(supplier::get);
        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static void withDataSource(String dataSource, Runnable runnable) {
        withDataSource(dataSource, () -> {
            runnable.run();
            return null;
        });
    }

    public static <T> T withDataSource(String dataSource, Function<JSONObject, T> runnable) {
        return withDataSource(dataSource, runnable, null);
    }

    public static <T> T withDataSource(String dataSource, Function<JSONObject, T> runnable, JSONObject params) {
        return withDataSource(dataSource, () -> runnable.apply(params));
    }

    public static Map<String, AfDataSourceWrapper> getDataSourceMap() {
        return dataSourceInfo;
    }

    public static AfDataSourceWrapper getWrapper() {
        String dataSource = getDataSource();
        AfDataSourceWrapper wrapper = getDataSourceMap().get(dataSource);
        if (wrapper == null) {
            LOGGER.error("不存在的数据源：[{}]", dataSource);
            throw new ServiceException("不存在的数据源", HttpStatus.ERROR);
        }
        return wrapper;
    }

    public static DbType getDbType() {
        return getWrapper().getDbType();
    }

    @NonNull
    @Override
    protected DataSource determineTargetDataSource() {
        String lookupKey = determineCurrentLookupKey();
        AfDataSourceWrapper wrapper = dataSourceInfo.get(lookupKey);
        if (wrapper == null) {
            wrapper = dataSourceInfo.get(DEFAULT_DATASOURCE_NAME);
        }
        if (wrapper == null) {
            throw new ServiceException("不存在的数据源 [" + lookupKey + "]");
        }
        return wrapper.getDataSource();
    }

    @Override
    protected String determineCurrentLookupKey() {
        return getDataSource();
    }
}
