package com.af.v4.system.common.datasource.config;

import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.CheckedException;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.provider.DataSourceProvider;
import com.af.v4.system.common.datasource.wrapper.AfDataSourceWrapper;
import com.zaxxer.hikari.HikariDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Component;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * 动态多数据源配置
 *
 * @author Mr.river
 */
@Configuration
@ConfigurationProperties(prefix = "db")
@Component
public class DynamicDataSourceConfig {

    private final static Logger LOGGER = LoggerFactory.getLogger(DynamicDataSourceConfig.class);

    /**
     * 数据源列表
     */
    private List<AfDataSourceConfig> dataSourceList;
    private List<String> subClass;
    private Integer minimumIdle;
    private Integer maximumPoolSize;
    private Integer connectionTimeout;
    private Integer idleTimeout;
    private Integer maxLifetime;

    @Bean
    @Primary
    public DynamicDataSource routingDataSource() {
        if (Optional.ofNullable(dataSourceList).isEmpty()) {
            throw new CheckedException("服务启动失败：请检查是否正确加载了nacos配置");
        }
        // 循环加入数据源（延迟加载），实例化动态数据源工具类
        Map<String, AfDataSourceWrapper> targetDataSources = new LinkedHashMap<>(dataSourceList.size());
        final StringBuilder dataSourcePrintList = new StringBuilder();
        dataSourceList.forEach(item -> {
            if (!StringUtils.isEmpty(item.getRedirect())) {
                AfDataSourceWrapper targetWrapper = targetDataSources.get(item.getRedirect());
                if (targetWrapper != null) {
                    targetDataSources.put(item.getName(), targetWrapper);
                    dataSourcePrintList.append(item.getName()).append(" -> ").append(item.getRedirect()).append(", ");
                    return;
                } else {
                    LOGGER.error("无效的数据源重定向：{}", item.getRedirect());
                    throw new ServiceException("无效的数据源重定向", HttpStatus.CONFIG_ERROR);
                }
            }
            dataSourcePrintList.append(item.getName()).append(", ");
            Supplier<HikariDataSource> dataSource = () -> DataSourceProvider.create(item, minimumIdle, maximumPoolSize,
                    connectionTimeout, idleTimeout, maxLifetime);
            targetDataSources.put(item.getName(), new AfDataSourceWrapper(dataSource, item));
        });
        if (dataSourcePrintList.length() >= 2) {
            dataSourcePrintList.delete(dataSourcePrintList.length() - 2, dataSourcePrintList.length());
        }
        LOGGER.info("识别到的数据源：[{}]", dataSourcePrintList);
        return new DynamicDataSource(targetDataSources);
    }

    public List<AfDataSourceConfig> getDataSourceList() {
        return dataSourceList;
    }

    public void setDataSourceList(List<AfDataSourceConfig> dataSourceList) {
        this.dataSourceList = dataSourceList;
    }

    public List<String> getSubClass() {
        return subClass;
    }

    public void setSubClass(List<String> subClass) {
        this.subClass = subClass;
    }

    public Integer getMinimumIdle() {
        return minimumIdle;
    }

    public void setMinimumIdle(Integer minimumIdle) {
        this.minimumIdle = minimumIdle;
    }

    public Integer getMaximumPoolSize() {
        return maximumPoolSize;
    }

    public void setMaximumPoolSize(Integer maximumPoolSize) {
        this.maximumPoolSize = maximumPoolSize;
    }

    public Integer getConnectionTimeout() {
        return connectionTimeout;
    }

    public void setConnectionTimeout(Integer connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
    }

    public Integer getIdleTimeout() {
        return idleTimeout;
    }

    public void setIdleTimeout(Integer idleTimeout) {
        this.idleTimeout = idleTimeout;
    }

    public Integer getMaxLifetime() {
        return maxLifetime;
    }

    public void setMaxLifetime(Integer maxLifetime) {
        this.maxLifetime = maxLifetime;
    }
}
