package com.af.v4.system.common.datasource.dialects.id;

import com.af.v4.system.common.datasource.dialects.Dialect;
import com.af.v4.system.common.datasource.dialects.DialectException;
import com.af.v4.system.common.datasource.dialects.Type;
import com.af.v4.system.common.datasource.dialects.enums.GenerationType;

import java.sql.Connection;

/**
 * Generate a Sorted UUID, total length = sortedLength + uuidLength, <br/>
 * for example, SortedUUIDGenerator(5,20): <br/>
 * 10001NmpQHeGLy8eozSSq2p1B<br/>
 * 10002DLIGkILFISKJF23KLSDF<br/>
 * 10003LVBIFI35LDFJIA31KDSF<br/>
 *
 * @since 1.0.6
 */
public record SortedUUIDGenerator(String name, int sortedLength, int uuidLength) implements IdGenerator {

    @Override
    public GenerationType getGenerationType() {
        return GenerationType.SORTED_UUID;
    }

    @Override
    public String getIdGenName() {
        return name;
    }

    @Override
    public Boolean dependOnAutoIdGenerator() {
        return true;
    }

    @Override
    public Object getNextID(Connection con, Dialect dialect, Type dataType) {
        String s = "" + AutoIdGenerator.INSTANCE.getNextID(con, dialect, dataType);
        if (s.length() > (sortedLength - 1))
            throw new DialectException("SortedLength should set bigger than auto generated ID length");
        StringBuilder sb = new StringBuilder("1");
        for (int i = 1; i < sortedLength - s.length(); i++)
            sb.append("0");
        sb.append(s);
        sb.append(UUIDAnyGenerator.getAnyLengthRadix36UUID(uuidLength));
        return sb.toString();
    }

    @Override
    public IdGenerator newCopy() {
        return new SortedUUIDGenerator(name, sortedLength, uuidLength);
    }

}
