package com.af.v4.system.common.datasource.dialects.id;

import com.af.v4.system.common.datasource.dialects.*;
import com.af.v4.system.common.datasource.dialects.enums.GenerationType;

import java.sql.Connection;

/**
 * Define an Identity type generator, supported by MySQL, SQL Server, DB2,
 * Derby, Sybase, PostgreSQL
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class IdentityIdGenerator implements IdGenerator {
    private String table;

    private String column;

    public IdentityIdGenerator(String table, String column) {
        this.table = table;
        this.column = column;
    }

    @Override
    public GenerationType getGenerationType() {
        return GenerationType.IDENTITY;
    }

    @Override
    public String getIdGenName() {
        return "IDENTITY";
    }

    @Override
    public Boolean dependOnAutoIdGenerator() {
        return false;
    }

    @Override
    public Object getNextID(Connection con, Dialect dialect, Type dataType) {
        if (!Boolean.TRUE.equals(dialect.ddlFeatures.getSupportsIdentityColumns()))
            throw new DialectException("Dialect '" + dialect + "' does not support identity type");
        String sql;
        if (Type.BIGINT.equals(dataType))
            sql = dialect.ddlFeatures.getIdentitySelectStringBigINT();
        else
            sql = dialect.ddlFeatures.getIdentitySelectString();
        if (StrUtils.isEmpty(sql) || DDLFeatures.NOT_SUPPORT.equals(sql))
            throw new DialectException("Dialect '" + dialect + "' does not support identity type");
        sql = StrUtils.replaceFirst(sql, "_table__col", table + "_" + column);
        return JdbcUtil.qryOneObject(con, sql);
    }

    @Override
    public IdGenerator newCopy() {
        return new IdentityIdGenerator(table, column);
    }

    // getter & setter==============
    public String getTable() {
        return table;
    }

    public void setTable(String table) {
        this.table = table;
    }

    public String getColumn() {
        return column;
    }

    public void setColumn(String column) {
        this.column = column;
    }
}
