package com.af.v4.system.common.datasource.dialects;

import java.util.HashSet;
import java.util.Set;

/**
 * @since 2.0.4
 */
public class SqlFormatter {
    private static final Set<String> keywords = new HashSet<>();

    static {
        keywords.add("select");
        keywords.add("insert");
        keywords.add("update");
        keywords.add("delete");
        keywords.add("create");
        keywords.add("drop");
        keywords.add("alter");

        keywords.add("from");
        keywords.add("in");
        keywords.add("exists");
        keywords.add("where");

        keywords.add("order");
        keywords.add("group");
        keywords.add("union");

        keywords.add("inner");
        keywords.add("left");
        keywords.add("right");
        keywords.add("outer");
    }

    private static boolean isSpace(char c) {
        return c == ' ' || c == '\t' || c == '\n' || c == '\r';
    }

    public static String DDLFormat(String template) {
        StringBuilder sb = new StringBuilder();
        char[] chars = ("  " + template + "  ").toCharArray();
        int inBrackets = 0;
        boolean justReturned = false;
        for (int i = 2; i < chars.length - 2; i++) {
            char c = chars[i];
            if (isSpace(c)) {
                if (!justReturned)
                    sb.append(c);
            } else if (c == '(') {
                sb.append(c);
                if (inBrackets == 0) {
                    sb.append("\n");
                    justReturned = true;
                }
                inBrackets++;
            } else if (c == ')') {
                inBrackets--;
                if (inBrackets == 0) {
                    sb.append("\n");
                    justReturned = true;
                }
                sb.append(c);
            } else if (c == ',') {
                sb.append(c);
                if (inBrackets == 1) {
                    sb.append("\n");
                    justReturned = true;
                }
            } else {
                if (justReturned && inBrackets > 0) {
                    // 缩进空格数
                    sb.append("    ");
                }
                sb.append(c);
                justReturned = false;
            }
        }
        return sb.toString();
    }

    public static String format(String template) {
        StringBuilder sb = new StringBuilder();
        char[] chars = ("  " + template + "  ").toCharArray();
        // 0 0 ' 1 1 1 ' 0 0 0
        boolean inStr = false;
        boolean justAddedSpace = true;
        StringBuilder lastWord = new StringBuilder();
        for (int i = 2; i < chars.length - 2; i++) {
            char c = chars[i];
            if (isSpace(c)) {
                if (inStr) {
                    sb.append(c);
                } else if (!justAddedSpace) {
                    addLast(sb, lastWord);
                    sb.append(' ');
                    justAddedSpace = true;
                }
            } else if (c == '\'') {
                if (!inStr)
                    addLast(sb, lastWord);
                sb.append('\'');
                inStr = !inStr;
                justAddedSpace = false;
            } else {
                justAddedSpace = false;
                if (inStr)
                    sb.append(c);
                else
                    lastWord.append(c);
            }
        }
        addLast(sb, lastWord);
        return sb.toString();
    }

    private static void addLast(StringBuilder sb, StringBuilder lastWord) {
        if (lastWord.isEmpty())
            return;
        String last = lastWord.toString();
        if (!sb.isEmpty() && keywords.contains(last.toLowerCase()))
            sb.append("\n");
        sb.append(last);
        lastWord.setLength(0);
    }

}
