package com.af.v4.system.common.datasource.config;

import com.af.v4.system.common.core.exception.CheckedException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.provider.DataSourceProvider;
import com.af.v4.system.common.datasource.wrapper.AfDataSourceWrapper;
import com.zaxxer.hikari.HikariDataSource;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * 动态多数据源配置
 *
 * @author Mr.river
 */
@Configuration
@ConfigurationProperties(prefix = "db")
@Component
public class DynamicDataSourceConfig {
    /**
     * 数据源列表
     */
    private List<AfDataSourceConfig> dataSourceList;
    private List<String> subClass;
    private Integer minimumIdle;
    private Integer maximumPoolSize;
    private Integer connectionTimeout;
    private Integer idleTimeout;
    private Integer maxLifetime;

    @Bean
    @Primary
    public DynamicDataSource routingDataSource() {
        if (Optional.ofNullable(dataSourceList).isEmpty()) {
            throw new CheckedException("服务启动失败：数据源加载失败，请检查项目是否正确加载了nacos中心对应的业务配置");
        }
        // 循环加入数据源，实例化动态数据源工具类
        Map<String, AfDataSourceWrapper> targetDataSources = new HashMap<>(dataSourceList.size());
        dataSourceList.forEach(item -> {
            HikariDataSource dataSource = DataSourceProvider.create(item, minimumIdle, maximumPoolSize,
                    connectionTimeout, idleTimeout, maxLifetime);
            targetDataSources.put(item.getName(), new AfDataSourceWrapper(dataSource, item));
        });
        return new DynamicDataSource(targetDataSources.get(DynamicDataSource.DEFAULT_DATASOURCE_NAME).getDataSource(), targetDataSources);
    }

    public List<AfDataSourceConfig> getDataSourceList() {
        return dataSourceList;
    }

    public void setDataSourceList(List<AfDataSourceConfig> dataSourceList) {
        this.dataSourceList = dataSourceList;
    }

    public List<String> getSubClass() {
        return subClass;
    }

    public void setSubClass(List<String> subClass) {
        this.subClass = subClass;
    }

    public Integer getMinimumIdle() {
        return minimumIdle;
    }

    public void setMinimumIdle(Integer minimumIdle) {
        this.minimumIdle = minimumIdle;
    }

    public Integer getMaximumPoolSize() {
        return maximumPoolSize;
    }

    public void setMaximumPoolSize(Integer maximumPoolSize) {
        this.maximumPoolSize = maximumPoolSize;
    }

    public Integer getConnectionTimeout() {
        return connectionTimeout;
    }

    public void setConnectionTimeout(Integer connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
    }

    public Integer getIdleTimeout() {
        return idleTimeout;
    }

    public void setIdleTimeout(Integer idleTimeout) {
        this.idleTimeout = idleTimeout;
    }

    public Integer getMaxLifetime() {
        return maxLifetime;
    }

    public void setMaxLifetime(Integer maxLifetime) {
        this.maxLifetime = maxLifetime;
    }
}
