package com.af.v4.system.common.datasource;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.common.datasource.wrapper.AfDataSourceWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.NamedThreadLocal;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;

import javax.sql.DataSource;
import java.util.*;
import java.util.function.Supplier;

/**
 * 动态数据源工具类
 */
public class DynamicDataSource extends AbstractRoutingDataSource {

    /**
     * 默认数据源名称
     */
    public static final String DEFAULT_DATASOURCE_NAME = "master";
    private static final Logger LOGGER = LoggerFactory.getLogger(DynamicDataSource.class);
    /**
     * 数据源切换堆栈
     */
    private static final ThreadLocal<Deque<String>> LOOKUP_KEY_HOLDER = new NamedThreadLocal<>("dynamic-datasource") {
        @Override
        protected Deque<String> initialValue() {
            return new ArrayDeque<>();
        }
    };
    private static Map<String, AfDataSourceWrapper> dataSourceInfo = new HashMap<>(3);

    public DynamicDataSource(DataSource dataSource, Map<String, AfDataSourceWrapper> dataSources) {
        dataSourceInfo = dataSources;
        Map<Object, Object> targetDataSourcesMap = new HashMap<>(dataSources.size());
        dataSources.forEach((key, value) -> targetDataSourcesMap.put(key, value.getDataSource()));
        super.setDefaultTargetDataSource(dataSource);
        super.setTargetDataSources(targetDataSourcesMap);
        super.afterPropertiesSet();
    }

    /**
     * 获取当前数据源名称
     *
     * @return 数据源名称
     */
    public static String getDataSource() {
        return Optional.ofNullable(LOOKUP_KEY_HOLDER.get().peek()).orElse(DEFAULT_DATASOURCE_NAME);
    }

    /**
     * 设置数据源
     *
     * @param dataSource 数据源名称
     */
    private static void setDataSource(String dataSource) {
        String dataSourceStr = StringUtils.isEmpty(dataSource) ? DynamicDataSource.DEFAULT_DATASOURCE_NAME : dataSource;
        LOOKUP_KEY_HOLDER.get().push(dataSourceStr);
    }

    /**
     * 设置数据源并执行业务
     *
     * @param dataSource 数据源名称
     * @param supplier   业务
     * @return 执行结果
     */
    public static Object withDataSource(String dataSource, Supplier<Object> supplier) {
        setDataSource(dataSource);
        try {
            return supplier.get();
        } finally {
            pollDataSource();
        }
    }

    public static void withDataSource(String dataSource, Runnable runnable) {
        withDataSource(dataSource, () -> {
            runnable.run();
            return null;
        });
    }

    /**
     * 移除数据源
     */
    private static void pollDataSource() {
        Deque<String> deque = LOOKUP_KEY_HOLDER.get();
        deque.poll();
        if (deque.isEmpty()) {
            clearDataSource();
        }
    }

    /**
     * 清空数据源
     */
    public static void clearDataSource() {
        LOOKUP_KEY_HOLDER.remove();
    }

    public static Map<String, AfDataSourceWrapper> getDataSourceMap() {
        return dataSourceInfo;
    }

    public static AfDataSourceWrapper getWrapper() {
        String dataSource = DynamicDataSource.getDataSource();
        AfDataSourceWrapper wrapper = getDataSourceMap().get(dataSource);
        if (wrapper == null) {
            LOGGER.error("不存在的数据源名称：{}", dataSource);
            throw new ServiceException("业务异常", 500);
        }
        return getDataSourceMap().get(DynamicDataSource.getDataSource());
    }

    public static DbType getDbType() {
        return getWrapper().getDbType();
    }

    @Override
    protected Object determineCurrentLookupKey() {
        return getDataSource();
    }
}
