package com.af.v4.system.common.datasource.config;

import com.af.v4.system.common.core.exception.CheckedException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.MyDruidDataSource;
import com.alibaba.druid.pool.DruidAbstractDataSource;
import com.alibaba.druid.pool.DruidDataSource;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Component;

import java.sql.SQLException;
import java.util.*;

/**
 * 动态多数据源配置
 *
 * @author Mr.river
 */
@Configuration
@ConfigurationProperties(prefix = "db")
@Component
public class DynamicDataSourceConfig {

    private final int DEFAULT_MAX_ACTIVE_SIZE = Runtime.getRuntime().availableProcessors() * 2 + 1;
    /**
     * 数据源列表
     */
    private List<MyDruidDataSource> dataSourceList;
    private List<String> subClass;
    private Integer initialSize;
    private Integer minIdle;
    private Integer maxActive;
    private Integer maxWait;
    private Integer connectTimeout;
    private Integer socketTimeout;

    @Bean
    @Primary
    public DynamicDataSource routingDataSource() {
        if (Optional.ofNullable(dataSourceList).isEmpty()) {
            throw new CheckedException("服务启动失败：数据源加载失败，请检查项目是否正确加载了nacos中心对应的业务配置");
        }
        // 循环加入数据源，实例化动态数据源工具类
        Map<Object, Object> targetDataSources = new HashMap<>(dataSourceList.size());
        dataSourceList.forEach(item -> {
            setProperties(item);
            targetDataSources.put(item.getName(), item);
        });
        return new DynamicDataSource(dataSourceList.get(0), targetDataSources);
    }

    private void setProperties(DruidDataSource item) {
        Properties connectProperties = new Properties();
        item.setInitialSize(Objects.requireNonNullElse(initialSize, 5));
        item.setMinIdle(Objects.requireNonNullElse(minIdle, 5));
        item.setMaxActive(Objects.requireNonNullElse(maxActive, DEFAULT_MAX_ACTIVE_SIZE));
        item.setMaxWait(Objects.requireNonNullElse(maxWait, 30000));
        item.setConnectTimeout(Objects.requireNonNullElse(connectTimeout, DruidAbstractDataSource.DEFAULT_TIME_CONNECT_TIMEOUT_MILLIS));
        item.setSocketTimeout(Objects.requireNonNullElse(socketTimeout, 30_000));
        item.setTimeBetweenEvictionRunsMillis(60000);
        item.setMinEvictableIdleTimeMillis(180000);
        item.setTestWhileIdle(true);
        item.setTestOnBorrow(false);
        item.setTestOnReturn(false);
        item.setUseUnfairLock(true);
        String driverClassName = item.getDriverClassName();
        if (driverClassName.contains("SQLServer") || driverClassName.contains("Oracle") || driverClassName.contains("clickhouse")) {
            item.setPoolPreparedStatements(true);
            item.setMaxPoolPreparedStatementPerConnectionSize(20);
            if (driverClassName.contains("Oracle")) {
                // 配置oracle连接超时时间
                connectProperties.setProperty("oracle.net.CONNECT_TIMEOUT", String.valueOf(item.getConnectTimeout()));
                connectProperties.setProperty("oracle.jdbc.ReadTimeout", String.valueOf(item.getSocketTimeout()));
            }
        }
        if (driverClassName.contains("Oracle")) {
            item.setValidationQuery("select 1 from dual");
        } else {
            item.setValidationQuery("select 1");
        }
        try {
            item.setFilters("stat,slf4j");
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
        boolean isMergeSql = !driverClassName.contains("clickhouse");
        connectProperties.setProperty("druid.stat.mergeSql", String.valueOf(isMergeSql));
        connectProperties.setProperty("druid.stat.slowSqlMilli", "5000");
        item.setConnectProperties(connectProperties);
    }

    public List<MyDruidDataSource> getDataSourceList() {
        return dataSourceList;
    }

    public void setDataSourceList(List<MyDruidDataSource> dataSourceList) {
        this.dataSourceList = dataSourceList;
    }

    public Integer getInitialSize() {
        return initialSize;
    }

    public void setInitialSize(Integer initialSize) {
        this.initialSize = initialSize;
    }

    public Integer getMinIdle() {
        return minIdle;
    }

    public void setMinIdle(Integer minIdle) {
        this.minIdle = minIdle;
    }

    public Integer getMaxActive() {
        return maxActive;
    }

    public void setMaxActive(Integer maxActive) {
        this.maxActive = maxActive;
    }

    public Integer getMaxWait() {
        return maxWait;
    }

    public void setMaxWait(Integer maxWait) {
        this.maxWait = maxWait;
    }

    public Integer getConnectTimeout() {
        return connectTimeout;
    }

    public void setConnectTimeout(Integer connectTimeout) {
        this.connectTimeout = connectTimeout;
    }

    public Integer getSocketTimeout() {
        return socketTimeout;
    }

    public void setSocketTimeout(Integer socketTimeout) {
        this.socketTimeout = socketTimeout;
    }

    public List<String> getSubClass() {
        return subClass;
    }

    public void setSubClass(List<String> subClass) {
        this.subClass = subClass;
    }
}
